package Utils::Stream::Serializer::XML;

use base qw(Utils::Stream::Serializer);

use qbit;
use QBit::HTTPAPI::XML;

sub init {
    my ($self) = @_;
    $self->SUPER::init();

    $self->{pretty} = $self->{pretty} ? 1 : 0;
    $self->{indent} = ($self->{indent} // 4) + 0;
    $self->{_numerate_from_} = 0;

    my $cr = $self->{pretty} ? "\n" : '';

    $self->{writer}->write("<?xml version=\"1.0\" encoding=\"utf-8\" ?>$cr<response>") unless $self->{no_headers};

    return;
}

sub _indent {
    my ($self) = @_;
    return '' if not $self->{pretty};
    my $shift_level = $self->{no_headers} ? 0 : 1;
    return "\n" . ' ' x ($self->{indent} * ($self->_level() + $shift_level));
}

sub _write {
    my ($self, $str) = @_;
    $self->{writer}->write($self->_indent() . $str);
    return;
}

sub hash_begin {
    my ($self) = @_;
    $self->_write('<hashref>');
    $self->SUPER::hash_begin();
    return;
}

sub hash_item {
    my ($self, $key, $value) = @_;

    $self->SUPER::hash_item($key, $value);

    $value = $$value if ref($value) eq 'SCALAR';

    my $defined = defined($value) ? '' : ' defined="false"';
    $value //= '';

    $key   = $self->_xml_escape($key);
    $value = $self->_xml_escape($value);

    $self->_write("<item key=\"$key\"$defined>$value</item>");

    return;
}

sub hash_key {
    my ($self, $key) = @_;

    $self->SUPER::hash_key($key);

    $key = $self->_xml_escape($key);

    $self->_write("<item key=\"$key\">");

    return;
}

sub hash_value_begin {
    my ($self) = @_;

    $self->SUPER::hash_value_begin();

    return;
}

sub hash_value_end {
    my ($self) = @_;
    $self->SUPER::hash_value_end();
    $self->_write('</item>');
    return;
}

sub hash_end {
    my ($self) = @_;
    $self->SUPER::hash_end();
    $self->_write('</hashref>');
    return;
}

sub array_begin {
    my ($self) = @_;

    $self->_write('<arrayref>');
    $self->SUPER::array_begin();

    $self->{_context_}{key} = 0;

    return;
}

sub array_item {
    my ($self, $item) = @_;

    $self->SUPER::array_item($item);

    $item = $$item if ref($item) eq 'SCALAR';

    my $defined = defined($item) ? '' : ' defined="false"';
    $item //= '';

    my $key = $self->{_context_}{key}++;
    $item = $self->_xml_escape($item);

    $self->_write("<item key=\"$key\"$defined>$item</item>");

    return;
}

sub array_item_begin {
    my ($self) = @_;

    my $key = $self->{_context_}{key}++;
    $self->_write("<item key=\"$key\">");
    $self->SUPER::array_item_begin();

    return;
}

sub array_item_end {
    my ($self) = @_;

    $self->SUPER::array_item_end();

    $self->_write('</item>');

    return;
}

sub array_end {
    my ($self) = @_;
    $self->SUPER::array_end();
    $self->_write('</arrayref>');
    return;
}

sub scalar {
    my ($self, $scalar) = @_;

    $self->SUPER::scalar($scalar);

    $scalar = $$scalar if ref($scalar) eq 'SCALAR';

    my $defined = defined($scalar) ? '' : ' defined="false"';
    $scalar //= '';

    $scalar = $self->_xml_escape($scalar);

    $self->_write("<scalar$defined>$scalar</scalar>");

    return;
}

sub end {
    my ($self) = @_;
    $self->SUPER::end();
    my $cr = $self->{pretty} ? "\n" : '';
    unless ($self->{no_headers}) {
        $self->{writer}->write("$cr</response>$cr");
    } elsif ($cr) {
        $self->{writer}->write($cr);
    }
    return;
}

sub _xml_escape {
    my ($class, $str) = @_;
    # ============================================================
    # Transforms and filters input characters to acceptable XML characters
    # (or filters them out completely).
    # ------------------------------------------------------------
    local $_ = $str;
    return '' if not defined $_;

    s/&/&amp;/g;
    s/</&lt;/g;
    s/>/&gt;/g;
    s/[\0\ca\cb\cc\cd\ce\cf\cg\ch\ck\cl\cn\co\cp\cq\cr\cs\ct\cu\cv\cw\cx\cy\cz\c[\c\\c]\c^\c_]//g;
    s/'/&apos;/g;
    s/"/&quot;/g;

    return $_;
}

sub _handle_batch {
    my ($self, $data_source, $batch_size) = @_;

    my $bulk = $data_source->get_bulk($batch_size);

    if (@$bulk) {
        my $xml = QBit::HTTPAPI::XML::_pl2xml(
            $bulk,
            $self->_level,
            pretty        => $self->{pretty},
            numerate_from => $self->{_numerate_from_},
        );

        my $indent = '';
        $indent = "\n" . ' ' x ($self->{indent} * $self->_level()) if $self->{pretty};
        my $head = $indent . '<arrayref>';
        my $tail = $indent . '</arrayref>';

        $xml = substr($xml, length($head), -length($tail));

        $self->{writer}->write($xml);
        $self->{_numerate_from_} += $batch_size;
    }

    $self->{_numerate_from_} = 0 if @$bulk < $batch_size;

    return;

}

1;
