package Utils::TSV;

=encoding UTF-8
=cut

=head1 Название

Utils::TSV - Инструменты для работы с tab separated values

=cut

our @ISA    = qw(Exporter);
our @EXPORT = qw(
  parse_tsv
  tsv_with_fields
  tsv_with_fields_and_names
  );

use String::Escape qw(printable unprintable);

=head2 parse_tsv

B<Параметры:> 1) $original_tsv

B<Возвращаемое значение:> 1) $data 2) % с одним из двух возможных ключей:
    headers        поля которые будут использованны на выходе
    string_escape  приводит заэскейпенные символы в оригинальные

Пример:

    На входе:
    "a      b
    1       2
    11      22"

    На выходе:
    [
        [0] {
            a   1,
            b   2
        },
        [1] {
            a   11,
            b   22
        }
    ]

=cut

sub parse_tsv {
    my ($original_tsv, %opts) = @_;

    my @headers = exists($opts{'headers'}) ? @{$opts{'headers'}} : ();
    my $data = [];

    foreach my $line (split /\n/, $original_tsv) {
        chomp($line);
        my @elems = map {$opts{'string_escape'} ? unprintable($_) : $_} split /\t/, $line;

        if (!@headers) {
            @headers = @elems;
            next;
        }

        my %hash;
        @hash{@headers} = @elems;
        push @{$data}, \%hash;

    }

    return $data;
}

=head2 tsv_with_fields

B<Параметры:> 1) $ со структурой 2) $ с полями, которые нужно отобразить в tsv 3) % с одним из двух возможных ключей:
    without_headers  сформированный tsv будет без заголовка
    string_escape    эскейпит значения

B<Возвращаемое значение:> 1) $ с tsv

Пример:

    my $data = [
              {
                'search_id' => '2809',
                'id' => '3629'
              },
              {
                'search_id' => '2838',
                'id' => '3688'
              },
              {
                'search_id' => '6859',
                'id' => '10868'
              }
            ];

    print tsv_with_fields(
        $data,
        [
            'id',
            'search_id',
        ]
    );

Вывод:

    id      search_id
    3629    2809
    3688    2838
    10868   6859

=cut

sub tsv_with_fields {
    my ($data, $fields, %opts) = @_;

    my $return;

    # header
    unless ($opts{'without_headers'}) {
        foreach my $f (@$fields) {
            $return .= $f . "\t";
        }
        chop $return;
        $return .= "\n";
    }

    # content
    foreach my $line (@$data) {
        foreach my $f (@$fields) {
            $return .= ($opts{'string_escape'} ? printable($line->{$f} // '') : $line->{$f} // '') . "\t";
        }
        chop $return;
        $return .= "\n";
    }

    $opts{'end_marker'} //= TRUE;
    $return .= "#END\n" if $opts{'end_marker'};

    return $return;
}

=head2 tsv_with_fields_and_names

B<Параметры:> 1) $ со структурой 2) $ с полями и их именами, которые нужно отобразить в tsv

B<Возвращаемое значение:> 1) $ с tsv

Пример:

    my $data = [
              {
                'search_id' => '2809',
                'id' => '3629'
              },
              {
                'search_id' => '2838',
                'id' => '3688'
              },
              {
                'search_id' => '6859',
                'id' => '10868'
              }
            ];

    print tsv_with_fields_and_names(
        $data,
        [
            {
                field => 'id',
                name => 'Идентификатор площадки',
            },
            {
                field => 'search_id',
                name => 'CLID',
            }
        ]
    );

Вывод:

    Идентификатор площадки  CLID
    3629    2809
    3688    2838
    10868   6859

=cut

sub tsv_with_fields_and_names {
    my ($data, $fields) = @_;

    my $return;

    # header
    foreach my $f (@$fields) {
        $return .= $f->{name} . "\t";
    }
    chop $return;
    $return .= "\n";

    # content
    foreach my $line (@$data) {
        foreach my $f (@$fields) {
            $return .= $line->{$f->{field}} . "\t";
        }
        chop $return;
        $return .= "\n";
    }

    return $return;
}

1;
