package Utils::XLS;

=encoding UTF-8
=cut

=head1 Название

Utils::XLS - Создание xls файлов

=head1 Описание

=cut

use Exporter;    #Для экспортирования переменных
@ISA    = ('Exporter');    #Для экспортирования переменных
@EXPORT = qw(
  data2xls
  data2xls_2
  xls_with_fields
  xls_with_fields_and_names
  xls_from_matrix
  );

use qbit;

use Excel::Writer::XLSX ();
use IO::Scalar;
use Math::Round;
use Spreadsheet::WriteExcel ();

=head1 Методы

=cut

# { data => <массив хешей>, fields => [ { title => '', field => '', style => {} }, ... ] }
sub data2xls {
    my ($dts) = @_;
    my @fields = map {$_->{'field'}} @{$dts->{'fields'}};
    my @titles = map {$_->{'title'}} @{$dts->{'fields'}};
    my $text = "" . join("\t", @titles) . "\r\n";
    for my $d (@{$dts->{'data'}}) {
        $text .= "" . join("\t", @{$d}{@fields}) . "\r\n";
    }
    return $text;
}

# Формат тот же, но отрисовывает уже непосредственно в XLS
sub data2xls_2 {
    my ($dts) = @_;
    my @fields = map {$_->{'field'}} @{$dts->{'fields'}};
    my @titles = map {$_->{'title'}} @{$dts->{'fields'}};

    my $text;
    tie *XLS, 'IO::Scalar', \$text;

    #    my $work = Spreadsheet::WriteExcel->new(\*STDOUT);
    my $work = Spreadsheet::WriteExcel->new(\*XLS);
    my $list = $work->addworksheet(gettext("Common data"));

    #    $list->write(0, 0,  "Hi Excel!");

    $list->write_row('A1', \@titles, $work->addformat(size => 8, bold => 1));

    my $f_bold_right = $work->addformat(left => 1, align => 'right');

    my $row = 1;
    for my $d (@{$dts->{'data'}}) {
        $row++;
        #        $list->write_row('A'.$row, [@{$d}{@fields}], $f_bold_right );
        $list->write_row('A' . $row, \@{$d}{@fields}, $f_bold_right);
    }

    _add_payment_disclaimer($work, $list, $row + 2, scalar(@fields));
    $work->close();

    return $text;
}

=head2 xls_from_matrix

B<Параметры:> 1) $ со структурой

B<Возвращаемое значение:> 1) $ с xls

Пример:

    my $data = [
        [ 'row1 column1', 'row1 column2'],
        [ 'row2 column1', 'row2 column2', 'row2 column3'],
    ];

    open F, ">report.xls";
    binmode(F);
    print F xls_from_matrix($data);
    close F;

=cut

sub xls_from_matrix {
    my ($data, $fields) = @_;

    my $text;
    tie *XLS, 'IO::Scalar', \$text;

    my $workbook  = Spreadsheet::WriteExcel->new(\*XLS);
    my $worksheet = $workbook->add_worksheet();
    my ($col, $row);
    $col = $row = 0;

    my $max_col = 0;
    foreach my $line (@$data) {
        foreach my $element (@$line) {
            $worksheet->write($row, $col, $element);
            $col++;
        }
        $max_col = $col if $col > $max_col;
        $col = 0;
        $row++;
    }

    _add_payment_disclaimer($workbook, $worksheet, $row + 2, $max_col);
    $workbook->close();

    return $text;
}

=head2 xls_with_fields

B<Параметры:> 1) $ со структурой 2) $ с полями, которые нужно отобразить в xls

B<Возвращаемое значение:> 1) $ с xls

Пример:

    my $data = [
              {
                'search_id' => '2809',
                'id' => '3629'
              },
              {
                'search_id' => '2838',
                'id' => '3688'
              },
              {
                'search_id' => '6859',
                'id' => '10868'
              }
            ];

    open F, ">report.xls";
    binmode(F);
    print F xls_with_fields(
        $data,
        [
            'id',
            'search_id',
        ]
    );
    close F;

=cut

sub xls_with_fields {
    my ($data, $fields) = @_;

    my $text;
    tie *XLS, 'IO::Scalar', \$text;

    my $workbook  = Spreadsheet::WriteExcel->new(\*XLS);
    my $worksheet = $workbook->add_worksheet();
    my ($col, $row);
    $col = $row = 0;

    my $header_format = $workbook->add_format();
    $header_format->set_bold();

    # header
    foreach my $f (@$fields) {
        $worksheet->write($row, $col, $f, $header_format);
        $col++;
    }
    $col = 0;
    $row++;

    # content
    foreach my $line (@$data) {
        foreach my $f (@$fields) {
            $worksheet->write($row, $col, $line->{$f});
            $col++;
        }
        $col = 0;
        $row++;
    }

    _add_payment_disclaimer($workbook, $worksheet, $row + 2, scalar(@$fields));
    $workbook->close();

    return $text;
}

=head2 xls_with_fields_and_names

B<Параметры:> 1) $ со структурой 2) $ с полями и их именами, которые нужно отобразить в xls

B<Возвращаемое значение:> 1) $ с xls

Пример:

    my $data = [
              {
                'search_id' => '2809',
                'id' => '3629'
              },
              {
                'search_id' => '2838',
                'id' => '3688'
              },
              {
                'search_id' => '6859',
                'id' => '10868'
              }
            ];

    open F, ">report.xls";
    binmode(F);
    print F xls_with_fields_and_names(
        $data,
        [
            {
                field => 'id',
                name => 'Идентификатор площадки',
                width => '123',  # optional
                type => 'string', # optional
            },
            {
                field => 'search_id',
                name => 'CLID',
            }
        ]
    );
    close F;

=cut

sub xls_with_fields_and_names {
    my ($data, $fields, %opts) = @_;

    throw "'field' should be arrayref" if ref $fields ne 'ARRAY';
    throw "'field' should have values" if @{$fields} == 0;

    my $default = $opts{'default'};

    foreach (@{$fields}) {
        throw "'field' element should be hashref" if ref($_) ne 'HASH';
        throw "'field' element should have 'field'" unless defined $_->{'field'};
        throw "'field' element should have 'name'"  unless defined $_->{'name'};

        $_->{'name'} = _fix_field_name($_->{'name'});
    }

    my $text;
    tie *XLS, 'IO::Scalar', \$text;

    my $workbook;
    if ($opts{format} && $opts{format} eq 'xlsx') {
        $workbook = Excel::Writer::XLSX->new(\*XLS);
    } else {
        $workbook = Spreadsheet::WriteExcel->new(\*XLS);
    }
    my $worksheet = $workbook->add_worksheet();
    my ($col, $row);
    $col = $row = 0;

    my $header_format = $workbook->add_format(bold => 1,);
    $worksheet->freeze_panes(1, 0);

    # header
    foreach my $f (@$fields) {
        my $length = $f->{'width'} || (length($f->{'name'}) + 2) || 10;
        $length = 10 if $length < 10;
        $worksheet->set_column($col, $col, $length);
        $worksheet->write($row, $col, $f->{'name'}, $header_format);
        $col++;
    }
    $col = 0;
    $row++;

    # content
    my %formats;
    foreach my $line (@$data) {
        foreach my $f (@$fields) {
            my $data = $line->{$f->{'field'}} // $default;

            my $format;
            if ($f->{'field'} eq 'date') {
                my $db_format;
                if ($data =~ /^[0-9]{4}\z/) {
                    $db_format = "$data-01-01";
                    $format    = 'YYYY';
                } elsif ($data =~ /^[0-9]{4}-[0-9]{2}\z/) {
                    $db_format = "$data-01";
                    $format    = 'MMMM, YYYY';
                } elsif ($data =~ /^[0-9]{4}-W[0-9]{2}\z/) {
                    $db_format = trdate('week', 'db', $data);
                    $format = 'DD.MM.YYYY';
                } else {
                    $db_format = $data;
                    $format    = 'DD.MM.YYYY';
                }

                $data = dates_delta_days('1899-12-30', $db_format, iformat => 'db');
            } elsif ($f->{'type'} && $f->{'type'} eq 'percent') {
                $format = '0%';
                $data = $data eq '-' ? '-' : round($data) / 100;
            }
            if ($format) {
                unless ($formats{$format}) {
                    my $wbf = $workbook->add_format();
                    $wbf->set_num_format($format);
                    $formats{$format} = $wbf;
                }
                $format = $formats{$format};
            }
            $worksheet->write($row, $col, $data, $format);
            $col++;
        }
        $col = 0;
        $row++;
    }

    _add_payment_disclaimer($workbook, $worksheet, $row + 2, scalar(@$fields));
    $workbook->close();

    return $text;
}

=head2 _fix_field_name

B<Параметры:> 1) $field_name

B<Возвращаемое значение:> 1) $fix_field_name

Пример:

    my $data = 'Вознаграждение (с&nbsp;НДС)';

    $data = _fix_field_name($data);

    print $data; # 'Вознаграждение (с НДС)'

=cut

sub _fix_field_name {
    my ($field_name) = @_;

    $field_name =~ s/&nbsp;/ /g;

    return $field_name;
}

sub _add_payment_disclaimer {
    my ($workbook, $worksheet, $row, $cols) = @_;
    $worksheet->set_row($row, 40);
    my $disclaimer = gettext('payment_disclaimer');
    $disclaimer =~ s/\.\s+/.\n/g;
    chomp $disclaimer;
    my $format = $workbook->add_format(text_wrap => 1,);
    $worksheet->merge_range($row, 0, $row, $cols - 1, $disclaimer, $format);
}

1;
