package WebInterface::Controller::DSP;

use qbit;

use base qw(WebInterface::Controller);

use Exception::Denied;
use Exception::Form;
use Exception::Validation::BadArguments;

__PACKAGE__->model_accessors(
    dsp                     => 'Application::Model::DSP',
    dsp_documentation       => 'Application::Model::DSP::Documentation',
    dsp_testing             => 'Application::Model::DSP::Testing',
    partner_db              => 'Application::Model::PartnerDB',
    statistics_dsp_response => 'Application::Model::DSP::StatisticsResponse',
    users                   => 'Application::Model::Users',
    rbac                    => 'Application::Model::RBAC',
);

__PACKAGE__->register_rights(
    [
        {
            name        => 'dsp',
            description => d_gettext('Rights to use dsp'),
            rights      => {
                dsp_statistics_responses  => d_gettext('Right to view DSP statistics responses'),
                dsp_view_field__yandexuid => d_gettext('Right to view field "yandexuid"'),
            }
        }
    ]
);

sub add : FORMCMD {
    my ($self) = @_;

    throw "/dsp/add is deprecated";
}

sub action : CMD : SAFE {
    my ($self) = @_;

    my %ignored_params = map {$_ => 1} qw(id type ajax);

    my $error;
    try {
        $self->dsp->do_action(
            $self->request->param('id'),
            $self->request->param('type'),
            map {$_ => $self->request->param($_)} grep {!exists($ignored_params{$_})} $self->request->param_names()
        );
    }
    catch Exception::Validation::BadArguments with {
        $error = shift->message;
    };

    if ($self->request->param('ajax')) {
        return $error ? $self->as_json({error => $error}) : $self->as_json(
            $self->dsp->get(
                $self->request->param('id'),
                fields => [grep {!exists($ignored_params{$_})} $self->request->param_names()]
            )
        );
    } else {
        return $error ? $self->error($error) : $self->redirect('list');
    }
}

sub approve : FORMCMD {
    my ($self) = @_;

    my $dsp_id = $self->request->param('id');

    my $dsp = $self->dsp->get($dsp_id, fields => [qw(short_caption type_captions url test_url)]);

    return (
        title => gettext('Approve DSP'),

        check_rights => ['do_dsp_action_approve'],

        fields => [
            {
                name  => 'id',
                type  => 'hidden',
                value => $dsp_id
            },
            {
                name     => 'short_caption',
                type     => 'input',
                value    => $dsp->{'short_caption'},
                label    => gettext('Caption'),
                required => TRUE,
                length   => 32,
                trim     => TRUE,
                check    => sub {
                    my ($form, $value) = @_;

                    my $dsp =
                      $self->dsp->get_all(fields => ['id', 'short_caption'], filter => {short_caption => $value});

                    throw Exception::Form gettext('DSP with name "%s" already exists', $dsp->[0]{'short_caption'})
                      if @$dsp > 1 || (@$dsp && $dsp->[0]{'id'} != $dsp_id);
                  }
            },
            {
                type  => 'text',
                label => gettext('DSP type'),
                value => join(', ', @{$dsp->{'type_captions'}}),
            },
            {
                name     => 'tag',
                type     => 'input',
                label    => gettext('Tag for BS'),
                required => TRUE,
                length   => 50,
                check    => sub {
                    my ($form, $value) = @_;

                    throw Exception::Form gettext(
                        'A tag must contain latin lowercase letters, no spaces and length from 5 to 50 characters')
                      unless $value =~ m#^[a-z][a-z0-9_]{4,49}$#;

                  }
            },
            {
                name     => 'url',
                type     => 'input',
                value    => $dsp->{'url'},
                label    => gettext('URL'),
                required => TRUE,
                length   => 1024,
                hint     => gettext('The HTTP address which the RTB host will use to send RTB request.'),
                check    => sub {
                    my ($self, $value) = @_;

                    throw Exception::Form gettext('URL must begin with "http://"')
                      unless $value =~ m#^http://#;
                  }
            },
            {
                name     => 'test_url',
                type     => 'input',
                value    => $dsp->{'test_url'},
                label    => gettext('Test server URL'),
                required => FALSE,
                length   => 1024,
                hint     => gettext('The HTTP address which the RTB host will use to send test RTB request.'),
                check    => sub {
                    my ($self, $value) = @_;

                    if (length($value) > 0) {
                        throw Exception::Form gettext('Test URL must begin with "http://"')
                          unless $value =~ m#^http://#;
                    }
                  }
            },
            {label => gettext('Approve'), type => 'button', subtype => 'submit'}
        ],

        save => sub {
            my ($form) = @_;

            $self->dsp->do_action($form->get_value('id'),
                'approve', map {$_ => $form->get_value($_)} qw(short_caption url test_url tag));

            $form->{'redirect_opts'} = {new_url => '/v2/dsp/connections?search={"id":["=","' . $dsp_id . '"]}'};
        },

        ajax_action => 'redirect'
    );
}

sub document_add : FORMCMD {
    my ($self) = @_;

    throw "/dsp/document_add is deprecated";
}

sub document_delete : CMD : SAFE {
    my ($self) = @_;

    my $id_document = $self->request->param('id');

    $self->dsp_documentation->do_action($id_document, 'delete');

    return $self->redirect('documentation');
}

sub document_edit : FORMCMD {
    my ($self) = @_;

    my $id_document = $self->request->param('id');

    my $file = $self->dsp_documentation->get($id_document, fields => [qw(name description language)]);

    return (
        title => gettext('Dsp document editing'),

        check_rights => ['do_dsp_documentation_edit'],

        fields => [
            {
                name  => 'id',
                type  => 'hidden',
                value => $id_document
            },
            {
                name     => 'name',
                label    => gettext('Name document'),
                type     => 'input',
                required => 1,
                length   => 255,
                value    => $file->{name}
            },
            {
                name     => 'language',
                label    => gettext('Document language'),
                type     => 'select',
                value    => $file->{language},
                required => 1,
                items    => [
                    {
                        label => 'Select language',
                        value => ''
                    },
                    {
                        label => gettext('Russian'),
                        value => 'ru'
                    },
                    {
                        label => gettext('English'),
                        value => 'en'
                    }
                ]
            },
            {
                name     => 'description',
                label    => gettext('Description'),
                type     => 'textarea',
                rows     => 7,
                required => 1,
                value    => $file->{description}
            },
            {label => gettext('Save'), type => 'button', subtype => 'submit'},
        ],

        save => sub {
            my ($form) = @_;

            $self->dsp_documentation->do_action($form->get_value('id'),
                'edit', map {$_ => $form->get_value($_)} qw(name language description));
        },

        pre_process => sub {
            my ($form) = @_;

            unless (exists($file->{'name'})) {
                $self->response->status(404);
                return $form->break();
            }
        },

        redirect => 'documentation'
    );
}

sub documentation : CMD {
    my ($self) = @_;

    return $self->redirect2url('https://tech.yandex.ru/rtb/doc/api/index-docpage');
}

sub get_data_for_dsp_errors_widget : CMD : SAFE {
    throw '/dsp/get_data_for_dsp_errors_widget is deprecated';
}

sub get_dsp_testing : CMD : SAFE {
    my ($self) = @_;

    my %params =
      map {$_ => $self->request->param($_)}
      qw(
      dsp_id
      tag
      device_userdata
      device_ip
      device_ua
      device_yandexuid
      moderation
      imp_id
      site_id
      site_referer
      site_rereferer
      );

    return $self->as_json({error => {form => gettext("Expected dsp_id")}}) unless $params{'dsp_id'};

    my $tmp_rights = $self->app->add_tmp_rights("dsp_view_field__multistate");

    my $dsp = $self->dsp->get($params{'dsp_id'}, fields => ['multistate']);

    undef($tmp_rights);

    return $self->as_json({error => {form => gettext('DSP with ID %s not found', $params{'dsp_id'})}}) unless $dsp;

    throw Exception::Denied unless $self->dsp->check_multistate_action($dsp->{'multistate'}, 'test');

    my @empty_params =
      grep {!$params{$_}} qw(tag device_ip device_ua imp_id site_id site_referer);

    my $error;

    foreach (@empty_params) {
        my $field = $_;
        $field =~ s/_/\./;
        $error->{'error'}{'fields'}{$_} = gettext('Field "%s" must be filled', $field);
    }

    return $self->as_json($error) if $error;

    $params{'device_yandexuid'} = $self->request->cookie('yandexuid')
      unless $self->check_rights('dsp_view_field__yandexuid') && $params{'device_yandexuid'};

    my $result = $self->dsp_testing->get_dsp_testing_results(%params);

    return $self->as_json($result);
}

sub get_file : CMD {
    my ($self) = @_;

    return $self->redirect2url('https://tech.yandex.ru/rtb/doc/api/index-docpage');
}

sub get_statistics_responses : CMD : SAFE {
    my ($self) = @_;

    throw Exception::Denied unless $self->check_rights('dsp_statistics_responses');

    my $dsp_id = $self->request->param('dsp_id');

    return $self->as_json({error => gettext('Select DSP')}) unless $dsp_id;

    my $dsp = $self->dsp->get($dsp_id, fields => ['id', 'multistate', 'short_caption'],);

    if (ref($dsp) eq 'HASH' and $dsp->{'id'} == $dsp_id) {
        my $period = $self->request->param('period__name') || '';
        $period = [$self->request->param('period__from'), $self->request->param('period__to')] if $period eq 'other';
        if ($period && !(ref($period) eq 'ARRAY')) {
            $period = [name2dates($period, '', '', iformat => 'db', oformat => 'db')];
        }

        return $self->as_json({error => gettext('Invalid period')})
          unless $period
              && ref($period) eq 'ARRAY'
              && check_date($period->[0], iformat => 'db')
              && check_date($period->[1], iformat => 'db')
              && $period->[0] le $period->[1];

        my $statistics = $self->statistics_dsp_response->get_statistics(
            fields => [qw(dt campaign_id error_type errors_sum)],
            filter => ['AND', [['dsp_id', '=', \$dsp_id], ['dt', '>=', \$period->[0]], ['dt', '<=', \$period->[1]]]],
            order_by => ['dt', 'campaign_id'],
        );

        return $self->as_json({data => $statistics});
    } else {
        return $self->as_json({error => gettext('DSP ID "%s" does not exists', $dsp_id)});
    }
}

sub list : CMD : DEFAULT {
    my ($self) = @_;

    return $self->redirect2url('/v2/dsp/connections/');
}

sub statistics_responses : CMD {
    my ($self) = @_;

    throw Exception::Denied unless $self->check_rights('dsp_statistics_responses');

    my $dsp_id = $self->request->param('dsp_id');

    my $dsp = $self->dsp->get_all(
        fields => ['id', 'multistate', 'multistate_name', 'short_caption'],
        filter => {multistate => 'created_in_bk and not deleted'},
    );

    return $self->not_found() unless @$dsp;

    my %hash_dsp = map {$_->{'id'} => $_} @$dsp;

    return $self->error(gettext('DSP ID "%s" does not exists', $dsp_id))
      if $dsp_id && !$hash_dsp{$dsp_id};

    my @error_values;
    my $count = 0;
    foreach ($self->dsp_testing->get_errors()) {
        push(@error_values, {id => $count, label => $_}) if $_;
        $count++;
    }

    my $dimensions = {
        dt          => {type => 'date',   label => gettext('Date')},
        campaign_id => {type => 'number', label => gettext('Site ID'), noFormat => TRUE},
        error_type => {
            type   => 'dictionary',
            label  => gettext('Error type'),
            values => \@error_values,
        }
    };

    return $self->from_bem_template(
        'dsp/statistics_responses.bem.tt2',
        vars => {
            metadata => {
                dim_order => ['dt', 'campaign_id', 'error_type', 'errors_sum'],
                dim_names => {
                    dt          => gettext('Date'),
                    campaign_id => gettext('Site ID'),
                    error_type  => gettext('Error type'),
                    errors_sum  => gettext('Count')
                },
                metric     => 'errors_sum',
                calendar   => 'dt',
                dimensions => $dimensions
            },
            dsp_id   => $dsp_id,
            hash_dsp => \%hash_dsp,
        }
    );
}

sub test : CMD {
    my ($self) = @_;

    throw Exception::Denied unless $self->check_rights('do_dsp_action_test');

    my $dsp_id = $self->request->param('id');

    my $dsp_list = $self->dsp->get_all_dsp_for_testing(with_disabled => TRUE);

    return $self->not_found() unless @$dsp_list;

    my %hash_dsp_list = map {$_->{'id'} => $_} @$dsp_list;

    return $self->error(gettext('DSP ID "%s" does not exists', $dsp_id))
      if $dsp_id && !$hash_dsp_list{$dsp_id};

    $dsp_id = undef if $dsp_id && $hash_dsp_list{$dsp_id}->{'disabled'};

    my $test_preset_params = [
        {
            label => '-',
            value => ''
        }
    ];

    my $count = 0;
    foreach my $type (qw(240×400 160×600 300×250 728×90)) {
        push(
            @$test_preset_params,
            {
                label => "partner2.yandex.ru — $type, " . gettext('CPM limit: 1000 RUB'),
                value => '118547-' . ++$count
            }
        );
        push(
            @$test_preset_params,
            {
                label => "partner2.yandex.ru — $type, " . gettext('without CPM limit'),
                value => '118547-' . ++$count
            }
        );
    }

    return $self->from_bem_template(
        'dsp/test.bem.tt2',
        vars => {
            # data from db
            dsp_id       => $dsp_id,
            hash_dsp     => \%hash_dsp_list,
            all_disabled => (@$dsp_list == grep {$_->{'disabled'}} @$dsp_list) ? TRUE : FALSE,

            #data from user
            device_ip        => $self->request->remote_addr(),
            device_ua        => $self->request->http_header('User-Agent'),
            device_yandexuid => $self->request->cookie('yandexuid'),

            test_preset_params => $test_preset_params,
        }
    );
}

TRUE;
