package WebInterface::Controller::Devel;

use qbit;

use base qw(WebInterface::Controller);

use RoleRights;
use QBit::Validator;

use Exception::Denied;
use Exception::Form;
use Exception::TextTemplate;
use Exception::Validation::BadArguments;
use Exception::Validator::Fields;
use Exception::Validator::Errors;

use PiConstants qw(
  $RUB_CURRENCY_ID
  $SIMPLE_NOTIFICATION_TYPES
  $NOTIFICATION_TYPE
  $NOTIFICATION_VIEW_TYPE
  $NOTIFICATION_ICON
  %MODERATION_REASON_TIMEOUTS
  );

__PACKAGE__->model_accessors(
    agreement_checker      => 'Application::Model::AgreementChecker',
    all_pages              => 'Application::Model::AllPages',
    api_balance            => 'QBit::Application::Model::API::Yandex::Balance',
    api_geobase            => 'QBit::Application::Model::API::Yandex::HTTPapi_geobase',
    bk_edit_page           => 'Application::Model::Log::BkEditPage',
    clickhouse_db          => 'Application::Model::ClickhouseDB',
    custom_bk_options      => 'Application::Model::CustomBKOptions',
    devel_bk_pi_comparison => 'Application::Model::BkPiComparison',
    intapi_acl             => 'Application::Model::IntAPI_ACL',
    kv_store               => 'QBit::Application::Model::KvStore',
    lang_detect            => 'QBit::Application::Model::YandexLangDetect',
    mailer                 => 'Application::Model::SendMail',
    moderation_reason      => 'Application::Model::ModerationReason',
    notification           => 'Application::Model::Notification',
    page_options           => 'Application::Model::PageOptions',
    partner_db             => 'Application::Model::PartnerDB',
    partner_logs_db        => 'Application::Model::PartnerLogsDB',
    queue                  => 'Application::Model::Queue',
    queue                  => 'Application::Model::Queue',
    rbac                   => 'Application::Model::RBAC',
    resources              => 'Application::Model::Resources',
    simple_notification    => 'Application::Model::SimpleNotification',
    statistics             => 'Application::Model::Statistics',
    text_template          => 'Application::Model::TextTemplate',
    users                  => 'Application::Model::Users',
);

__PACKAGE__->register_rights(
    [
        {
            name        => 'devel',
            description => sub {gettext('Right for the developer interface')},
            rights      => {
                devel_create_client                 => d_gettext('Right to create client_id in balance'),
                devel_drop_user                     => d_gettext('Right to drop test user'),
                remove_user_client_association      => d_gettext('Right to remove client_id from user'),
                devel_die                           => d_gettext('Right to perform a test die'),
                run_cron_job                        => d_gettext('Right to run cron jobs'),
                view_cron_job                       => d_gettext('Right to view cron jobs'),
                view_check_fields                   => d_gettext('Right to use fields check tool'),
                create_or_update_place_switch       => d_gettext('Right to use create_or_update_place switch'),
                devel_update_options                => d_gettext('Right to use update options'),
                devel_insert_additional_income_stat => d_gettext('Right to add statistics'),
            },
        }
    ]
);

# проверяет только живость приложения, для использования из /v2/alive
sub alive : CMD : WOAUTH {
    my ($self) = @_;

    return $self->response->data(1);
}

# проверяет его работоспособность
sub readiness_probe : CMD : WOAUTH {
    my ($self) = @_;

    my ($alive, $msg) = $self->app->check_alive->is_alive(raw => TRUE);

    return $self->response->data($alive || $msg);
}

sub bk_pi_comparison : CMD {
    my ($self) = @_;

    return $self->denied() unless $self->check_rights('devel_bk_pi_comparison');

    my %opts = (
        state                => $self->request->param('state'),
        domain               => $self->request->param('domain'),
        opt_readonly         => $self->request->param('opt_readonly'),
        opt_dontshowsex      => $self->request->param('opt_dontshowsex'),
        opt_dontshowbehavior => $self->request->param('opt_dontshowbehavior'),
        mirrors              => $self->request->param('mirrors'),
        excludeddomains      => $self->request->param('excludeddomains'),
    );

    my $context_diff = $self->devel_bk_pi_comparison->get_diff(
        type => 'context',
        %opts,
    );

    my $search_diff = $self->devel_bk_pi_comparison->get_diff(
        type => 'search',
        %opts,
    );

    return $self->from_bem_template(
        'devel/bk_pi_comparison.bem.tt2',
        vars => {
            context => $context_diff,
            search  => $search_diff,
        },
    );
}

sub bk_send_log : CMD {
    my ($self) = @_;

    return $self->denied() unless $self->check_rights('bk_edit_page_view_bk_send_log');

    my $search_params = $self->request->param_array('search_json');

    my $vo = $self->get_vo(
        model       => $self->bk_edit_page,
        sort        => 'dt',
        sortdesc    => TRUE,
        show_search => $self->check_rights('view_search'),
        !@$search_params
        ? (limit => 3)
        : ()
    );

    my %model_opts = $vo->get_model_opts();

    my $data = [];
    $data = $self->bk_edit_page->get_all(
        fields => [qw(id dt page_id login  request  response  error)],
        %model_opts,
    ) if $vo->show_search_data();

    my $set_count = $self->bk_edit_page->found_rows();

    my @table;
    my ($request, $response, $page_id);

    foreach my $el (@{$data}) {
        push @table,
          {
            request  => to_json(from_json($el->{'request'}), pretty => 1),
            response => $el->{'error'}
            ? $el->{'response'}
            : to_json(from_json($el->{'response'}), pretty => 1),
            map {$_ => $el->{$_}} qw( id  dt  page_id  login  error )
          };
    }

    return $self->from_bem_template(
        'devel/bk_send_log.bem.tt2',
        vars => {
            table     => \@table,
            set_count => $set_count,
            $vo->get_template_vars()
        }
    );
}

=head2 check_timeout

<Lhttps://wiki.yandex-team.ru/partner/w/partner2-check-timeout/>

=cut

sub check_timeout : CMD : WOAUTH {
    my ($self) = @_;

    if ($self->rbac->has_role_from_group_dev()) {

        sleep $self->request->param('seconds');

        return $self->response->data('ok');

    } else {
        return $self->denied();
    }

}

sub create_client : FORMCMD {
    my ($self) = @_;

    return (

        check_rights => ['devel_create_client'],

        description =>
          gettext('After pressing the submit button the new client_id will be created in balance and you will get it.'),

        fields => [
            {
                type    => 'button',
                subtype => 'submit',
                label   => gettext('Create client_id'),
            },
        ],

        save => sub {
            my ($form) = @_;

            my $user = $self->get_option('cur_user', {});
            my $client_id = $self->api_balance->create_client(operator_uid => $user->{id},);

            $form->{complete_message} = gettext("New client_id: '%s'", $client_id);
        },

    );
}

sub insert_additional_income_stat : FORMCMD {
    my ($self) = @_;

    return (

        check_rights => ['devel_insert_additional_income_stat'],

        description => gettext('After pressing the submit button statistics will be uploaded.'),

        fields => [
            {
                type  => 'textarea',
                name  => 'json_stat',
                check => sub {
                    my ($form, $value) = @_;

                    my $stat;
                    try {
                        $stat = from_json($value);
                    }
                    catch {
                        throw Exception::Form gettext('Incorrect json');
                    };

                    my $qv = QBit::Validator->new(
                        data     => $stat,
                        template => {
                            type     => 'array',
                            size_min => 1,
                            all      => {
                                type   => 'hash',
                                fields => {
                                    date      => {type => 'date', format => 'db'},
                                    client_id => {type => 'int_un'},
                                    money     => {
                                        check => sub {
                                            throw Exception::Validator::Fields gettext('Expected number')
                                              unless looks_like_number($_[1]);
                                          }
                                    },
                                },
                            },
                            check => sub {
                                my ($qv, $stat) = @_;

                                my $client_ids = {};
                                foreach my $row (@$stat) {
                                    throw Exception::Validator::Fields gettext('Duplicate rows: %s_%s', $row->{'date'},
                                        $row->{'client_id'})
                                      if $client_ids->{$row->{'date'}}{$row->{'client_id'}};

                                    $client_ids->{$row->{'date'}}{$row->{'client_id'}} = TRUE;
                                }

                                my $uniq_client_ids = array_uniq(map {keys(%$_)} values(%$client_ids));

                                my $users = $qv->app->users->get_all(
                                    fields => [qw(client_id)],
                                    filter => {client_id => $uniq_client_ids}
                                );

                                if (@$users != @$uniq_client_ids) {
                                    throw Exception::Validator::Fields gettext(
                                        'Users not found: %s',
                                        join(', ',
                                            @{arrays_difference($uniq_client_ids, [map {$_->{'client_id'}} @$users])})
                                    );
                                }
                              }
                        },
                        app => $self
                    );

                    if ($qv->has_errors) {
                        throw Exception::Form join(
                            '; ',
                            map {
                                my $field = $qv->_get_key($_->{'path'});
                                my $error = ($field ? "$field: " : '') . join(', ', @{$_->{'msgs'}});
                              } $qv->get_fields_with_error()
                        );
                    }
                },
            },
            {
                type    => 'button',
                subtype => 'submit',
                label   => gettext('Upload'),
            },
        ],

        save => sub {
            my ($form) = @_;

            my $json = $form->get_value('json_stat');
            my $stat = from_json($json);

            my %users = map {$_->{'client_id'} => $_->{'id'}} @{
                $self->users->get_all(
                    fields => [qw(id client_id)],
                    filter => {client_id => array_uniq(map {$_->{'client_id'}} @$stat)}
                )
              };

            my $filter = $self->clickhouse_db->filter();
            foreach (@$stat) {
                $_->{'dt'}             = delete($_->{'date'});
                $_->{'partner_wo_nds'} = $self->statistics->_get_int(delete($_->{'money'}));
                $_->{'user_id'}        = $users{delete($_->{'client_id'})};
                $_->{'currency_id'}    = $RUB_CURRENCY_ID;

                $filter->or({dt => $_->{'dt'}, user_id => $_->{'user_id'}, currency_id => $RUB_CURRENCY_ID});
            }

            $self->clickhouse_db->init_db([qw(statistics_additional_income_temporary)]);

            my @pk = qw(dt user_id currency_id);
            my %pk = map {$_ => ''} @pk;
            my @metrics =
              grep {!exists($pk{$_})} $self->clickhouse_db->statistics_additional_income_temporary->field_names;

            my $query = $self->clickhouse_db->query->select(
                table  => $self->clickhouse_db->statistics_additional_income,
                fields => {%pk, map {$_ => ['*' => [\-1, {SUM => [$_]}]]} @metrics},
                filter => $filter
            );

            $query->group_by(@pk);

            $self->clickhouse_db->statistics_additional_income_temporary->add($query);

            $self->clickhouse_db->statistics_additional_income_temporary->add_multi($stat);

            $query = $self->clickhouse_db->query->select(
                table  => $self->clickhouse_db->statistics_additional_income_temporary,
                fields => {%pk, map {$_ => {SUM => [$_]}} @metrics},
            );

            $query->group_by(@pk);

            $self->clickhouse_db->statistics_additional_income->add($query);

            $self->clickhouse_db->statistics_additional_income_temporary->drop();

            $form->{'complete_message'} = gettext("Statistics uploaded");
        },
    );
}

sub create_or_update_place_switch : FORMCMD {
    my ($self) = @_;

    throw Exception::Denied unless $self->resources->get_get_resources()->{'devel_create_or_update_place_switch'};

    my $key = 'use_create_or_update_place';

    my $is_checked = $self->kv_store->get($key);

    return (
        check_rights => ['create_or_update_place_switch'],
        fields       => [
            {
                name    => $key,
                type    => 'checkbox',
                checked => $is_checked,
                label   => gettext('Send data to Balance.CreateOrUpdatePlace'),
            },
            {
                type    => 'button',
                subtype => 'submit',
                label   => gettext('Save'),
            },
        ],

        save => sub {
            my ($form) = @_;

            my $value = $form->get_value($key) ? 1 : 0;

            $self->kv_store->set($key, $value);

            $form->{'complete_message'} = gettext('Saved');
        },

    );

}

sub cron_jobs : CMD {
    my ($self) = @_;

    throw Exception::Denied unless $self->check_rights('view_cron_job');
    throw gettext('Cannot import cron settings: %s', $@) unless (eval {require Cron});

    my $cron    = Cron->new();
    my $methods = $cron->get_cron_methods();

    my $list = {};
    foreach my $path (keys %$methods) {
        foreach my $method (keys %{$methods->{$path}}) {
            my $job = $methods->{$path}{$method};
            $list->{$job->{package}} = [] unless defined $list->{$job->{package}};
            push @{$list->{$job->{package}}},
              {
                path   => $path,
                method => $method,
                attrs  => $job->{'attrs'},
                time   => $job->{'time'},
              };
        }
    }
    return $self->from_bem_template(
        'devel/cron_methods.bem.tt2',
        vars => {
            joblist   => $list,
            allow_run => $self->get_option('allow_cron_jobs_testing') && $self->check_rights('run_cron_job'),
        }
    );
}

sub die : CMD {
    my ($self) = @_;

    return $self->denied() unless $self->check_rights('devel_die');

    throw 'Test die';
}

sub drop_user : FORMCMD {
    my ($self) = @_;

    throw Exception::Denied unless $self->resources->get_get_resources()->{'devel_drop_user'};

    return (
        title  => gettext('Drop user'),
        fields => [
            {
                name     => 'login',
                label    => gettext('Login'),
                type     => 'input',
                required => TRUE,
                trim     => TRUE,
                suggest  => {cmd => 'login_sgst', path => 'settings', opts => {partners => 1}},
            },
            {
                name  => 'create_persons',
                type  => 'checkbox',
                label => gettext('Create test persons in balance'),
            },
            {label => gettext('Drop'), type => 'button', subtype => 'submit'},
        ],
        save => sub {
            my ($form) = @_;

            my $user = $self->users->get_by_login($self->request->param('login'), fields => [qw(id client_id)]);
            throw Exception::Form gettext('No such user') unless $user;
            my $uid = $user->{'id'};
            my $cur_user = $self->get_option('cur_user', {});

            $form->{'complete_message'} = gettext('User %s successfully removed', $self->request->param('login'));

            try {
                $self->api_balance->remove_user_client_association($cur_user->{'id'}, $user->{'client_id'}, $uid);
            }
            catch {
                # throw Exception::Form 'Remove association before drop user: ' . $_[0]->message();
            };

            try {
                $self->partner_db->user_role->delete($self->partner_db->filter({user_id => $uid}));
                $self->partner_db->invites->delete($self->partner_db->filter({partner_id => $uid}));
                $self->users->partner_db_action_log_table()->delete($self->partner_db->filter({user_id => $uid}));
                $self->users->partner_db_action_log_table()->delete($self->partner_db->filter({elem_id => $uid}));
                $self->partner_db->users->delete($uid);
            }
            catch {
                throw Exception::Form 'Drop user: ' . $_[0]->message();
            };

            return unless ($form->get_value('create_persons'));

            my $client_id = $self->api_balance->create_client(NAME => 'Client', operator_uid => $cur_user->{'id'},);

            my $hardcoded_data = [
                # yt
                {
                    client_id => 2132961,
                    person_id => 824994,
                },
                # yt
                {
                    client_id => 1188592,
                    person_id => 777387,
                },
                # yt
                {
                    client_id => 1271705,
                    person_id => 777364,
                },
                # yt
                {
                    client_id => 3107443,
                    person_id => 1177959,
                },
                # yt
                {
                    client_id => 1292640,
                    person_id => 777366,
                },
                # ur
                {
                    client_id => 2365491,
                    person_id => 918391,
                },
                # ur
                {
                    client_id => 1035333,
                    person_id => 605029,
                },
                # ur
                {
                    client_id => 946176,
                    person_id => 743870,
                },
                # ur
                {
                    client_id => 946176,
                    person_id => 681875,
                },
                # sw_yt
                {
                    client_id => 3029423,
                    person_id => 1154766,
                },
                # sw_yt
                {
                    client_id => 2360577,
                    person_id => 933847,
                },
                # sw_yt
                {
                    client_id => 2706507,
                    person_id => 1068747,
                },
                # sw_yt
                {
                    client_id => 439028,
                    person_id => 1138565,
                },
                # ph Гилязова
                {
                    client_id => 1442157,
                    person_id => 559460,
                },
                # ph Метелица
                {
                    client_id => 3979897,
                    person_id => 1498861,
                },
                # ph Бурибаев (счет в Я.Деньги)
                {
                    client_id => 2323511,
                    person_id => 908552,
                },
            ];

            my @persons = map {
                my $person = $_;
                grep {$_->{'id'} == $person->{'person_id'}}
                  @{$self->api_balance->get_client_persons($person->{'client_id'}, 1)}
            } @$hardcoded_data;

            throw Exception::Form 'No persons in Balance' unless (@persons);

            my @keys_to_change_underscore = qw(
              passport_code
              passport_d
              passport_e
              passport_n
              passport_s
              yamoney_wallet
              );

            try {
                for my $person (@persons) {
                    delete($person->{$_}) foreach (qw(id client_id));
                    foreach (keys %$person) {delete $person->{$_} if (length($person->{$_}) == 0)}

                    foreach my $key (keys %{$person}) {
                        if (in_array($key, \@keys_to_change_underscore)) {
                            my $new_key = $key;
                            $new_key =~ s/_/-/g;
                            $person->{$new_key} = $person->{$key};
                            delete $person->{$key};
                        }
                    }

                    @$person{qw(operator_uid client_id)} = ($cur_user->{'id'}, $client_id);
                    $person->{'person_id'} = SOAP::Data->type(int => -1);
                    $person->{'is-partner'} = 1;
                    $self->api_balance->create_person(%$person);
                }
            }
            catch {
                throw Exception::Form "Create person " . $_[0]->message();
            };

            try {
                $self->api_balance->create_user_client_association(
                    operator_uid => $cur_user->{'id'},
                    client_id    => $client_id,
                    user_id      => $uid
                );
            }
            catch {
                throw Exception::Form 'Create user association: ' . $_[0]->message();
            };
        }
    );
}

sub field : FORMCMD {
    my ($self) = @_;

    my $data = {};
    eval {$data = from_json($self->request->cookie('devel_field_value'));};
    my $value = $data->{data};

    $self->response->delete_cookie('devel_field_value');

    my $field = $self->request->param('field') // $self->request->cookie('devel_field');

    my @additional_fields = ();

    if ($field eq 'ph_payments') {
        push
          @additional_fields,
          {
            label => gettext('Last name'),
            name  => 'lname',
            type  => 'input',
          },
          {
            label => gettext('First name'),
            name  => 'fname',
            type  => 'input',
          },
          {
            label => gettext('Middle name'),
            name  => 'mname',
            type  => 'input',
          },
          {
            label => gettext('Passport serial'),
            name  => 'passport-s',
            type  => 'input',
          },
          {
            label => gettext('Passport number'),
            name  => 'passport-n',
            type  => 'input',
          },
          ;
    }

    throw Exception::Form gettext('Field required') if not defined $field;

    return (

        check_rights => ['view_check_fields'],

        fields => [
            {
                label => 'field',
                name  => 'field',
                type  => 'hidden',
                value => $field,
            },
            @additional_fields,
            {
                label    => $field,
                name     => $field,
                type     => $field,
                value    => $value,
                required => TRUE,
            },
            {
                type    => 'button',
                subtype => 'submit',
                label   => gettext('Submit'),
            },
        ],

        save => sub {
            my ($form) = @_;

            $self->response->add_cookie(devel_field => $self->request->param('field'));
            $self->response->add_cookie(devel_field_value => to_json({data => $form->get_value($field),}));

            $form->{complete_message} = gettext("Field has correct value");
        },

    );

}

sub fields : CMD {
    my ($self) = @_;

    return $self->denied() unless $self->check_rights('view_check_fields');

    my ($path) = __FILE__ =~ m{(.*/)Devel\.pm$};

    my @fields_list = split /\n/, `ls -1 $path../../QBit/WebInterface/Controller/BEMHTML/Form/Field/`;

    map {$_ =~ s/\.pm$//} @fields_list;

    return $self->from_bem_template('devel/fields.bem.tt2', vars => {fields_list => \@fields_list,});
}

sub get_attach : CMD {
    my ($self) = @_;

    throw Exception::Denied unless $self->check_rights('mailer_view');

    my $attachment = $self->partner_logs_db->query->select(
        table  => $self->partner_logs_db->email_attachment,
        fields => [qw(filename data)],
        filter => ['id' => '=' => \$self->request->param('id')],
    )->get_all()->[0];

    return $self->response->status(404) unless $attachment;

    return $self->send_file(
        data     => $attachment->{'data'},
        filename => $attachment->{'filename'},
    );
}

# Пример страницы для создания client side реализации поля 'iban_or_account'.
#
# TODO удалить FORMCMD iban_or_account после того как client side будет готов
#
sub iban_or_account : FORMCMD {
    my ($self) = @_;

    my @form = (

        fields => [
            {
                name => 'iban_or_account',
                type => 'iban_or_account',
            },
            {
                type    => 'button',
                subtype => 'submit',
                label   => gettext('Next'),
            },
        ],

        save => sub {
            my ($form) = @_;

            my $ioa = $form->get_value('iban_or_account');

            $form->{'complete_message'} =
              sprintf("Saved. IBAN: '%s'. Account: '%s'.", ($ioa->{'iban'} // ''), ($ioa->{'account'} // ''),);
        },

    );

    return @form;
}

sub intapi_acl_list : CMD {
    my ($self) = @_;

    return $self->denied() unless $self->check_rights('intapi_acl_view');

    return $self->from_bem_template(
        'devel/intapi_acl.bem.tt2',
        vars => {
            methods  => $self->intapi_acl->get_all(with_acl => TRUE),
            location => $self->intapi_acl->get_location(),
        }
    );
}

sub intapi_acl_set : CMD {
    my ($self) = @_;

    return $self->denied() unless $self->check_rights('intapi_acl_edit');

    my $result;
    try {
        $self->intapi_acl->set_acl(
            $self->request->param('path'),
            $self->request->param('method'),
            $self->request->param('acl')
        );
        $result = {acl => $self->request->param('acl')};
    }
    catch {
        $result = {error => shift->message()};
    };

    return $self->as_json($result);
}

sub intapi_acl_subnets : CMD {
    my ($self) = @_;

    return $self->denied() unless $self->check_rights('intapi_acl_view');

    return $self->as_text(
        join(
            "\n",
            $self->intapi_acl->acl2subnets(
                $self->intapi_acl->get_acl($self->request->param('path'), $self->request->param('method'))
            )
        )
    );
}

sub remove_user_client_association : FORMCMD {
    my ($self) = @_;

    return (

        check_rights => ['remove_user_client_association'],

        description => gettext(
            'After filling the form the association between selected client_id and the selected uid will be removed.'),

        complete_message => gettext('Association has been successfully removed.'),

        fields => [
            {
                name     => 'uid',
                label    => gettext('uid'),
                type     => 'input',
                required => TRUE,
            },
            {
                name     => 'client_id',
                label    => gettext('client_id'),
                type     => 'input',
                required => TRUE,
            },
            {
                type    => 'button',
                subtype => 'submit',
                label   => gettext('Remove association'),
            },
        ],

        save => sub {
            my ($form) = @_;

            my $user = $self->get_option('cur_user', {});

            my $operator_uid = $user->{id};
            my $user_uid     = $self->request->param('uid');
            my $client_id    = $self->request->param('client_id');

            try {
                $self->api_balance->remove_user_client_association($operator_uid, $client_id, $user_uid);
            }
            catch {
                my $error_message = $_[0]->message();
                $error_message = "<pre>$error_message</pre>";
                throw Exception::Form $error_message;
            }

        },

    );

}

sub run_cron_job : CMD : SAFE {
    my ($self) = @_;

    return $self->denied()
      unless $self->get_option('allow_cron_jobs_testing') && $self->check_rights('run_cron_job');

    throw gettext('Cannot import cron settings: %s', $@) unless (eval {require Cron});

    my $path   = $self->request->param('path');
    my $method = $self->request->param('method');
    return $self->denied(gettext('Bad arguments')) unless ($path && $method);

    my $cron    = Cron->new();
    my $methods = $cron->get_cron_methods();
    my $ok;
    if (exists($methods->{$path})) {
        if (exists($methods->{$path}{$method})) {
            $ok = 1;
        } elsif ($method =~ /^(.+)_(\d+)$/) {
            my $name     = $1;
            my $instance = $2;
            $ok = 1
              if ( exists($methods->{$path}{$name})
                && defined($methods->{$path}{$name}{attrs})
                && defined($methods->{$path}{$name}{attrs}{instances})
                && $methods->{$path}{$name}{attrs}{instances} >= $instance
                and $instance > 0);
        }
    }
    return $self->denied(gettext('Bad arguments'))
      unless $ok;

    if (fork) {
        return $self->redirect2url_internal($self->request->param('retpath'));
    } else {
        Cron->new()->do($path, $method);
        exit();
    }
}

sub show_menu : CMD {
    my ($self) = @_;

    if (!in_array($self->get_option('stage', 'unknown'), [qw(dev test)])) {
        return $self->denied();
    }

    $self->app->_menu();

    my $menu = $self->get_option('menu');

    return $self->as_json($menu);
}

sub simple_notification_list : CMD {
    my ($self) = @_;

    throw Exception::Denied unless $self->check_rights('simple_notification_view');

    my $vo = $self->get_vo(
        model       => $self->simple_notification,
        sort        => 'id',
        sortdesc    => FALSE,
        show_search => $self->check_rights('view_search'),
    );

    my $notifications;
    my $count;

    if ($vo->show_search_data()) {
        $notifications = $self->simple_notification->get_all(
            all_locales => TRUE,
            fields      => [qw(id ntype message multistate_name actions editable_fields)],
            $vo->get_model_opts()
        );
        $count = $self->simple_notification->found_rows();
    }
    return $self->from_bem_template(
        'devel/simple_notification.bem.tt2',
        vars => {
            notifications => $notifications,
            count         => $count,
            $vo->get_template_vars(),
        }
    );
}

sub simple_notification_delete : CMD {
    my ($self) = @_;

    throw Exception::Denied unless $self->check_rights('do_simple_notification_action_delete');

    my $notification_id = $self->request->param('id');

    my $message = $self->simple_notification->get($notification_id, fields => ['id']);
    throw Exception::Validation::BadArguments gettext('Unknown simple_notification "%s"', $notification_id)
      unless defined $message;

    try {
        $self->simple_notification->do_action($notification_id, 'delete',);
    }
    catch Exception::Validation::BadArguments with {
        throw Exception::Form $@->message;
    };

    return $self->redirect('simple_notification_list');
}

sub simple_notification_add : FORMCMD {
    my ($self) = @_;

    my @locales = sort(keys(%{$self->get_option('locales')}));
    my @roles   = @{$self->rbac->get_roles()};

    return (
        check_rights => ['do_simple_notification_action_add'],
        fields       => [
            {
                name     => "ntype",
                type     => 'select',
                label    => gettext('Notification type'),
                items    => [map {+{value => $_, label => $_,}} @$SIMPLE_NOTIFICATION_TYPES],
                required => TRUE,
            },
            (
                map {{name => "message_$_", type => 'input', label => gettext('Message (%s)', $_), required => TRUE,},}
                  @locales
            ),
            {
                name     => "icon",
                type     => 'input',
                label    => gettext('Icon'),
                required => FALSE,
            },
            {
                name     => "link",
                type     => 'input',
                label    => gettext('URL'),
                required => FALSE,
            },
            {
                name     => "short_name",
                type     => 'input',
                label    => gettext('Short name'),
                required => FALSE,
            },
            (
                map {{name => "title_$_", type => 'input', label => gettext('Title (%s)', $_), required => FALSE,},}
                  @locales
            ),
            (
                map {
                    {
                        name     => "button_text_$_",
                        type     => 'input',
                        label    => gettext('Button text (%s)', $_),
                        required => FALSE,
                    },
                  } @locales
            ),
            (
                map {
                    {
                        name  => "role_$_->{id}",
                        type  => 'checkbox',
                        label => sprintf('%s — %s', $_->{id}, $_->{name}),
                    }
                  } sort {
                    $a->{name} cmp $b->{name}
                  } @roles
            ),
            {
                type    => 'button',
                subtype => 'submit',
                label   => gettext('Set notification'),
            },
        ],

        save => sub {
            my ($form) = @_;

            my $ntype       = $form->get_value("ntype");
            my $message     = {map {$_ => $form->get_value("message_$_")} @locales};
            my $icon        = $form->get_value("icon");
            my $link        = $form->get_value("link");
            my $short_name  = $form->get_value("short_name");
            my $title       = {map {$_ => $form->get_value("title_$_")} @locales};
            my $button_text = {map {$_ => $form->get_value("button_text_$_")} @locales};

            my @selected_role_ids =
              map  {$_->{id}}
              grep {$form->get_value("role_$_->{id}")} @roles;

            my $id;
            try {
                $id = $self->simple_notification->add(
                    message     => $message,
                    ntype       => $ntype,
                    roles       => \@selected_role_ids,
                    icon        => $icon,
                    link        => $link,
                    short_name  => $short_name,
                    title       => $title,
                    button_text => $button_text,
                );
            }
            catch Exception::Validator::Fields with {
                throw Exception::Form shift->message;
            }
            catch Exception::Validation::BadArguments with {
                throw Exception::Form shift->message;
            };
            $form->{'redirect_opts'}{'search_json'} = to_json({id => $id});
        },
        redirect => 'simple_notification_list',
    );
}

sub simple_notification_edit : FORMCMD {
    my ($self) = @_;

    my $notification_id = $self->request->param('id');

    my $message = $self->simple_notification->get(
        $notification_id,
        fields      => [qw(id ntype message roles icon link short_name title button_text)],
        all_locales => TRUE
    ) // throw Exception::Validation::BadArguments gettext('Unknown simple_notification "%s"', $notification_id);

    my @locales           = sort(keys(%{$self->get_option('locales')}));
    my @roles             = @{$self->rbac->get_roles()};
    my %selected_role_ids = map {$_ => TRUE} @{$message->{roles}};

    return (
        check_rights => ['do_simple_notification_action_edit'],
        fields       => [
            {
                name     => 'id',
                type     => 'hidden',
                required => TRUE,
                value    => $notification_id,
            },
            {
                name     => "ntype",
                type     => 'select',
                label    => gettext('Notification type'),
                items    => [map {+{value => $_, label => $_,}} @$SIMPLE_NOTIFICATION_TYPES],
                required => TRUE,
                value    => $message->{ntype},
            },
            (
                map {
                    {
                        name     => "message_$_",
                        type     => 'input',
                        label    => gettext('Message (%s)', $_),
                        required => TRUE,
                        value    => $message->{"message_$_"},
                    },
                  } @locales
            ),
            {
                name     => 'icon',
                type     => 'input',
                label    => gettext('Icon'),
                required => FALSE,
                value    => $message->{icon},
            },
            {
                name     => 'link',
                type     => 'input',
                label    => gettext('URL'),
                required => FALSE,
                value    => $message->{link},
            },
            {
                name     => "short_name",
                type     => 'input',
                label    => gettext('Short name'),
                required => FALSE,
                value    => $message->{short_name},
            },
            (
                map {
                    {
                        name     => "title_$_",
                        type     => 'input',
                        label    => gettext('Title (%s)', $_),
                        required => FALSE,
                        value    => $message->{"title_$_"},
                    },
                  } @locales
            ),
            (
                map {
                    {
                        name     => "button_text_$_",
                        type     => 'input',
                        label    => gettext('Button text (%s)', $_),
                        required => FALSE,
                        value    => $message->{"button_text_$_"},
                    },
                  } @locales
            ),
            (
                map {
                    {
                        name    => "role_$_->{id}",
                        type    => 'checkbox',
                        label   => sprintf('%s — %s', $_->{id}, $_->{name}),
                        checked => $selected_role_ids{$_->{id}},
                    },
                  } sort {
                    $a->{name} cmp $b->{name}
                  } @roles
            ),
            {
                type    => 'button',
                subtype => 'submit',
                label   => gettext('Set notification'),
            },
        ],

        save => sub {
            my ($form) = @_;

            my $notification_id = $form->get_value('id');
            my $ntype           = $form->get_value("ntype");
            my $message         = {map {$_ => $form->get_value("message_$_")} @locales};
            my $icon            = $form->get_value("icon");
            my $link            = $form->get_value("link");
            my $short_name      = $form->get_value("short_name");
            my $title           = {map {$_ => $form->get_value("title_$_")} @locales};
            my $button_text     = {map {$_ => $form->get_value("button_text_$_")} @locales};

            my @selected_role_ids =
              map  {$_->{id}}
              grep {$form->get_value("role_$_->{id}")} @roles;

            try {
                $self->simple_notification->do_action(
                    $notification_id, 'edit',
                    message     => $message,
                    ntype       => $ntype,
                    roles       => \@selected_role_ids,
                    icon        => $icon,
                    link        => $link,
                    short_name  => $short_name,
                    title       => $title,
                    button_text => $button_text,
                );
            }
            catch Exception::Validator::Fields with {
                throw Exception::Form shift->message;
            }
            catch Exception::Validation::BadArguments with {
                throw Exception::Form shift->message;
            };
        },
        redirect => 'simple_notification_list'
    );
}

sub text_template_add : FORMCMD {
    my ($self) = @_;

    my @locales = sort keys(%{$self->get_option('locales')});

    return (
        title => gettext('Add text template'),

        check_rights => ['do_text_template_add'],

        fields => [
            {
                name     => 'caption',
                type     => 'input',
                label    => gettext('Caption'),
                length   => 255,
                required => TRUE,
            },
            (
                map {
                    {
                        name     => "content_$_",
                        type     => 'textarea',
                        label    => gettext('Content (%s)', $_),
                        length   => 5000,
                        required => TRUE,
                        check    => sub {
                            my ($form, $value) = @_;

                            try {
                                $self->text_template->process_template($value);
                            }
                            catch Exception::TextTemplate with {
                                throw Exception::Form $@->message;
                            }
                        },
                    },
                  } @locales
            ),
            {label => gettext('Add'), type => 'button', subtype => 'submit'}
        ],

        save => sub {
            my ($form) = @_;

            my %content = map {$_ => $form->get_value("content_$_")} @locales;

            my $id;
            try {
                $id = $self->text_template->add(
                    content => \%content,
                    map {$_ => $form->get_value($_)} qw(caption)
                );
            }
            catch Exception::Validation::BadArguments with {
                throw Exception::Form $@->message;
            };

            $form->{'redirect_opts'}{'search_json'} = to_json({id => $id});
        },

        redirect => 'text_template_list'
    );
}

sub text_template_edit : FORMCMD {
    my ($self) = @_;

    my $text_template = $self->text_template->get(
        $self->request->param('id'),
        fields      => [qw(id caption content editable_fields)],
        all_locales => TRUE
    ) // throw Exception::Validation::BadArguments gettext('Unknown text template');

    %$text_template = hash_transform($text_template, [qw(id), keys(%{$text_template->{'editable_fields'}})]);

    throw Exception::Denied unless keys(%$text_template) > 1;

    my @locales = sort keys(%{$self->get_option('locales')});

    return (
        title => gettext('Edit text template'),

        check_rights => ['do_text_template_edit'],

        fields => [
            {
                name     => 'id',
                type     => 'hidden',
                required => TRUE,
                value    => $text_template->{'id'},
            },
            (
                exists($text_template->{'caption'})
                ? (
                    {
                        name     => 'caption',
                        type     => 'input',
                        label    => gettext('Description'),
                        length   => 255,
                        required => TRUE,
                        value    => $text_template->{'caption'},
                    },
                  )
                : ()
            ),
            (
                exists($text_template->{'content'}) ? (
                    map {
                        {
                            name     => "content_$_",
                            type     => 'textarea',
                            label    => gettext('Content (%s)', $_),
                            length   => 5000,
                            required => TRUE,
                            check    => sub {
                                my ($form, $value) = @_;

                                try {
                                    $self->text_template->process_template($value);
                                }
                                catch Exception::TextTemplate with {
                                    throw Exception::Form $@->message;
                                }
                            },
                            value => $text_template->{'content'}{$_},
                        },
                      } @locales
                  )
                : ()
            ),
            {label => gettext('Change'), type => 'button', subtype => 'submit'}
        ],

        save => sub {
            my ($form) = @_;

            try {
                my %data = map {$_ => $form->get_value($_)} grep {exists($text_template->{$_})} qw(caption);
                $data{'content'} = {map {$_ => $form->get_value("content_$_")} @locales}
                  if exists($text_template->{'content'});

                $self->text_template->do_action($text_template->{'id'}, 'edit', %data);
            }
            catch Exception::Validation::BadArguments with {
                throw Exception::Form $@->message;
            };
        },

        redirect => 'text_template'
    );
}

sub text_template_list : CMD {
    my ($self) = @_;

    throw Exception::Denied unless $self->check_rights('text_template_view');

    my $vo = $self->get_vo(
        model       => $self->text_template,
        sort        => 'id',
        sortdesc    => FALSE,
        show_search => $self->check_rights('view_search'),
    );

    my $text_templates;
    my $count;

    if ($vo->show_search_data()) {
        $text_templates = $self->text_template->get_all(
            fields => [qw(id caption content multistate_name actions editable_fields)],
            $vo->get_model_opts()
        );

        $count = $self->text_template->found_rows();
    }

    return $self->from_bem_template(
        'devel/text_template.bem.tt2',
        vars => {
            text_templates => $text_templates,
            count          => $count,
            $vo->get_template_vars(),
        }
    );
}

sub update_options : WIZARDCMD {
    my ($self) = @_;

    return (
        caption      => gettext('Page options'),
        check_rights => ['devel_update_options'],
        steps        => [
            {
                title  => gettext('Enter page'),
                fields => [
                    {
                        name     => 'page_id',
                        type     => 'input',
                        value    => $self->request->param('page_id') // '',
                        required => 1,
                        check    => sub {
                            my ($wizard, $value) = @_;

                            throw Exception::Form gettext('Unknown page_id')
                              unless $self->all_pages->get_all(fields => ['page_id'], filter => {page_id => $value})
                                  ->[0];
                        },
                    },
                    {
                        label   => gettext('Next'),
                        type    => 'button',
                        subtype => 'submit',
                    },
                ],
            },
            {
                title  => gettext('Edit options'),
                fields => sub {
                    my ($wizard) = @_;

                    my $page_id = $wizard->get_value('page_id');

                    my $options = $self->page_options->get_options($page_id);

                    return (
                        {
                            name  => 'enable',
                            label => gettext('Enable'),
                            type  => 'textarea',
                            value => $wizard->get_value('enable') // join("\n", @{$options->{'Enable'}}),
                        },
                        {
                            name  => 'disable',
                            label => gettext('Disable'),
                            type  => 'textarea',
                            value => $wizard->get_value('disable') // join("\n", @{$options->{'Disable'}}),
                        },
                        {
                            label   => gettext('Save'),
                            type    => 'button',
                            subtype => 'submit',
                        },
                    );
                  }
            }
        ],
        save => sub {
            my ($wizard) = @_;

            my $page_id = $wizard->get_value('page_id');

            try {
                $self->page_options->update(
                    $page_id,
                    enable  => [split(/\s*,\s*|\s+/, $wizard->get_value('enable'))],
                    disable => [split(/\s*,\s*|\s+/, $wizard->get_value('disable'))]
                );

                $self->queue->add(
                    method_name => 'update_in_bk',
                    params      => {page_ids => [$page_id],},
                );
            }
            catch {
                throw Exception::Form shift->message();
            };
        },

        redirect => 'update_options'
    );
}

sub view_mail : CMD {
    my ($self) = @_;

    throw Exception::Denied unless $self->check_rights('mailer_view');

    my $vo = $self->get_vo(
        model       => $self->mailer,
        sort        => 'id',
        sortdesc    => 1,
        show_search => $self->check_rights('view_search'),
    );

    my $mails;
    my $mails_count;
    my $count;

    if ($vo->show_search_data()) {
        $mails = $self->mailer->get_all(
            fields => [
                qw(id
                  dt
                  subject
                  last_send_successful
                  from
                  to
                  cc
                  bcc
                  email_data
                  attachments
                  )
            ],
            $vo->get_model_opts()
        );

        $mails_count = $self->mailer->found_rows();

        $count = $self->mailer->get_fields_cnt(
            fields => ['id'],
            filter => {$vo->get_model_opts()}->{'filter'},
        );
    }

    return $self->from_bem_template(
        'devel/mail.bem.tt2',
        vars => {
            mails       => $mails,
            mails_count => $mails_count,
            count_id    => $count->{'cnt_id'},
            $vo->get_template_vars(),
        }
    );
}

sub test_yamoney : FORMCMD {
    my ($self) = @_;

    throw Exception::Denied unless $self->check_rights('devel_test_yamoney');

    my @form = (
        check_rights => ['devel_test_yamoney'],
        description  => gettext('Check work with Yandex.Money'),

        fields => [
            {
                label => gettext('Yandex.Money account number'),
                name  => 'account',
                type  => 'input',
            },
            {
                label => gettext('Last name'),
                name  => 'lname',
                type  => 'input',
            },
            {
                label => gettext('First name'),
                name  => 'fname',
                type  => 'input',
            },
            {
                label => gettext('Middle name'),
                name  => 'mname',
                type  => 'input',
            },
            {
                label => gettext('Passport serial'),
                name  => 'passport-s',
                type  => 'input',
            },
            {
                label => gettext('Passport number'),
                name  => 'passport-n',
                type  => 'input',
            },
            {
                type    => 'button',
                subtype => 'submit',
                label   => gettext('Check'),
            },
        ],
        save => sub {
            my ($form) = @_;
            if (
                $self->app->api_yamoney->is_account_identificated(
                    account    => $form->get_value('account'),
                    lastname   => $form->get_value('lname'),
                    firstname  => $form->get_value('fname'),
                    middlename => $form->get_value('mname'),
                    passport   => $form->get_value('passport-s') . ' ' . $form->get_value('passport-n'),
                )
               )
            {
                $form->{'complete_message'} = gettext("Correct data");
            } else {
                my $hash = $self->app->api_yamoney->_iddoc_hash(
                    $form->get_value('lname'),
                    $form->get_value('fname'),
                    $form->get_value('mname'),
                    $form->get_value('passport-s') . ' ' . $form->get_value('passport-n')
                );
                $form->{'complete_message'} =
                  gettext("Incorrect Yandex.Money account number") . " iddoc_hash = \"$hash\"";
            }
        },
    );
    return @form;
}

sub _remove_fields_from_search_json {
    my ($self, $search_json, @fields) = @_;

    my %fields = map {$_ => 1} @fields;
    my $search = from_json($search_json);

    my (@removed, @left);

    foreach my $expr (@{$search->[1]}) {
        if ($fields{$expr->[0]}) {
            push(@removed, $expr);
        } else {
            push(@left, $expr);
        }
    }

    if (@left) {
        $search->[1] = \@left;
    } else {
        $search = '';
    }

    return (\@removed, ($search ? to_json($search) : ''));
}

sub notification_list : CMD {
    my ($self) = @_;

    throw Exception::Denied unless $self->check_rights('notification_view');

    my $vo = $self->get_vo(
        model            => $self->notification,
        sort             => 'id',
        sortdesc         => TRUE,
        show_search      => $self->check_rights('view_search'),
        permanent_filter => ['type' => '=' => 'custom'],
    );

    my $notifications;
    my $count;

    if ($vo->show_search_data()) {
        $notifications = $self->notification->get_all(
            all_locales => TRUE,
            fields      => [
                qw(
                  id
                  create_date
                  type
                  view_type
                  ttl
                  caption
                  icon_id
                  multistate_name
                  actions
                  short_name
                  )
            ],
            $vo->get_model_opts(),
        );
        $count = $self->notification->found_rows();
    }

    return $self->from_bem_template(
        'devel/notification.bem.tt2',
        vars => {
            notifications => $notifications,
            count         => $count,
            $vo->get_template_vars(),
        }
    );
}

sub notification_edit : FORMCMD {
    my ($self) = @_;

    my $notification_id = $self->request->param('id');

    my $message;
    if ($notification_id) {
        $message = $self->notification->get(
            $notification_id,
            fields => [
                qw(
                  id
                  create_date
                  type
                  view_type
                  ttl
                  caption
                  icon_id
                  message
                  button_caption
                  url
                  expectation_caption
                  expected_growth
                  roles
                  logins
                  short_name
                  )
            ],
            all_locales => TRUE
        );
        throw Exception::Validation::BadArguments gettext('Unknown notification "%s"', $notification_id)
          unless $message;
        throw Exception::Validation::BadArguments gettext('Cannot edit notification "%s"', $notification_id)
          if $message->{type} eq 'auto';
    }

    my @locales = sort keys %{$self->get_option('locales')};
    my @LOCAL_NOTIFICATION_TYPE = grep {$_ ne 'auto'} @$NOTIFICATION_TYPE;

    my @fields = (
        (
            $notification_id
            ? {
                name     => 'id',
                type     => 'hidden',
                required => TRUE,
                value    => $notification_id,
              }
            : ()
        ),
        {
            name     => "type",
            type     => 'select',
            label    => gettext('Notification type'),
            items    => [map +{value => $_, label => $_}, @LOCAL_NOTIFICATION_TYPE],
            required => TRUE,
            ($notification_id ? (disabled => 'disabled', value => $message->{type}) : ()),
        },
        {
            name     => "view_type",
            type     => 'select',
            label    => gettext('Notification view_type'),
            items    => [map +{value => $_, label => $_}, @$NOTIFICATION_VIEW_TYPE],
            required => TRUE,
            ($notification_id ? (value => $message->{view_type}) : ()),
        },
        {
            name     => "short_name",
            type     => 'input',
            label    => gettext('Short Name'),
            required => TRUE,
            trim     => 1,
            ($notification_id ? (value => $message->{short_name}) : ()),
        },
        (
            map +{
                name     => "caption_$_",
                type     => 'input',
                label    => gettext('Caption (%s)', $_),
                required => TRUE,
                trim     => 1,
                ($notification_id ? (value => $message->{"caption_$_"}) : ())
            },
            @locales
        ),
        {
            name     => "icon_id",
            type     => 'select',
            label    => gettext('Icon'),
            items    => [map +{value => $_, label => $_}, @$NOTIFICATION_ICON],
            required => TRUE,
            ($notification_id ? (value => $message->{icon_id}) : ()),
        },
        (
            map +{
                name     => "message_$_",
                type     => 'input',
                label    => gettext('Message (%s)', $_),
                required => TRUE,
                trim     => 1,
                ($notification_id ? (value => $message->{"message_$_"}) : ())
            },
            @locales
        ),
        {
            name     => "ttl",
            type     => 'input',
            label    => gettext('TTL'),
            required => TRUE,
            trim     => 1,
            ($notification_id ? (value => $message->{ttl}) : ()),
        },
        {
            name     => "url",
            type     => 'input',
            label    => gettext('URL'),
            required => FALSE,
            trim     => 1,
            ($notification_id ? (value => $message->{url}) : ()),
        },
        (
            map +{
                name     => "button_caption_$_",
                type     => 'input',
                label    => gettext('Button Caption (%s)', $_),
                required => FALSE,
                trim     => 1,
                ($notification_id ? (value => $message->{"button_caption_$_"}) : ())
            },
            @locales
        ),
        (
            map +{
                name     => "expectation_caption_$_",
                type     => 'input',
                label    => gettext('Expectation Caption (%s)', $_),
                required => FALSE,
                trim     => 1,
                ($notification_id ? (value => $message->{"expectation_caption_$_"}) : ())
            },
            @locales
        ),
        {
            name     => "expected_growth",
            type     => 'input',
            label    => gettext('expected_growth'),
            required => FALSE,
            trim     => 1,
            ($notification_id ? (value => $message->{expected_growth}) : ()),
        },
        {
            name => "roles",
            type => 'multi_checkbox',
            ($notification_id ? (value => $message->{roles}) : ()),
            items => [
                map +{
                    label => sprintf('%s — %s', $_->{id}, $_->{name}),
                    value => $_->{'id'}
                },
                sort {$a->{'id'} <=> $b->{'id'}} @{$self->rbac->get_roles()}
            ],
        },
        {
            name     => "logins",
            type     => 'textarea',
            label    => gettext('logins'),
            required => FALSE,
            trim     => 1,
            ($notification_id ? (value => join(', ', @{$message->{logins} // []})) : ()),
        },
        {
            type    => 'button',
            subtype => 'submit',
            label   => ($notification_id ? gettext('Save notification') : gettext('Add notification')),
        },
    );

    return (
        check_rights => ($notification_id ? ['do_notification_action_edit'] : ['do_notification_action_add']),
        fields       => \@fields,
        save         => sub {
            my ($form) = @_;

            my @locales = sort keys %{$self->get_option('locales')};

            my $type = $form->get_value("type");
            if ($type eq 'auto') {
                throw Exception::Form 'Cannot create notification with type=auto';
            }

            my $message             = {map {$_ => $form->get_value("message_$_")} @locales};
            my $caption             = {map {$_ => $form->get_value("caption_$_")} @locales};
            my $expectation_caption = {map {$_ => $form->get_value("expectation_caption_$_")} @locales};
            my $button_caption      = {map {$_ => $form->get_value("button_caption_$_")} @locales};
            my $expected_growth = $form->get_value("expected_growth");
            my $url             = $form->get_value("url");

            my %params = (
                message    => $message,
                caption    => $caption,
                view_type  => $form->get_value("view_type"),
                ttl        => $form->get_value("ttl"),
                short_name => $form->get_value("short_name"),
                icon_id    => $form->get_value("icon_id"),
            );

            if (grep {$_ ne ''} values %$button_caption) {
                $params{button_caption} = $button_caption;
            }

            if (grep {$_ ne ''} values %$expectation_caption) {
                $params{expectation_caption} = $expectation_caption;
            }

            if ($url ne '') {
                $params{url} = $url;
            }

            if ($expected_growth ne '') {
                $params{expected_growth} = $expected_growth;
            }

            $params{roles} = $form->get_value("roles");
            $params{logins} = [map {s/^\s+//; s/\s+$//; $_} split /,/, $form->get_value("logins")];

            my $id;
            try {
                if (my $notification_id = $form->get_value('id')) {
                    $self->notification->do_action($notification_id, 'edit', %params);
                } else {
                    $id = $self->notification->add(type => $type, %params);
                }
            }
            catch Exception::Validator::Fields with {
                throw Exception::Form shift->message;
            }
            catch Exception::Validation::BadArguments with {
                throw Exception::Form shift->message;
            };
        },
        redirect => 'notification_list'
    );
}

sub notification_do_action {
    my ($self, $action) = @_;

    throw Exception::Denied unless $self->check_rights('do_notification_action_' . $action);

    my $notification_id = $self->request->param('id');

    my $message = $self->notification->get($notification_id, fields => ['id']);
    throw Exception::Validation::BadArguments gettext('Unknown notification "%s"', $notification_id)
      unless defined $message;

    try {
        $self->notification->do_action($notification_id, $action);
    }
    catch Exception::Validation::BadArguments with {
        throw Exception::Form $@->message;
    };

    return $self->redirect('notification_list');
}

sub notification_start : CMD {
    my ($self) = @_;
    return $self->notification_do_action('start');
}

sub notification_delete : CMD {
    my ($self) = @_;
    return $self->notification_do_action('delete');
}

sub notification_restore : CMD {
    my ($self) = @_;
    return $self->notification_do_action('restore');
}

sub notification_from_file : FORMCMD {
    my ($self) = @_;

    my @fields = (
        {
            name     => "note_file",
            type     => 'attach',
            label    => gettext('File'),
            required => TRUE,
        },
        {
            type    => 'button',
            subtype => 'submit',
            label   => gettext('Load notifications'),
        },
    );

    return (
        enctype      => 'multipart/form-data',
        check_rights => ['do_notification_action_add'],
        fields       => \@fields,
        save         => sub {
            my ($form) = @_;

            my $file = $form->get_value("note_file");

            my $id;
            my @content = split /\r?\n/, $file->{content} // '';
            throw Exception::Form 'No data' unless @content > 1;
            my %fields = (
                caption             => undef,
                message             => undef,
                icon_id             => 'default',
                logins              => [],
                ttl                 => undef,
                expected_growth     => 0,
                button_caption      => '',
                expectation_caption => '',
                url                 => '',
                view_type           => 'default',
                short_name          => undef,
            );
            my $header = shift @content;
            $header =~ s/^\W+//;
            $header =~ s/\s+$//;
            $header = [split /\t/, $header];
            my %header_map;

            for my $i (0 .. $#$header) {
                my $f = $header->[$i];
                next unless exists $fields{$f};
                $header_map{$f} = $i;
            }

            my %defaults = (type => 'custom');
            while (my ($k, $v) = each %fields) {
                unless (exists $header_map{$k}) {
                    if (defined $v) {
                        $defaults{$k} = $v;
                    } else {
                        throw Exception::Form "absents obligatory field '$k'";
                    }
                }
            }
            my @map_fields = keys %header_map;
            my @map_values = values %header_map;

            my @data;
            my @to_lang  = qw(message caption button_caption expectation_caption);
            my @to_array = qw(logins);
            for my $row (@content) {
                $row =~ s/\s+$//;
                next unless $row;
                $row = fix_utf($row);
                my @line = split /\t/, $row;
                my %row;
                @row{@map_fields} = @line[@map_values];

                for my $f (keys %row) {
                    delete $row{$f} unless defined $row{$f};
                }

                for my $f (@to_lang) {
                    if (defined(my $str = $row{$f})) {
                        $row{$f} = {
                            ru => $str,
                            en => $str,
                        };
                    }
                }
                for my $f (@to_array) {
                    if (defined $row{$f}) {
                        $row{$f} = [split /\s*,\s*/, $row{$f}];
                    }
                }
                push @data, {%defaults, %row};
            }

            my $row_count = 0;
            $self->partner_db->transaction(
                sub {
                    my @errors;
                    for my $row (@data) {
                        $row_count++;
                        try {
                            $self->notification->add(%$row);
                        }
                        catch {
                            my $e = shift;
                            push @errors, sprintf "%d: %s\n", $row_count, $e->message;
                        };
                    }
                    if (@errors) {
                        throw Exception::Form \@errors;
                    }
                }
            );
        },
        redirect => 'notification_list'
    );
}

sub moderation_reason_list : CMD {
    my ($self) = @_;

    throw Exception::Denied unless $self->check_rights('moderation_reason_tool');

    my $vo = $self->get_vo(
        model       => $self->moderation_reason,
        sort        => 'id',
        sortdesc    => TRUE,
        show_search => $self->check_rights('view_search'),
    );

    my $reasons;
    my $count;

    if ($vo->show_search_data()) {
        $reasons = $self->moderation_reason->get_all(
            all_locales => TRUE,
            fields      => [
                qw(
                  id
                  create_date
                  owner
                  manager_txt
                  partner_txt
                  timeout_txt
                  multistate_name
                  actions
                  )
            ],
            $vo->get_model_opts(),
        );
        $count = $self->moderation_reason->found_rows();
    }

    return $self->from_bem_template(
        'devel/moderation_reason.bem.tt2',
        vars => {
            reasons => $reasons,
            count   => $count,
            $vo->get_template_vars(),
        }
    );
}

sub moderation_reason_edit : FORMCMD {
    my ($self) = @_;

    my $reason_id = $self->request->param('id');

    my $message;
    my $is_exists;
    if ($reason_id) {
        $message = $self->moderation_reason->get(
            $reason_id,
            fields => [
                qw(
                  id
                  create_date
                  manager_txt
                  partner_txt
                  timeout
                  owner
                  )
            ],
            all_locales => TRUE
        );
        $is_exists = TRUE if $message;
    }

    my @locales = sort keys %{$self->get_option('locales')};

    my @fields = (
        (
            $is_exists
            ? {
                name     => 'id',
                type     => 'hidden',
                required => TRUE,
                value    => $reason_id,
              }
            : {
                name     => 'id',
                type     => 'input',
                required => TRUE,
                value    => $reason_id,
              }
        ),
        (
            map +{
                name     => "partner_txt_$_",
                type     => 'input',
                label    => gettext('Text for partner (%s)', $_),
                required => FALSE,
                trim     => 1,
                ($is_exists ? (value => $message->{"partner_txt_$_"}) : ())
            },
            @locales
        ),
        (
            map +{
                name     => "manager_txt_$_",
                type     => 'input',
                label    => gettext('Text for manager (%s)', $_),
                required => FALSE,
                trim     => 1,
                ($is_exists ? (value => $message->{"manager_txt_$_"}) : ())
            },
            @locales
        ),
        {
            name  => "timeout",
            type  => 'select',
            label => gettext('Timeout'),
            items => [
                map {+{value => $_, label => $MODERATION_REASON_TIMEOUTS{$_}->(),}}
                sort {$a <=> $b} keys %MODERATION_REASON_TIMEOUTS
            ],
            required => TRUE,
            ($is_exists ? (value => $message->{timeout}) : ()),
        },
        {
            type    => 'button',
            subtype => 'submit',
            label   => ($is_exists ? gettext('Save reason') : gettext('Add reason')),
        },
    );

    return (
        check_rights => ($is_exists ? ['do_moderation_reason_action_edit'] : ['do_moderation_reason_action_add']),
        fields       => \@fields,
        save         => sub {
            my ($form) = @_;

            my %params;
            $params{id} = $reason_id unless $is_exists;
            $params{timeout} = $form->get_value('timeout');

            for my $f (qw(partner_txt manager_txt)) {
                my $ru = $form->get_value($f . "_ru");
                throw Exception::Form gettext('Field "%s" is required', $f) unless $ru;
                my $en = $form->get_value($f . "_en") || $ru;
                $params{$f} = {
                    ru => $ru,
                    en => $en,
                };
            }

            try {
                if ($is_exists) {
                    $self->moderation_reason->do_action($reason_id, 'edit', %params);
                } else {
                    $self->moderation_reason->add(%params);
                }
            }
            catch Exception::Validation::BadArguments with {
                my ($e) = @_;
                throw Exception::Form $e->message;
            }
            catch Exception::Validator::Fields with {
                my ($e) = @_;
                throw Exception::Form $e->message;
            };
        },
        redirect => 'moderation_reason_list'
    );
}

sub moderation_reason_do_action {
    my ($self, $action) = @_;

    throw Exception::Denied unless $self->check_rights('do_moderation_reason_action_' . $action);

    my $reason_id = $self->request->param('id');

    my $message = $self->moderation_reason->get($reason_id, fields => ['id']);
    throw Exception::Validation::BadArguments gettext('Unknown reason "%s"', $reason_id)
      unless defined $message;

    try {
        $self->moderation_reason->do_action($reason_id, $action);
    }
    catch Exception::Validation::BadArguments with {
        throw Exception::Form $@->message;
    };

    return $self->redirect('moderation_reason_list');
}

sub moderation_reason_delete : CMD {
    my ($self) = @_;
    return $self->moderation_reason_do_action('delete');
}

sub moderation_reason_restore : CMD {
    my ($self) = @_;
    return $self->moderation_reason_do_action('restore');
}

sub custom_bk_options_list : CMD {
    my ($self) = @_;

    throw Exception::Denied unless $self->check_rights('custom_bk_options_edit');

    my $vo = $self->get_vo(
        model       => $self->custom_bk_options,
        sort        => 'id',
        sortdesc    => TRUE,
        show_search => $self->check_rights('view_search'),
    );

    my $list;
    my $count;
    if ($vo->show_search_data()) {
        $list = $self->custom_bk_options->get_all(
            fields => [
                qw(
                  id
                  multistate_name
                  name
                  description
                  bk_name
                  bk_value
                  actions
                  )
            ],
            $vo->get_model_opts(),
        );
        $count = $self->custom_bk_options->found_rows();
    }

    return $self->from_bem_template(
        'devel/custom_bk_options.bem.tt2',
        vars => {
            custom_bk_list => $list,
            count          => $count,
            $vo->get_template_vars(),
        }
    );
}

sub custom_bk_options_edit : FORMCMD {
    my ($self) = @_;

    my $opt_id = $self->request->param('id');

    my $record;
    if ($opt_id) {
        $record = $self->custom_bk_options->get(
            $opt_id,
            fields => [
                qw(
                  id
                  name
                  description
                  bk_name
                  bk_value
                  )
            ],
        );
    }

    my @fields = (
        {
            name => 'id',
            type => 'hidden',
            ($record ? (value => $opt_id) : ()),
        },
        {
            name     => 'name',
            type     => 'input',
            label    => gettext('Name'),
            required => TRUE,
            trim     => TRUE,
            ($record ? (value => $record->{name}) : ()),
        },
        {
            name     => 'description',
            type     => 'input',
            label    => gettext('Description'),
            required => TRUE,
            trim     => TRUE,
            ($record ? (value => $record->{description}) : ()),
        },
        {
            name     => 'bk_name',
            type     => 'input',
            label    => gettext('BK name'),
            required => TRUE,
            trim     => TRUE,
            ($record ? (value => $record->{bk_name}) : ()),
        },
        {
            name     => 'bk_value',
            type     => 'input',
            label    => gettext('BK value'),
            required => TRUE,
            trim     => TRUE,
            ($record ? (value => $record->{bk_value}) : ()),
        },
        {
            type    => 'button',
            subtype => 'submit',
            label   => ($record ? gettext('Save') : gettext('Add')),
        },
    );

    return (
        check_rights => ['custom_bk_options_edit'],
        fields       => \@fields,
        save         => sub {
            my ($form) = @_;
            my $id = $form->get_value('id');
            my %params;

            foreach my $field_name (qw(name description bk_name bk_value)) {
                $params{$field_name} = $form->get_value($field_name);
            }

            try {
                if ($id) {
                    $self->custom_bk_options->do_action($id, 'edit', %params);
                } else {
                    $id = $self->custom_bk_options->add(%params);
                }
            }
            catch Exception::Validation::BadArguments with {
                my ($e) = @_;
                throw Exception::Form $e->message;
            }
            catch Exception::Validator::Fields with {
                my ($e) = @_;
                throw Exception::Form $e->message;
            };
            return 1;

        },
        redirect => 'custom_bk_options_list',
    );
}

sub custom_bk_options_do_action {
    my ($self, $action) = @_;

    throw Exception::Denied unless $self->check_rights('do_custom_bk_options_action_' . $action);

    my $opt_id = $self->request->param('id');

    my $opt = $self->custom_bk_options->get($opt_id, fields => ['id']);
    throw Exception::Validation::BadArguments gettext('Unknown option "%s"', $opt_id)
      unless defined $opt;

    try {
        $self->custom_bk_options->do_action($opt_id, $action);
    }
    catch Exception::Validation::BadArguments with {
        throw Exception::Form $@->message;
    };

    return $self->redirect('custom_bk_options_list');
}

sub custom_bk_options_delete : CMD {
    my ($self) = @_;
    return $self->custom_bk_options_do_action('delete');
}

sub custom_bk_options_restore : CMD {
    my ($self) = @_;
    return $self->custom_bk_options_do_action('restore');
}

TRUE;
