package WebInterface::Controller::Log;

use qbit;

use Exception::Validation::BadArguments::InvalidJSON;

use base qw(WebInterface::Controller);

__PACKAGE__->model_accessors(users => 'Application::Model::Users');

sub view_logs : CMD : DEFAULT {
    my ($self) = @_;

    my $model_accessor = $self->request->param('model');
    my %all_models     = %{$self->app->get_models()};
    return $self->not_found()
      unless $model_accessor
          && $all_models{$model_accessor};

    my $action_log_model_accessor = $self->app->$model_accessor->action_log_accessor();
    return $self->not_found()
      unless $action_log_model_accessor
          && $all_models{$action_log_model_accessor};

    return $self->denied()
      unless $self->check_rights($self->app->$model_accessor->get_right('view_action_log'));

    my $fields             = $self->app->$action_log_model_accessor->get_table_elem_keys();
    my %obj_id             = map {$_ => $self->request->param($_)} @$fields;
    my @additional_columns = grep {!defined($obj_id{$_})} keys(%obj_id);
    delete(@obj_id{@additional_columns});
    my $single_object = !@additional_columns;
    my $page_id_field_name;
    if ($self->app->$model_accessor->isa('Application::Model::Page')) {
        $page_id_field_name = $self->app->$model_accessor->get_page_id_field_name;
    }

    my %table_desc;
    my $title = $self->app->$action_log_model_accessor->get_product_name();
    if ($single_object) {
        my @model_fields = (@$fields, 'multistate');
        push @model_fields, $page_id_field_name
          if $page_id_field_name;

        my $obj = $self->app->$model_accessor->get(\%obj_id, fields => \@model_fields);
        return $self->not_found() unless $obj;

        my $id_string = '';
        my $id =
          $self->app->$model_accessor->can('public_id') ? $self->app->$model_accessor->public_id($obj) : $obj->{'id'};
        if ($obj && defined($page_id_field_name)) {
            $id_string = sprintf('(ID = %s, Page ID = %s)', $id, ($obj->{$page_id_field_name} // 'undef'));
        } else {
            $id_string = sprintf('(ID = %s)', $id);
        }
        $table_desc{title}           = $title . ' ' . $id_string;
        $table_desc{multistate}      = gettext('Current multistate: (%s) ', $obj->{'multistate'});
        $table_desc{desc_multistate} = $self->app->$model_accessor->get_multistate_name($obj->{'multistate'});
    } else {
        $table_desc{title} = $title;
        $table_desc{display_id} = $page_id_field_name ? 'PageID' : 'ID';
    }

    # GET: filter
    my @filters;
    my $filter = $self->request->param('filter');
    if ($filter) {
        my $invalid_json = FALSE;
        try {
            $filter = from_json($filter);
        }
        catch Exception::Validation::BadArguments::InvalidJSON with {
            $invalid_json = TRUE;
        };
        return $self->not_found()
          if $invalid_json;

        push(@filters, $filter)
          if (ref($filter) eq 'ARRAY' && @$filter)
          || (ref($filter) eq 'HASH' && %$filter);
    }

    if ($model_accessor eq 'business_rules') {
        # GET: login
        my $login = $self->request->param('login');
        if ($login) {
            my @rule_ids = map {$_->{'rule_id'}} @{
                $self->app->business_rules->get_all(
                    fields => qw(rule_id),
                    filter => [users => 'MATCH' => [login => '=' => $login]]
                )
              };
            $self->not_found() unless @rule_ids;
            push(@filters, [elem_rule_id => 'IN' => \@rule_ids]);
        }
    }

    # GET: id && campaign_id && page_id && rule_id ...
    push(@filters, map {["elem_${_}" => '=' => $obj_id{$_}]} keys(%obj_id));

    my $have_opts     = $self->app->$action_log_model_accessor->have_opts();
    my @elem_table_pk = map {"elem_${_}"} @$fields;
    my $elem_id       = @elem_table_pk == 1 ? $elem_table_pk[0] : 'elem_id';
    my $logs          = $self->app->$action_log_model_accessor->get_all(
        fields => [
            ($single_object ? () : 'elem_public_id'),
            @elem_table_pk,
            qw(dt login old_multistate old_multistate_name new_multistate new_multistate_name comment action),
            ($have_opts ? 'data_opts' : ())
        ],
        (@filters ? (filter => ['AND', \@filters]) : ()),
        order_by => [['dt', TRUE], @elem_table_pk],
        limit    => 1000
    );

    my %elem_ids;    # gather uniq values for ID column
                     # opts pretty printing
    foreach (@$logs) {
        $elem_ids{$_->{$elem_id}} = $_->{$elem_id};
        if ($_->{'data_opts'} && %{$_->{'data_opts'}}) {
            $_->{'data_opts'} = to_json($_->{'data_opts'}, pretty => TRUE);
        } elsif ($_->{'data_opts'}) {
            delete($_->{'data_opts'});
        }
    }

    # PublicID or PageID
    unless ($single_object) {
        unless ($page_id_field_name) {
            $_->{'display_id'} = $_->{'elem_public_id'} foreach @$logs;
        } else {
            my %id_page_id_map;

            # For internal models PageID equals to ID
            if (in_array($model_accessor, $self->app->product_manager->get_external_product_accessors())) {
                my $page_ids = $self->app->$model_accessor->get_all(
                    fields => [('id', $page_id_field_name)],
                    filter => [id => 'IN' => [sort keys(%elem_ids)]]
                );
                %id_page_id_map = map {($_->{'id'} => $_->{$page_id_field_name})} @$page_ids;
            } else {
                %id_page_id_map = %elem_ids;
            }

            foreach (@$logs) {
                my $page_id = $id_page_id_map{$_->{$elem_id}};
                if ($page_id) {
                    $_->{'display_id'} = $page_id;
                } else {
                    $_->{'display_id'} = $_->{'elem_public_id'} . ' (id)';
                }
            }
        }
    }

    return $self->from_bem_template('log/list.bem.tt2', vars => {%table_desc, logs => $logs});
}

TRUE;
