package WebInterface::Controller::News;

use qbit;

use base qw(WebInterface::Controller);

use Exception::Form;

__PACKAGE__->model_accessors(news => 'Application::Model::News');

__PACKAGE__->register_rights(
    [
        {
            name        => 'news',
            description => d_gettext('Rights to use news manager'),
            rights      => {news_list_view => d_gettext('Right to view news list')},
        }
    ]
);

sub add : FORMCMD {
    my ($self) = @_;

    return (
        title => gettext('News adding'),

        check_rights => ['news_add'],

        fields => [
            {
                name     => 'date',
                label    => gettext('Date'),
                type     => 'input',
                length   => 10,
                hint     => '2013-12-31',
                required => TRUE,
                check    => sub {
                    my ($field, $value) = @_;

                    throw Exception::Form gettext('Invalid date') unless $value =~ /^[0-9]{4}-[0-9]{2}-[0-9]{2}$/;
                  }
            },
            {name => 'title', label => gettext('Title'), type => 'input',    length => 1000, required => TRUE},
            {name => 'text',  label => gettext('Text'),  type => 'textarea', rows   => 20},
            {
                name      => 'show_on_main_page',
                label     => gettext('Show on main page'),
                type      => 'switch',
                on_label  => gettext('Yes'),
                off_label => gettext('No')
            },
            {label => gettext('Add'), type => 'button', subtype => 'submit'}
        ],

        save => sub {
            my ($form) = @_;

            my $id = $self->news->add(map {$_ => $form->get_value($_)} qw(date title text show_on_main_page));

            $form->{'redirect_opts'}{'search_json'} = to_json({id => $id});
        },

        redirect => 'list'
    );
}

sub edit : CMD : SAFE {
    my ($self) = @_;

    return $self->denied() unless $self->check_rights('news_edit');

    my %new_values =
      map {my $v = $self->request->param($_); defined($v) ? ($_ => $v) : ()} qw(date title text show_on_main_page);

    $self->news->edit($self->request->param('id'), %new_values);

    return $self->as_json(TRUE);
}

sub list : CMD {
    my ($self) = @_;

    return $self->denied() unless $self->check_rights('news_list_view');

    my $vo = $self->get_vo(model => $self->news, sort => 'date', sortdesc => TRUE);

    my $news = $self->news->get_all(
        fields => [qw(id date title text show_on_main_page)],
        $vo->get_model_opts()
    );

    return $self->from_bem_template(
        'news/list.bem.tt2',
        vars => {
            news           => $news,
            news_count     => $self->news->found_rows(),
            search_example => 'title like "*Яндекс*"',
            $vo->get_template_vars(),
        }
    );

    return $self->from_bem_template('');
}

# export NEWS for PI1
sub list_pi_one_rpt : CMD : WOAUTH {
    my ($self) = @_;

    # access only from PI 1
    return $self->denied()
      unless in_array(
        $self->request->remote_addr(),
        [
            '178.154.224.237', '93.158.132.150', '93.158.132.151', '178.154.224.129', '178.154.224.130',
            '178.154.224.229'
        ]
      );

    my $vo = $self->get_vo(model => $self->news, sort => 'date', sortdesc => TRUE);

    my $news = $self->news->get_all(
        fields => [qw(id date title text show_on_main_page)],
        $vo->get_model_opts()
    );

    return $self->as_json($news);
}

sub top : CMD : WOAUTH {
    my ($self) = @_;

    return $self->as_json(
        $self->news->get_all(
            fields   => [qw(id date title)],
            filter   => {show_on_main_page => TRUE},
            order_by => [[date => TRUE]],
            limit    => 2
        )
    );
}

TRUE;
