package WebInterface::Controller::Queue;

use qbit;

use base qw(WebInterface::Controller);

use Exception::Denied;
use Exception::Form;
use Utils::PublicID;

__PACKAGE__->model_accessors(
    queue      => 'Application::Model::Queue',
    statistics => 'Application::Model::Statistics',
    users      => 'Application::Model::Users',
    support    => 'Application::Model::Queue::Support',
);

sub update_in_bk : FORMCMD {
    my ($self) = @_;

    throw Exception::Denied
      unless $self->check_rights('queue_add_update_in_bk');

    return (
        title  => gettext('Send Page IDs to BK'),
        fields => [
            {
                name     => 'page_ids',
                label    => gettext('Page IDs'),
                type     => 'textarea',
                required => TRUE,
                trim     => TRUE,
            },
            {label => gettext('Send'), type => 'button', subtype => 'submit'},
        ],
        save => sub {
            my ($form) = @_;

            # values split by spaces or commas with possible extra spaces around them
            my @page_ids = @{array_uniq([split(/\s*(?:,|\s)\s*/, $form->get_value('page_ids'))])};

            unless (@page_ids) {
                throw Exception::Form gettext('You must enter at least one Page ID');
            }

            foreach my $page_id (@page_ids) {
                unless ($page_id =~ /^[0-9]+\z/) {
                    throw Exception::Form gettext('Incorrect Page ID: "%s"', $page_id);
                }
            }

            try {
                $self->queue->add(
                    method_name => 'update_in_bk',
                    params      => {page_ids => \@page_ids,},
                );

            }
            catch {
                throw Exception::Form $_[0]->message;
            };

            $form->{'complete_message'} =
              gettext('%s Page ID added to queue: %s', scalar(@page_ids), join(', ', sort {$a <=> $b} @page_ids));

            $form->{'redirect_opts'} = {new_url => '/v2/queue/list'};

            return FALSE;
        },

        ajax_action => 'redirect',
    );
}

sub continue_update : CMD {
    my ($self) = @_;

    my $id = $self->request->param('id');
    $self->queue->do_action($id, 'restart');

    return $self->as_json(
        $self->queue->get(
            $id, fields => [qw(id multistate_name remaining_time completed_percent can_be_run error_description)],
        )
    );
}

sub list : CMD : DEFAULT {
    my ($self) = @_;

    my $types = $self->queue->get_type();

    return $self->denied() unless @$types;

    my $vo = $self->get_vo(
        model    => $self->queue,
        sort     => 'add_dt',
        sortdesc => TRUE,
    );

    my $queue = $self->queue->get_all(
        fields => [
            qw(
              id
              add_dt
              multistate_name
              remaining_time
              completed_percent
              can_be_run
              login
              error_description
              method_name
              method_name_view
              log
              additional_info
              )
        ],
        $vo->get_model_opts(),
    );

    return $self->from_bem_template(
        'queue/list.bem.tt2',
        vars => {
            queue           => $queue,
            total_count     => $self->queue->found_rows(),
            available_queue => $self->queue->get_available_queue(),
            types           => $types,
        }
    );
}

sub statistics_intake : FORMCMD {
    my ($self) = @_;

    return (
        title => gettext('Adding task to update statistics'),

        check_rights => ['queue_add_statistics_intake'],

        fields => [
            {
                name         => 'login',
                label        => gettext('Login'),
                type         => 'input',
                multisuggest => {cmd => 'login_sgst', path => 'settings', opts => {all_partners => TRUE}},
            },
            {
                name     => 'range',
                label    => gettext('Period'),
                type     => 'daterange',
                required => TRUE,
            },
            (
                sort {
                        $a->{'sort_priority'} && $b->{'sort_priority'}
                      ? $a->{'sort_priority'} <=> $b->{'sort_priority'} || $a->{'after_label'} cmp $b->{'after_label'}
                      : $a->{'after_label'} cmp $b->{'after_label'}
                  }
                  map {
                    my $method = "statistics_$_";
                    my ($title, $priority);
                    try {
                        $title    = $self->statistics->get_level_full_title($_);
                        $priority = $self->statistics->get_level_sort_priority($_);
                    };
                    +{
                        name          => 'rel_product__' . $self->app->$method->id(),
                        type          => 'switch',
                        after_label   => $title // $_,
                        off_label     => gettext("don't reload"),
                        on_label      => gettext('reload'),
                        sort_priority => $priority // 9999,
                     }
                  } $self->statistics->get_level_ids_for_reloading()
            ),
            {label => gettext('Add'), type => 'button', subtype => 'submit'}
        ],

        save => sub {
            my ($form) = @_;

            my @products =
              grep {$form->get_value("rel_product__$_")}
              map {$_ && /^rel_product__(.+)$/ ? $1 : ()} $form->get_field_names();

            throw Exception::Form gettext('Please select products for reloading') unless @products;

            my $login = [split(/,/, $form->get_value('login'))];
            my $start = $form->get_value('range')->{'from'};
            my $end   = $form->get_value('range')->{'to'};

            $self->queue->add(
                method_name => 'statistics_intake',
                params      => {
                    start    => $start,
                    end      => $end,
                    products => \@products,
                    (@{$login} ? (logins => $login) : ()),
                },
            );

            $form->{'redirect_opts'} = {new_url => '/v2/queue/list'};

            return FALSE;
        },

        ajax_action => 'redirect',
    );
}

sub send_email_to_processing : FORMCMD {
    my ($self) = @_;

    return (
        title => gettext('Adding task to send email to processing about user'),

        check_rights => ['queue_add_send_email_to_processing'],

        fields => [
            {
                label        => gettext('Login'),
                multisuggest => {cmd => 'login_sgst', path => 'settings', opts => {all_partners => TRUE}},
                name         => 'login',
                required     => TRUE,
                type         => 'input',
            },
            {label => gettext('Add'), type => 'button', subtype => 'submit'}
        ],

        save => sub {
            my ($form) = @_;

            my @logins = split(/\s*,\s*/, $form->get_value('login'));
            my $ids = $self->users->get_all(
                fields => [qw(id)],
                filter => {login => \@logins},
            );

            $self->queue->add(
                method_name => 'send_email_to_processing',
                params      => {user_id => $_->{id},},
            ) foreach @$ids;

            $form->{'redirect_opts'} = {new_url => '/v2/queue/list'};

            return FALSE;
        },

        ajax_action => 'redirect',
    );
}

sub update : CMD {
    my ($self) = @_;

    return $self->denied() unless @{$self->queue->get_type};

    my $ids = $self->request->param_array('id');
    $ids = [-1] unless @$ids;

    my $tasks = $self->queue->get_all(
        fields => [qw(id multistate_name remaining_time completed_percent can_be_run error_description)],
        filter => {id => $ids}
    );

    return $self->as_json($tasks);
}

sub change_blocked_brands : FORMCMD {
    my ($self) = @_;

    $self->support->get_object('change_blocked_brands')->make_process($self->app);
}

sub turn_on_unmoderate_dsp : FORMCMD {
    my ($self) = @_;

    $self->support->get_object('turn_on_unmoderate_dsp')->make_process($self->app);
}

sub do_action : WIZARDCMD {
    my ($self) = @_;

    return (
        caption => gettext('Do action'),

        check_rights => ['queue_add_do_action'],

        steps => [
            {
                title  => gettext('Select the model'),
                fields => sub {
                    my ($wizard) = @_;

                    my $items = [
                        map {{label => $self->app->$_->get_product_name(), value => $_}}
                          grep {$self->app->$_->DOES('Application::Model::Role::JavaJsonApiProxy')}
                          @{$self->app->product_manager->get_block_model_accessors()}
                    ];

                    return (
                        {
                            name     => 'model',
                            label    => gettext('Model'),
                            type     => 'select',
                            required => TRUE,
                            items    => $items,
                        },
                        {type => 'button', subtype => 'submit', label => gettext('Next')},
                    );
                },
            },

            {
                title  => gettext('Select the action and other params'),
                fields => sub {
                    my ($wizard) = @_;

                    my $model = $wizard->get_value('model');

                    my $actions = $self->app->$model->get_all_available_multistate_actions();

                    my $has_page_id_field = exists($self->app->$model->get_model_fields()->{'page_id'});

                    return (
                        {
                            label    => gettext('Login'),
                            suggest  => {cmd => 'login_sgst', path => 'settings'},
                            name     => 'login',
                            required => TRUE,
                            type     => 'input',
                            value => $self->get_option('cur_user', {login => ''})->{'login'},
                        },
                        {
                            name     => 'action',
                            label    => gettext('Action'),
                            type     => 'select',
                            required => TRUE,
                            items    => [
                                map {{label => $actions->{$_}, value => $_,}}
                                sort {$actions->{$a} cmp $actions->{$b}} keys(%$actions)
                            ]
                        },
                        {
                            name     => 'ids',
                            label    => gettext('IDs'),
                            type     => 'textarea',
                            required => !$has_page_id_field,
                            trim     => TRUE,
                        },
                        (
                            $has_page_id_field
                            ? {
                                label => gettext('Page ID'),
                                name  => 'page_id',
                                type  => 'input',
                              }
                            : ()
                        ),
                        {label => gettext('Add'), type => 'button', subtype => 'submit'}
                    );
                },
            }
        ],

        save => sub {
            my ($wizard) = @_;

            my $login = $wizard->get_value('login');
            my $user = $self->users->get_all(fields => ['id'], filter => {login => $login})->[0]
              // throw Exception::Form gettext('User with login "%s" not found', $login);

            my @ids = split(/\s*,\s*|\n/, $wizard->get_value('ids') // '');
            my $page_id = $wizard->get_value('page_id');

            throw Exception::Form gettext('Set ids or page_id') if @ids == 0 && !defined($page_id);

            my $id = $self->queue->add(
                method_name => 'do_action',
                # если указать только id такие таски будут выстраиваться в очередь
                # если указать page_id должно работать нормально
                # а модели без page_id будут так же выстраиваться в очередь
                # надеюсь мы такие модели будем переносить за раз
                group_id => defined($page_id) ? $page_id : 0,
                params => {
                    userId     => $user->{'id'},
                    modelName  => $wizard->get_value('model'),
                    actionName => $wizard->get_value('action'),
                    (@ids              ? (modelIds => \@ids)    : ()),
                    (defined($page_id) ? (pageId   => $page_id) : ()),
                },
            );

            $wizard->{'redirect_opts'} = {new_url => '/v2/queue/list'};

            return FALSE;
        },

        ajax_action => 'redirect',
    );
}

TRUE;
