package WebInterface::Controller::Settings;

use qbit;

use base qw(WebInterface::Controller);

use PiConstants qw($PARTNERS_INTERNAL_ACCESS_REQUESTS_MAIL);

use Exception::Denied;
use Exception::Form;

__PACKAGE__->model_accessors(
    api_blackbox        => 'QBit::Application::Model::API::Yandex::BlackBox',
    mail_notification   => 'Application::Model::MailNotification',
    rbac                => 'Application::Model::RBAC',
    statistics_charging => 'Application::Model::Statistics::Charging',
    statistics_payment  => 'Application::Model::Statistics::Payment',
    users               => 'Application::Model::Users',
);

__PACKAGE__->register_rights(
    [
        {
            name        => 'settings_cmd',
            description => d_gettext('Rights to view settings pages'),
            rights      => {
                settings_cmd_ucard  => d_gettext('Right to view user card'),
                feedback_for_advnet => d_gettext('Right to view feedback for ADVNET'),
                feedback_for_dsp    => d_gettext('Right to view feedback for DSP'),
                view_foot_help      => d_gettext('Right to view block "help"')
            }
        }
    ]
);

sub banner_store_form : FORMCMD {
    my ($self) = @_;

    my $cur_user = $self->get_option('cur_user');

    throw Exception::Denied gettext("You don't have access to this page")
      unless $self->users->check_action($cur_user->{id}, 'created_partner_in_banner_store');

    return (
        title => gettext('Media repository registration'),

        fields => [
            {
                type  => 'bemjson',
                value => sub {
                    "{
                         tag: 'p',
                         content: '"
                      . QBit::WebInterface::Controller::BEMHTML::_bem_encode(
                        gettext(
'You should create an account in a media repository to get started. Remember your login and password, you\'ll need it to access the API. After this step, you will be able to download the documentation.'
                        )
                      )
                      . "'
                     }"
                  }
            },
            {
                name     => 'login',
                label    => gettext('Login'),
                type     => 'input',
                value    => $cur_user->{'login'} || '',
                disabled => 'disabled',
                required => 1,
                trim     => 1
            },
            {
                name       => 'password',
                label      => gettext('Create a password'),
                type       => 'input',
                input_type => 'password',
                required   => 1,
                trim       => 1,
            },
            {
                name       => 'password_copy',
                label      => gettext('Repeat password'),
                type       => 'input',
                input_type => 'password',
                required   => 1,
                trim       => 1
            },
            {
                name     => 'company_name',
                label    => gettext('Company name'),
                type     => 'input',
                length   => 50,
                required => 1,
                trim     => 1,
            },
            {label => gettext('Save'), type => 'button', subtype => 'submit'},
        ],

        save => sub {
            my ($form) = @_;

            throw Exception::Form gettext('Passwords do not match')
              if $form->get_value('password') ne $form->get_value('password_copy');

            $self->users->do_action(
                $cur_user->{'id'},
                'created_partner_in_banner_store',
                password     => $form->get_value('password'),
                company_name => $form->get_value('company_name')
            );

            $form->{'redirect_opts'} = {path => 'dashboard'};
        },

        redirect => '',
    );
}

sub login_sgst : CMD {
    my ($self) = @_;

    my $users = $self->users->suggest_login(
        hash_transform(
            {map {$_ => $self->request->param($_)} $self->request->param_names()},
            [],
            {
                part                => 'login_part',
                all_partners        => 'partner',
                an_inernal_managers => 'adv_net_internal_manager',
                dsp                 => 'dsp_partner',
                dsp_managers        => 'dsp_manager',
                an_partner          => 'adv_net_partner',
                video_an_partner    => 'video_an_partner',
            }
        )
    );

    return $self->as_jsonp(
        $self->request->param('callback'),
        [$self->request->param('part'), [map {[user => $_->{'login'} => $_->{'full_name'}]} @$users], {r => 1, n => 0}],
    );
}

sub set_role : FORMCMD {
    my ($self) = @_;

    return (
        fields => [
            {
                name     => 'login',
                label    => gettext('Login'),
                type     => 'input',
                suggest  => {cmd => 'login_sgst'},
                value    => $self->request->param('login'),
                required => 1,
                trim     => 1
            },
            {
                name  => 'role',
                label => gettext('Role'),
                type  => 'select',
                items => [map {{value => $_->{'id'}, label => $_->{'name'}}} @{$self->rbac->get_roles()}],
            },
            {label => gettext('Save'), type => 'button', subtype => 'submit'},
        ],

        check_rights => ['rbac_user_role_set'],

        save => sub {
            my ($form) = @_;

            my $user = $self->users->get_by_login($form->get_value('login'), fields => ['id'])
              // throw Exception::Form gettext('User does not exists');

            # throw Dumper($form);
            $self->rbac->set_user_role($user->{'id'}, $form->get_value('role'));

            $form->{'redirect_opts'} = {path => 'users', search => "(id = $user->{'id'})"};
        },

        redirect => 'list',
    );
}

sub ucard : CMD {
    my ($self) = @_;

    return $self->denied() unless $self->check_rights('settings_cmd_ucard');

    my $user = $self->users->get_by_login($self->request->param('login'),
        fields => [qw(login full_name email client_id contracts phone)])
      || return $self->error(gettext('Not found'));

    my @groups = ([gettext('Client ID') => 'client_id'], [gettext('E-Mail') => 'email', gettext('Phone') => 'phone'],);

    my @group_fields;
    foreach my $group (@groups) {
        my @ugroup;
        while (my ($title, $key) = splice(@$group, 0, 2)) {
            push(@ugroup, {title => $title, value => $user->{$key}});
        }
        push(@group_fields, \@ugroup) if @ugroup;
    }
    $user->{'groups'} = \@group_fields;

    return $self->from_bem_template(
        'settings/ucard.bem.tt2',
        vars => {
            user                  => $user,
            available_stat_levels => {map {$_ => $self->$_->is_available()} qw(statistics_payment statistics_charging)}
        }
    );
}

TRUE;
