package WebInterface::Controller::Utils;

use qbit;

use base qw(WebInterface::Controller);

use Utils::CSV;
use Utils::XLS;

use Exception::Validation::BadArguments;
use Exception::Denied;

my %CONTENT_TYPES = (
    csv  => 'text/csv',
    tsv  => 'text/tab-separated-values',
    xls  => 'application/vnd.ms-excel',
    xlsx => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
);

use PiConstants qw( $XLS_LIMIT  $XLSX_LIMIT);

sub save_as_xls2 : CMD {
    _save_as2(@_, format => 'xls');
}

sub save_as_xlsx2 : CMD {
    _save_as2(@_, format => 'xlsx');
}

sub save_as_csv2 : CMD {
    _save_as2(@_, format => 'csv');
}

sub save_as_tsv2 : CMD {
    _save_as2(@_, format => 'tsv');
}

sub _save_as2 {
    my ($self, %opts) = @_;

    my $format = $opts{format} // 'xls';
    throw gettext('Unknown iformat "%s"', $format) unless $CONTENT_TYPES{$format};

    my $rows_limit = undef;
    $rows_limit = $XLS_LIMIT  if $format eq 'xls';
    $rows_limit = $XLSX_LIMIT if $format eq 'xlsx';

    my $filename = $self->request->param('filename', "data.$format");
    my $report_hash = $self->request->param('report_hash');
    my $data;
    if (!$report_hash && $self->check_rights('statistics_mol_view')) {
        my $query_data = $self->request->param('query');
        throw Exception::Validation::BadArguments gettext('Missed required fields: %s', 'query') unless $query_data;
        my $query = from_json $query_data;

        $data = $self->app->statistics->get_data_from_hash2(
            query      => $query,
            rows_limit => $rows_limit
        );
    } else {
        throw gettext('Unknown iformat "%s"', $format) unless $CONTENT_TYPES{$format};

        my @missed_req_fields = grep {!$self->request->param($_)} qw(report_hash sort_by sort_order);

        throw Exception::Validation::BadArguments gettext('Missed required fields: %s', join(', ', @missed_req_fields))
          if @missed_req_fields;

        $data = $self->app->statistics->get_data_from_hash2(
            report_hash => $report_hash,
            rows_limit  => $rows_limit
        );
    }
    my $result;
    if ($format =~ /^xlsx?$/) {
        $result = xls_with_fields_and_names($data->{'sorted_stat_data'}, $data->{'fields'}, %opts);
    } elsif ($format =~ /^[ct]sv$/) {
        $result = csv_with_fields_and_names($data->{'sorted_stat_data'},
            $data->{'fields'}, ($format eq 'tsv' ? (sep_char => "\t") : ()), %opts);
    }

    return $self->send_file(
        data         => $result,
        filename     => $filename,
        content_type => $CONTENT_TYPES{$format},
    );
}

TRUE;
