package Yandex::ArcanumAPI;

=encoding UTF-8

=head1 DESCRIPTION

Yandex::ArcanumAPI - небольшая perl библиотека, которая реализует
работу с Arc API по аналогии с Yandex::Startrek

Библиотека поддерживает лишь часть API.

Библиотека сделана для того чтобы была единая точка входа в API
котoрую можно было бы просто менять и дорабатывать.

L<https://docs.yandex-team.ru/arcanum/communication/public-api>

=head1 SYNOPSYS

    my $arc_api = Yandex::ArcanumAPI->new(
        oauth_token => $ENV{'ARC_TOKEN'}
    );

    my $pr_info = $arc_api->get_pr($pr_id => [qw/status summary/]);

    # $pr_info = {
    #   status  => 'merged',
    #   summary => 'PI-xxx add some stuff'
    # }

=cut

use strict;
use warnings FATAL => 'all';
use feature 'say';
use utf8;
use open qw(:std :utf8);

use Carp;
use LWP::UserAgent;
use IO::Socket::SSL qw(SSL_VERIFY_NONE);
use HTTP::Request;
use JSON::PP;
use URI;
use Time::Local;

=head1 new

    my $arc = Yandex::ArcanumAPI->new(
        oauth_token => 'abc...',
    );

=cut

sub new {
    my ($class, %params) = @_;

    my $self = {};
    bless $self, $class;

    $self->{_oauth_token} = delete $params{oauth_token};
    croak "No oauth_token" if not defined $self->{_oauth_token};

    $self->{_retry} = delete $params{retry} // 1;
    $self->{_delay} = delete $params{delay} // 0;
    $self->{_lwp} = LWP::UserAgent->new(ssl_opts => {verify_hostname => 0, SSL_verify_mode => SSL_VERIFY_NONE});

    return $self;
}

=head1 get_pr

Получить параметры Pull-Request

    $arc_api->get_pr( $pd_id, \@field_list );

L<https://a.yandex-team.ru/api/swagger/index.html#/public_api/getPullRequest>

=cut

sub get_pr {
    my ($self, $pr_id, $fields) = @_;

    croak "No PR id" if not defined $pr_id;
    croak "Incorrect PR id" if $pr_id !~ /^\d+$/;

    my ($json, $status, $response_str) = $self->_get_api_response(
        method => 'GET',
        url    => "/v1/pull-requests/$pr_id?fields=" . join("%2C", @$fields),
    );
    croak "Got unexpected status $status. Full content: $response_str" unless $json;

    return $json;
}

sub _get_api_response {
    my ($self, %opts) = @_;

    my $request = HTTP::Request->new(
        $opts{method} => "https://a.yandex-team.ru/api" . $opts{url},
        [
            'Authorization' => 'OAuth ' . $self->{_oauth_token},
            'Content-Type'  => 'application/json',
        ],
        $opts{content_data} ? encode_json($opts{content_data}) : ()
    );

    my $response;
    while (1) {
        $response = $self->{_lwp}->request($request);

        last if $response->is_success || --$self->{_retry} == 0;

        sleep($self->{_delay});
        warn
          sprintf("Retrying %s %s, previous request returned %s\n", $opts{method}, $opts{url}, $response->status_line);
    }

    my $content = $response->decoded_content // 'No content';

    my $json;
    if ($response->is_success) {
        $json = eval {decode_json($content)->{data}};
        warn $@ if $@;
    }

    utf8::decode($content) unless utf8::is_utf8($content);

    return ($json, $response->code, $content);
}

1;
