package Yandex::StarTrek;

=encoding UTF-8

=head1 DESCRIPTION

Yandex::StarTrek - небольшая perl библиотека, которая реализует работу со
внутренним трекером Яндекса.

Библиотека поддерживает лишь часть API.

Библиотека сделана для того чтобы была единая точка входа в API трекера,
котрую можно было бы просто менять и дорабатывать.

L<https://wiki.yandex-team.ru/tracker/api>

=head1 SYNOPSYS

    my $issue_key = Yandex::StarTrek->new(oauth_token => $ENV{'TRACKER_TOKEN'});

    my $issue_key = $ys->create_issue(
        queue => "TEST",
        summary => "Тема задачи",
    );

=cut

use strict;
use warnings FATAL => 'all';
use feature 'say';
use utf8;
use open qw(:std :utf8);

use Carp;
use Net::INET6Glue::INET_is_INET6;
use LWP::UserAgent;
use IO::Socket::SSL qw(SSL_VERIFY_NONE);
use HTTP::Request;
use JSON::PP;
use URI;
use Time::Local;

our $ticket_re = qr/^[A-Z]{2,}-\d+$/;

=head1 new

    my $ys = Yandex::StarTrek->new(
        oauth_token => 'abc...',
    );

=cut

sub new {
    my ($class, %params) = @_;

    my $self = {};
    bless $self, $class;

    $self->{_oauth_token} = delete $params{oauth_token};
    croak "No oauth_token" if not defined $self->{_oauth_token};

    $self->{_retry} = delete $params{retry} // 1;
    $self->{_delay} = delete $params{delay} // 0;
    $self->{_lwp} = LWP::UserAgent->new(ssl_opts => {verify_hostname => 0, SSL_verify_mode => SSL_VERIFY_NONE});

    return $self;
}

=head1 ping

Метод для проверки работы API. Возвращет true если для указанного
OAuth токена есть доступ в API, иначе false.

=cut

sub ping {
    my ($self) = @_;

    my ($json, $status, $response_str) = $self->_get_api_response(
        method => 'GET',
        url    => '/v2/myself',
    );

    if ($json) {
        return 1;
    } elsif ($status eq '401') {
        return 0;
    } else {
        croak "Got unexpected status $status. Full content: $response_str";
    }
}

=head1 create_issue

    my $issue_key = $ys->create_issue(
        queue => "TEST",
        summary => "Тема задачи",
    );

Параметры queue и summary являются обязательными.

Метод возвращает id созданного тикета (например "TEST-814").

В случае каких-либо проблем метод умирает.

Опционально можно передавать следующие параметры:

=over

=item followers - массив со списком логинов людей кто будет наблюдателями
созданного тикета

=item description - описание тикета

=item assignee - исполнитель

=item type - тип тикета (например, 'bug' или 'release', полный список типов
можно получить из метода issuetypes()), по умолчанию тикет создается с типом
task

=item parent - id тикета, который станет родителем для нового тикета

=item fix_versions - массив строк с версиями

=item tags - массив строк с тэгами

=item components - массив компонентов

=item abcService - массив ABC Service

=item links - массив links

=back

L<https://wiki.yandex-team.ru/tracker/api/issues/create>

=cut

sub create_issue {
    my ($self, %params) = @_;

    my $queue = delete $params{queue};
    croak 'No queue' if not defined $queue;

    my $summary = delete $params{summary};
    croak 'No summary' if not defined $summary;

    my $followers = delete $params{followers};
    croak 'Incorrect followers' if defined $followers and ref($followers) ne 'ARRAY';

    my $fix_versions = delete $params{fix_versions};
    croak 'Incorrect fix_versions' if defined $fix_versions and ref($fix_versions) ne 'ARRAY';

    my $tags = delete $params{tags};
    croak 'Incorrect tags' if defined $tags and ref($tags) ne 'ARRAY';

    my $components = delete $params{components};
    croak 'Incorrect components' if defined $components and ref($components) ne 'ARRAY';

    my $abcService = delete $params{abcService};
    croak 'Incorrect ABC Service' if defined $abcService and ref($abcService) ne 'ARRAY';
    foreach (@$abcService) {
        croak 'Incorrect ABC Service' if ref($_) ne 'HASH';
    }

    my $links = delete $params{links};
    croak 'Incorrect links' if defined $links and ref($links) ne 'ARRAY';
    foreach (@$links) {
        croak 'Incorrect links' if ref($_) ne 'HASH';
    }

    my $description         = delete $params{description};
    my $assignee            = delete $params{assignee};
    my $parent              = delete $params{parent};
    my $deadline            = delete $params{deadline};
    my $serviceServiceGroup = delete $params{serviceServiceGroup};
    my $sourceLanguage      = delete $params{sourceLanguage};
    my $targetLanguage      = delete $params{targetLanguage};

    my $type = delete $params{type};
    $type = 'task' if not defined $type;

    croak "Got unknown params: " . join(', ', keys %params) if %params;

    if (defined $fix_versions) {
        foreach my $version (@{$fix_versions}) {
            # Version must be string. StarTrek API thorws error if the version is number.
            $version = $version . '';
        }
    }

    my ($json, $status, $response_str) = $self->_get_api_response(
        method       => 'POST',
        url          => '/v2/issues',
        content_data => {
            queue   => $queue,
            summary => $summary,
            type    => $type,
            ($followers                    ? (followers           => $followers)           : (),),
            ($components                   ? (components          => $components)          : (),),
            ($abcService                   ? (abcService          => $abcService)          : (),),
            ($links                        ? (links               => $links)               : (),),
            (defined($description)         ? (description         => $description)         : (),),
            (defined($assignee)            ? (assignee            => $assignee)            : (),),
            (defined($parent)              ? (parent              => $parent)              : (),),
            (defined($fix_versions)        ? (fixVersions         => $fix_versions)        : (),),
            (defined($tags)                ? (tags                => $tags)                : (),),
            (defined($deadline)            ? (deadline            => $deadline)            : (),),
            (defined($serviceServiceGroup) ? (serviceServiceGroup => $serviceServiceGroup) : (),),
            (defined($sourceLanguage)      ? (sourceLanguage      => $sourceLanguage)      : (),),
            (defined($targetLanguage)      ? (targetLanguage      => $targetLanguage)      : (),),
        },
    );

    croak "Got unexpected status $status. Full content: $response_str" unless $json;

    return $json->{key};
}

=head1 add_comment

    $ys->add_comment($issue_key, $comment_text, $summonees);

Значения $issue_key и $comment_text являются обязательными.

Метод ничего не возвращает.

В случае каких-либо проблем метод умирает.

L<https://wiki.yandex-team.ru/tracker/api/issues/comments/create>

=cut

sub add_comment {
    my ($self, $issue_key, $comment, $summonees) = @_;

    croak 'No issue_key' if not defined $issue_key;
    croak 'No comment'   if not defined $comment;

    my ($json, $status, $response_str) = $self->_get_api_response(
        method       => 'POST',
        url          => "/v2/issues/$issue_key/comments",
        content_data => {
            text => $comment,
            $summonees ? (summonees => $summonees) : ()
        },
    );
    croak "Got unexpected status $status. Full content: $response_str" unless $json;
    my ($year, $mon, $mday, $hour, $min, $sec) = split(/[T.:-]+/, $json->{createdAt});
    my $comment_timestamp = timegm($sec, $min, $hour, $mday, $mon - 1, $year);

    return $comment_timestamp . '000' // 0;
}

=head1 get_comments

    $ys->get_comments( $issue_key );

Возвращает структуру с комментариями тикета.

L<https://wiki.yandex-team.ru/tracker/api/issues/comments/list/>

=cut

sub get_comments {
    my ($self, $issue_key) = @_;

    croak "No issue key" if not defined $issue_key;
    croak "Incorrect issue key" if $issue_key !~ $ticket_re;

    my ($json, $status, $response_str) = $self->_get_api_response(
        method => 'GET',
        url    => "/v2/issues/$issue_key/comments",
    );
    croak "Got unexpected status $status. Full content: $response_str" unless $json;

    return $json;
}

=head1 search

    $ys->search(
        query => 'Queue: PI and Status: Closed',
    );

или:

    $ys->search(
        filters => [
            'queue:TEST',
        ],
    );

Нужно обязательно указать либо параметр query, либо параметр filters.

Рекомендуется использовать query. Описание языка запросов для query есть
на странице L<https://wiki.yandex-team.ru/tracker/vodstvo/query#poljaiprimeryispolzovanija>.

И в query рекомендуется использовать id очередей, а не их названия (название
может поменяться, а id очереди - это более стабильная штука).

Опционально можно передавать следующие параметры:

=over

=item page - страница данных. По умолчанию - 1

=item per_page - количество тикетов на странице (макс. 100), по-умолчанию - 50

=item fields - ARRAYREF с перечислением полей которые нужно вернуть. Если
параметр не указан, то возвращается дефолтный набор полей.

=back

L<https://wiki.yandex-team.ru/tracker/api/issues/list>

=cut

sub search {
    my ($self, %params) = @_;

    my $query   = delete $params{query};
    my $filters = delete $params{filters};

    croak 'No query and no filters' if !defined($filters) && !defined($query);

    my @filters;

    if (defined $filters) {
        croak 'Incorrect filters' if ref $filters ne 'ARRAY';
        croak 'Specified 0 filters' if @{$filters} == 0;

        @filters = map {(filter => $_)} @{$filters};
    }

    my $page = delete $params{page};
    $page = 1 if not defined $page;

    my $per_page = delete $params{per_page};
    $per_page = 50 if not defined $per_page;

    my $fields = delete $params{fields};

    croak "Got unknown params: " . join(', ', keys %params) if %params;

    my $uri = URI->new('/v2/issues/');

    $uri->query_form(
        @filters,
        (defined($query) ? (query => $query) : ()),
        page    => $page,
        perPage => $per_page,
        (defined($fields) ? (fields => $fields) : ()),
    );

    my ($json, $status, $response_str) = $self->_get_api_response(
        method => 'GET',
        url    => $uri->as_string(),
    );
    croak "Got unexpected status $status. Full content: $response_str" unless $json;

    return $json;
}

=head1 search_all

То же самое что и search(), но только возвращает все результаты без
паджинации.

=cut

sub search_all {
    my ($self, %params) = @_;

    my $query   = delete $params{query};
    my $filters = delete $params{filters};
    my $fields  = delete $params{fields};

    croak "Got unknown params: " . join(', ', keys %params) if %params;

    my @results;

    my $page = 1;

    while (
        my @tmp_results = @{
            $self->search(
                query    => $query,
                filters  => $filters,
                per_page => 100,
                page     => $page,
                (defined($fields) ? (fields => $fields) : ()),
            )
        }
      )
    {
        push @results, @tmp_results;
        $page++;
    }

    return \@results;
}

=head1 versions

    $ys->versions();

Получение списка версий. Отдаются версии по всем проектам к которому есть
доступ.

Опционально можно передавать следующие параметры:

=over

=item page - страница данных. По умолчанию - 1

=item per_page - количество тикетов на странице (макс. 100), по-умолчанию - 50

=back

Возвращает ARRAYREF со списком версий.

L<https://wiki.yandex-team.ru/tracker/api/versions/list>

=cut

sub versions {
    my ($self, %params) = @_;

    my $page = delete $params{page};
    $page = 1 if not defined $page;

    my $per_page = delete $params{per_page};
    $per_page = 50 if not defined $per_page;

    croak "Got unknown params: " . join(', ', keys %params) if %params;

    my $uri = URI->new('/v2/versions/');

    $uri->query_form(
        page    => $page,
        perPage => $per_page,
    );

    my ($json, $status, $response_str) = $self->_get_api_response(
        method => 'GET',
        url    => $uri->as_string(),
    );
    croak "Got unexpected status $status. Full content: $response_str" unless $json;

    return $json;
}

=head1 versions_queue

    $ys->versions_queue( queue => 'PI' );

То же самое что и versions(), но отдается список версий только по указанной
очереди.

Параметр queue является обязательным.

Возвращает ARRAYREF со списком версий.

L<https://wiki.yandex-team.ru/tracker/api/queues/versions/>

=cut

sub versions_queue {
    my ($self, %params) = @_;

    my $queue = delete $params{queue};
    croak 'No queue' if not defined $queue;

    croak "Got unknown params: " . join(', ', keys %params) if %params;

    my ($json, $status, $response_str) = $self->_get_api_response(
        method => 'GET',
        url    => "/v2/queues/$queue/versions",
    );
    croak "Got unexpected status $status. Full content: $response_str" unless $json;

    return $json;
}

=head1 issuetypes

    $ys->issuetypes();

Возвращает ARRAYREF со списком типов тикетов.

L<https://wiki.yandex-team.ru/tracker/api/types/list>

=cut

sub issuetypes {
    my ($self, %params) = @_;

    croak "Got unknown params: " . join(', ', keys %params) if %params;

    my ($json, $status, $response_str) = $self->_get_api_response(
        method => 'GET',
        url    => '/v2/issuetypes/',
    );
    croak "Got unexpected status $status. Full content: $response_str" unless $json;

    return $json;
}

=head1 get_issue

    $ys->get_issue( $issue_key );

Возвращает структуру с данными тикета.

L<https://wiki.yandex-team.ru/tracker/api/issues/get>

=cut

sub get_issue {
    my ($self, $issue_key) = @_;

    croak "No issue key" if not defined $issue_key;
    croak "Incorrect issue key" if $issue_key !~ $ticket_re;

    my ($json, $status, $response_str) = $self->_get_api_response(
        method => 'GET',
        url    => "/v2/issues/$issue_key",
    );
    croak "Got unexpected status $status. Full content: $response_str" unless $json;

    return $json;
}

=head1 get_issue_links

    $ys->get_issue_links( $issue_key );

Возвращает структуру со связями тикета.

L<https://wiki.yandex-team.ru/tracker/api/issues/links/list>

=cut

sub get_issue_links {
    my ($self, $issue_key) = @_;

    croak "No issue key" if not defined $issue_key;
    croak "Incorrect issue key" if $issue_key !~ $ticket_re;

    my ($json, $status, $response_str) = $self->_get_api_response(
        method => 'GET',
        url    => "/v2/issues/$issue_key/links",
    );
    croak "Got unexpected status $status. Full content: $response_str" unless $json;

    return $json;
}

=head1 get_issue_remotelinks

    $ys->get_issue_remotelinks( $issue_key );

Возвращает структуру с внешними связями тикета.

L<https://wiki.yandex-team.ru/tracker/api/issues/remotelinks/list>

=cut

sub get_issue_remotelinks {
    my ($self, $issue_key) = @_;

    croak "No issue key" if not defined $issue_key;
    croak "Incorrect issue key" if $issue_key !~ $ticket_re;

    my ($json, $status, $response_str) = $self->_get_api_response(
        method => 'GET',
        url    => "/v2/issues/$issue_key/remotelinks",
    );
    croak "Got unexpected status $status. Full content: $response_str" unless $json;

    return $json;
}

=head1 get_board_sprints

    $ys->get_board_sprints( $board_id );

Возвращает даты начала и конца всех всех спринтов

L<https://wiki.yandex-team.ru/tracker/api/boards/sprints/list/>

=cut

sub get_board_sprints {
    my ($self, $board_id) = @_;

    croak "No board_id" if not defined $board_id;
    croak "Incorrect board_id" if $board_id !~ /^\d+$/;

    my ($json, $status, $response_str) = $self->_get_api_response(
        method => 'GET',
        url    => "/v2/boards/$board_id/sprints",
    );
    croak "Got unexpected status $status. Full content: $response_str" unless $json;

    return $json;
}

=head1 edit_ticket

    $ys->edit_ticket( $ticket_id );

Редактирование тикета

Значение $issue_key является обязательным.

Метод ничего не возвращает.

В случае каких-либо проблем метод умирает.

L<https://wiki.yandex-team.ru/tracker/api/issues/update/

=cut

sub edit_ticket {
    my ($self, $issue_key, $update_data_ref) = @_;

    croak "No issue_key" if not defined $issue_key;
    croak "Incorrect issue key" if $issue_key !~ m/^\w+-\d+$/;

    my ($json, $status, $response_str) = $self->_get_api_response(
        method       => 'PATCH',
        url          => "/v2/issues/$issue_key",
        content_data => ($update_data_ref ? $update_data_ref : {}),
    );
    croak "Got unexpected status $status. Full content: $response_str" unless $json;

    return 0;
}

=head1 make_transition

    $ys->make_transition( $issue_key, $transition, $resolution, $comment );

Выполненяет шаг воркфлоу тикета

Значения $issue_key, $transition и $resolution являются обязательными.

Метод ничего не возвращает.

В случае каких-либо проблем метод умирает.

L<https://wiki.yandex-team.ru/tracker/api/issues/transitions-execute/

=cut

sub make_transition {
    my ($self, $issue_key, $transition, $resolution, $comment) = @_;

    croak "No issue_key" if not defined $issue_key;
    croak "Incorrect issue key" if $issue_key !~ $ticket_re;

    croak "No transition" if not defined $transition;
    croak "No resolution" if not defined $resolution;

    my ($json, $status, $response_str) = $self->_get_api_response(
        method       => 'POST',
        url          => "/v2/issues/$issue_key/transitions/$transition/_execute",
        content_data => {
            resolution => $resolution,
            ($comment ? (comment => $comment) : ()),
        }
    );

    croak "Got unexpected status $status. Full content: $response_str" unless $json;

    return 0;
}

sub _get_api_response {
    my ($self, %opts) = @_;

    my $request = HTTP::Request->new(
        $opts{method} => $self->_get_base_api_host() . $opts{url},
        [
            'Authorization' => 'OAuth ' . $self->{_oauth_token},
            'Content-Type'  => 'application/json',
        ],
        $opts{content_data} ? encode_json($opts{content_data}) : ()
    );

    my $response;
    while (1) {
        $response = $self->{_lwp}->request($request);

        last if $response->is_success || --$self->{_retry} == 0;

        sleep($self->{_delay});
        warn
          sprintf("Retrying %s %s, previous request returned %s\n", $opts{method}, $opts{url}, $response->status_line);
    }

    my $content = $response->decoded_content // 'No content';

    my $json;
    if ($response->is_success) {
        $json = eval {decode_json($content)};
        warn $@ if $@;
    }

    utf8::decode($content) unless utf8::is_utf8($content);

    return ($json, $response->code, $content);
}

sub _get_base_api_host {
    my ($self) = @_;

    return 'https://st-api.yandex-team.ru';
}

1;
