#ABSTRACT: Pragma qbit

=head1 Name

qbit - Setup envirement to development modern perl applications and add some functions.

=head1 Description

Using this pragma is equivalent:

 use strict;
 use warnings FATAL => 'all';
 use utf8;
 use open qw(:std utf8);

 use Scalar::Util qw(set_prototype blessed dualvar isweak readonly refaddr reftype tainted weaken isvstring looks_like_number);
 use Data::Dumper qw(Dumper);
 use Clone        qw(clone);

 use QBit::Exceptions;
 use QBit::Log;
 use QBit::Array;
 use QBit::Hash;
 use QBit::GetText;
 use QBit::Packages;
 use QBit::StringUtils;
 use QBit::Date;
 use QBit::File;
 use QBit::System;

=head1 Synopsis

 use qbit;

 sub myfunc {
     my ($a1, $a2) = @_;

     throw Exception::Validation::BadArguments gettext('First argument must be defined')
         unless defined($a1);

     return ....
 }

 try {
    my $data = myfunc(@ARGV);
    ldump($data);
 } catch Exception::Validation::BadArguments with {
     l shift->as_string();
 };

=head1 Internal packages

=over

=item B<L<QBit::Exceptions>> - realize base classes and functions to use exception in perl;

=item B<L<QBit::Log>> - there're some function to simple logging;

=item B<L<QBit::Array>> - there're some function to working with arrays;

=item B<L<QBit::Hash>> - there're some function to working with hashes;

=item B<L<QBit::GetText>> - there're some function to internationalization your's software;

=item B<L<QBit::Packages>> - there're some function to access package internals;

=item B<L<QBit::StringUtils>> - there're some function to working with strings;

=item B<L<QBit::Date>> - there're some function to working with dates;

=item B<L<QBit::File>> - there're some function to manage files;

=item B<L<QBit::System>> - there're some function to working with system;

=back

=cut

package qbit;

use strict;
use warnings FATAL => 'all';
use utf8;
use open();
use Scalar::Util ();
use Data::Dumper ();
use Clone        ();

use QBit::Exceptions    ();
use QBit::Log           ();
use QBit::Array         ();
use QBit::Hash          ();
use QBit::GetText       ();
use QBit::Packages      ();
use QBit::StringUtils   ();
use QBit::Date          ();
use QBit::File          ();
use QBit::System        ();
use QBit::Role::Consume ();

sub import {
    $^H |= $utf8::hint_bits;
    $^H |= 0x00000002 | 0x00000200 | 0x00000400;

    ${^WARNING_BITS} |= $warnings::Bits{'all'};
    ${^WARNING_BITS} |= $warnings::DeadBits{'all'};

    my $pkg         = caller;
    my $pkg_sym_tbl = QBit::Packages::package_sym_table($pkg);

    {
        no strict 'refs';
        *{"${pkg}::TRUE"}  = sub () {1};
        *{"${pkg}::FALSE"} = sub () {''};
    }

    Scalar::Util->export_to_level(
        1, undef,
        @{
            QBit::Array::arrays_difference(
                [
                    qw(set_prototype blessed dualvar isweak readonly refaddr reftype tainted weaken isvstring looks_like_number)
                ],
                [keys(%$pkg_sym_tbl)]
            )
          }
    );    # Don't export functions, if they were imported before

    Data::Dumper->export_to_level(1, qw(Dumper));
    $Data::Dumper::Sortkeys = 1;

    Clone->export_to_level(1, undef, qw(clone));

    QBit::Exceptions->export_to_level(1);
    QBit::Log->export_to_level(1);
    QBit::Array->export_to_level(1);
    QBit::Hash->export_to_level(1);
    QBit::Packages->export_to_level(1);
    QBit::GetText->export_to_level(1);
    QBit::StringUtils->export_to_level(1);
    QBit::Date->export_to_level(1);
    QBit::File->export_to_level(1);
    QBit::System->export_to_level(1);
    QBit::Role::Consume->export_to_level(1);

    @_ = qw(open :std :utf8);
    goto &open::import;
}

1;
