#!/usr/bin/perl

use File::Slurp qw(read_file);

use lib::abs qw(
  ../../lib
  );

use qbit;
use Utils::ScriptWrapper;

run(
    sub {
        my ($app, $opts) = @_;

        my @reports_to_keep =
          map {0 + $_}
          grep {$_ =~ /^\d+$/} @{from_json(read_file('migrations/after_release/reports_with_visits.json'))};

        my $mol_fields        = _get_mol_fields($app);
        my $mol_entity_fields = _get_mol_entity_fields($app);

        my %block_model = map {$_ => 1} @{$app->product_manager->get_block_model_accessors};
        my %page_model  = map {$_ => 1} @{$app->product_manager->get_page_model_accessors};

        #select json_widgets_settings->'$[*].settings.report_id' as aa from widgets limit 100
        my @widget_report_id_list;
        my $use_as_widget_list = $app->partner_db->widgets->get_all(fields => [qw(json_widgets_settings)]);
        for my $widget (@$use_as_widget_list) {
            if (defined $widget->{json_widgets_settings}) {
                push @widget_report_id_list,
                  (
                    map {$_->{settings}{report_id}}
                      grep {defined $_->{settings}{report_id} && $_->{settings}{report_id} =~ /^\d+$/}
                      @{from_json($widget->{json_widgets_settings})}
                  );
            }
        }

        my $user_report_list = $app->partner_db->statistics_reports->get_all(
            fields => [qw(id report_id owner_id level query)],
            filter => [
                'OR' => [
                    # reports with more than 1 visit
                    [
                        'AND',
                        [['id', 'IN', \\@reports_to_keep], ['report_type', '=', \'pi'], ['owner_id', 'IS NOT', \undef]]
                    ],
                    # reports with widgets on dashboard
                    [
                        'AND',
                        [
                            ['id',          'IN',     \\@widget_report_id_list],
                            ['report_type', '=',      \'pi'],
                            ['owner_id',    'IS NOT', \undef]
                        ]
                    ],
                ]
            ],
        );

        my %unsupported_level = (
            'dsp'                    => 1,
            'mobile_mediation_block' => 1,
        );
        my %unmapped_fields = (
            fields        => {},
            entity_fields => {},
        );

        for my $report (@$user_report_list) {
            my $can_convert = 1;
            print logstr('~REPORT_ID=' . $report->{id});

            my $level          = $report->{level};
            my $level_accessor = ('payment' eq $level || 'charging' eq $level) ? undef : "statistics_$level";
            my $product_level  = $level_accessor ? $app->$level_accessor->product->accessor : $level;

            if (exists $unsupported_level{$level}) {
                $can_convert = 0;
                next;
            } else {
                my $query = from_json($report->{query});
                for my $chart_table (($query->{tableLevel}[0], $query->{chartLevel}[0])) {
                    next unless $chart_table;

                    my $level_filter = _get_level_filter($product_level, \%block_model, \%page_model);
                    unless ($level_filter) {
                        print logstr('WARN: NO_LEVEL_FILTER: id='
                              . $report->{id}
                              . ' level='
                              . $level
                              . ' product='
                              . $product_level);
                    } else {
                        if ($chart_table->{levels}[0]{filter}[0]) {
                            push @{$chart_table->{levels}[0]{filter}[1]}, $level_filter;
                            $chart_table->{levels}[0]{id} = 'payment';
                        } else {
                            $chart_table->{levels}[0]{filter} = ["AND", [$level_filter]];
                            $chart_table->{levels}[0]{id} = 'payment';
                        }
                    }

                    if ($can_convert) {

                        my @from_dimension_fields = grep {$_ !~ /^date\|/} @{$chart_table->{dimension_fields}};

                        my @fields_converted;
                        for my $field (@{$chart_table->{fields}}) {
                            my $converted_field =
                              _guess_mol_field($mol_fields, $field, $product_level, \%block_model, \%page_model);
                            if ($converted_field) {
                                if ('SKIP' ne $converted_field) {
                                    print logstr(
                                        'CONVERTED: ' . $field . '->' . $converted_field . ' lEVEL: ' . $level);
                                    push @fields_converted, $converted_field;
                                } else {
                                    print logstr('SKIPPED: ' . $field . '->' . $converted_field . ' lEVEL: ' . $level);
                                }
                            } else {
                                $unmapped_fields{fields}{$field}++;
                                print logstr('NOT_CONVERTED: ' . $field . ' LEVEL: ' . $level);
                                $can_convert = 0;
                            }
                        }
                        unless (@fields_converted) {
                            print logstr('WARN: NO_COUNTABLE_FIELDS');
                            $can_convert = 0;
                        }

                        my @entity_fields_converted;
                        for my $entity_field ((@{$chart_table->{entity_fields}}, @from_dimension_fields)) {
                            my $converted_field =
                              _guess_mol_field($mol_entity_fields, $entity_field, $product_level, \%block_model,
                                \%page_model);
                            if ($converted_field) {
                                if ('SKIP' ne $converted_field) {
                                    print logstr(
                                        'CONVERTED: ' . $entity_field . '->' . $converted_field . ' lEVEL: ' . $level);
                                    push @entity_fields_converted, $converted_field;
                                } else {
                                    print logstr(
                                        'SKIPPED: ' . $entity_field . '->' . $converted_field . ' lEVEL: ' . $level);
                                }
                            } else {
                                $unmapped_fields{entity_fields}{$entity_field}++;
                                print logstr('NOT_CONVERTED: ' . $entity_field . ' LEVEL: ' . $level);
                                $can_convert = 0;
                            }
                        }
                        unless (@entity_fields_converted) {
                            print logstr('WARN: NO_FIELDS');
                        }
                    }
                }
            }

            if ($can_convert) {
                print logstr('OK_REPORT_ID=' . $report->{id});
            } else {
                print logstr('FAIL_REPORT_ID=' . $report->{id});
            }
        }

        print logstr [sort keys %{$unmapped_fields{fields}}];
        print logstr [sort keys %{$unmapped_fields{entity_fields}}];
    }
   );

sub _get_level_filter {
    my ($level, $block_model, $page_model) = @_;

    return unless $level;

    if (exists $block_model->{$level}) {
        return ['block_level', '=', $level];
    }
    if (exists $page_model->{$level}) {
        return ['page_level', '=', $level];
    }
}

sub _guess_mol_field {
    my ($mol_fields, $field, $level, $block_model, $page_model) = @_;
    return unless $field && $level;

    $level =~ s/^(?:advnet_|internal_advnet_)(\w+)$/$1/;

    my $is_block = exists $block_model->{$level};
    my $is_page  = exists $page_model->{$level};

    my %h = (
        # entity_fields
        'campaign_id'        => 'page_id',
        'app_bundle_id'      => 'domain',
        'application_id'     => 'SKIP',
        'block_type_label'   => 'SKIP',
        'caption'            => $is_block ? 'block_caption' : ($is_page ? 'page_caption' : undef),
        'direct_block'       => 'SKIP',
        'id'                 => $is_block ? 'public_id' : ($is_page ? 'page_id' : undef),
        'media_formats_list' => 'SKIP',
        'name'               => 'SKIP',
        'owner_name'         => 'SKIP',
        'place_id'           => 'SKIP',
        'public_id'          => 'public_id',
        'seller_name'        => 'SKIP',
        'short_caption'      => 'SKIP',
        'store_id'           => 'domain',
        "type"               => 'SKIP',
        "type_label"         => 'SKIP',

        # countable_fields
        'direct_context_clicks'                                    => 'clicks_direct',
        'direct_context_cpm_direct_context_all_wo_nds'             => 'cpmv_all_wo_nds',
        'direct_context_cpm_direct_context_partner_wo_nds'         => 'cpmv_partner_wo_nds',
        'direct_context_cpmh_direct_context_partner_wo_nds'        => 'ecpm_partner_wo_nds',
        'direct_search_clicks'                                     => 'clicks_direct',
        'direct_search_cpm_direct_search_all_wo_nds'               => 'cpmv_all_wo_nds',
        'direct_search_cpm_direct_search_partner_wo_nds'           => 'cpmv_partner_wo_nds',
        'direct_search_cpmh_direct_search_partner_wo_nds'          => 'ecpm_partner_wo_nds',
        'direct_search_ctr'                                        => 'ctr_direct',
        'market_api_context_clicks'                                => 'clicks_direct',
        'market_api_context_cpm_market_api_context_partner_wo_nds' => 'cpmv_partner_wo_nds',
        'market_api_context_ctr'                                   => 'ctr_direct',
        'premium_clicks'                                           => 'clicks_direct',
        'premium_ctr'                                              => 'ctr_direct',
        'stripe_clicks'                                            => 'clicks_direct',
        'stripe_ctr'                                               => 'ctr_direct',

        'direct_context_cpc_direct_context_partner_wo_nds' => 'SKIP',
        'direct_context_ctr'                               => 'ctr_direct',
        'direct_search_cpc_direct_search_partner_wo_nds'   => 'SKIP',
        'dsp_bids'                                         => 'hits',
        'dsp_charging_wo_nds'                              => 'partner_wo_nds',
        'instream_block_ctr'                               => 'ctr_direct',
        'instream_block_midroll_conversion_percent'        => 'SKIP',
        'instream_block_open_player'                       => 'SKIP',
        'instream_block_pauseroll_conversion_percent'      => 'SKIP',
        'instream_block_postroll_conversion_percent'       => 'SKIP',
        'instream_block_preroll_conversion_percent'        => 'SKIP',
        'instream_block_view'                              => 'hits_render',
        'market_search_clicks'                             => 'clicks_direct',
        'mcb_search_clicks'                                => 'clicks_direct',
        'mcb_search_cpm_mcb_search_partner_wo_nds'         => 'cpmv_partner_wo_nds',
        'mcb_search_ctr'                                   => 'ctr_direct',
        'premium_cpc_partner_wo_nds'                       => 'SKIP',
        'stripe_cpc_partner_wo_nds'                        => 'SKIP',

        'an_campaign_caption'      => 'page_caption',
        'categories'               => 'SKIP',
        'context_campaign_caption' => 'page_caption',
        'currency_id'              => 'SKIP',
        'internal_mobile_caption'  => 'page_caption',
        'mobile_block_type_label'  => 'SKIP',
        'mobile_caption'           => 'page_caption',
        'search_campaign_caption'  => 'page_caption',
        'tag_caption'              => 'tag_id',
        'video_caption'            => 'page_caption',

        'direct_context_cpc_direct_context_all_wo_nds'             => 'SKIP',
        'direct_search_cpc_direct_search_all_wo_nds'               => 'SKIP',
        'market_api_context_cpc_market_api_context_partner_wo_nds' => 'SKIP',

    );

    if (exists $h{$field}) {
        return $h{$field};
    } elsif ($field =~ /_cover_ratio$/) {
        return 'fillrate';
    } elsif ($field =~ /_direct_clicks$/) {
        return 'clicks_direct';
    } elsif ($field =~ /_direct_ctr$/) {
        return 'ctr_direct';
    } elsif ($field =~ /_cpm_all_wo_nds$/) {
        return 'cpmv_partner_wo_nds';
    } elsif ($field =~ /_cpm_all_wo_nds$/) {
        return 'cpmv_all_wo_nds';
    } elsif ($field =~ /_cpmh_partner_wo_nds$/) {
        return 'ecpm_partner_wo_nds';
    } elsif ($field =~ /_cpmh_all_wo_nds$/) {
        return 'ecpm_all_wo_nds';
    } elsif ($field =~ /_cpm_partner_wo_nds$/) {
        return 'cpmv_partner_wo_nds';
    }

    #$level
    my $entity_fields_map = {

    };

    $level =~ s/^(?:advnet_|internal_advnet_)(\w+)$/$1/;
    $field =~ s/^(?:\w+_block_)(\w+)$/$1/;

    if (exists $mol_fields->{$field}) {
        return $field;
    } else {
        if (my @t = grep {$field =~ /$_/} sort {$b cmp $a} keys %$mol_fields) {
            unless ($field =~ /cpm|ctr|rpm|cpc/) {
                return $t[0];
            } else {
                if (my @tt = grep {$_ =~ /cpm|ctr|rpm|cpc/} @t) {
                    return $tt[0];
                }
            }
        }
    }
}

sub _get_mol_fields {
    my ($app) = @_;

    $app->set_cur_user({id => 763068973});
    return {map {$_->{id} => 1} @{$app->bk_statistics->_get_tree2__fields()}};
}

sub _get_mol_entity_fields {
    my ($app) = @_;

    $app->set_cur_user({id => 763068973});
    return {map {$_->{id} => 1} @{$app->bk_statistics->_get_tree2__entity_fields()},};
}
