#!/usr/bin/perl

use lib::abs qw(
  ../../lib
  );
use feature 'say';
use qbit;

use Application;
use File::Temp qw(tempfile);
use List::Util qw(min);

my %conflicts = (
    'amp,desktop,mobile' => sub {
            has_amp_name($_[0])    ? 'amp'
          : has_mobile_name($_[0]) ? 'mobile'
          :                          'desktop';
    },

    'desktop,mobile,turbo' => sub {
            has_turbo_name($_[0])  ? 'turbo'
          : has_mobile_name($_[0]) ? 'mobile'
          :                          'desktop';
    },

    'amp,desktop'    => sub {has_amp_name($_[0])    ? 'amp'    : 'desktop'},
    'amp,mobile'     => sub {has_amp_name($_[0])    ? 'amp'    : 'mobile'},
    'desktop,mobile' => sub {has_mobile_name($_[0]) ? 'mobile' : 'desktop'},
    'desktop,turbo'  => sub {has_turbo_name($_[0])  ? 'turbo'  : 'desktop'},
    'mobile,turbo'   => sub {has_turbo_name($_[0])  ? 'turbo'  : 'mobile'},
    'amp'            => sub {'amp'},
    'desktop'        => sub {'desktop'},
    'mobile'         => sub {'mobile'},
    'turbo'          => sub {'turbo'},

    # AMP and Turbo mixed up
    'amp,desktop,mobile,turbo' => sub {
            has_amp_and_turbo_name($_[0]) ? 'desktop'
          : has_amp_name($_[0])           ? 'amp'
          : has_turbo_name($_[0])         ? 'turbo'
          : has_mobile_name($_[0])        ? 'mobile'
          :                                 'desktop';
    },
    'amp,desktop,turbo' => sub {
            has_amp_and_turbo_name($_[0]) ? 'desktop'
          : has_amp_name($_[0])           ? 'amp'
          : has_turbo_name($_[0])         ? 'turbo'
          :                                 'desktop';
    },
    'amp,mobile,turbo' => sub {
            has_amp_and_turbo_name($_[0]) ? 'mobile'
          : has_amp_name($_[0])           ? 'amp'
          : has_turbo_name($_[0])         ? 'turbo'
          :                                 'mobile';
    },
    'amp,turbo' => sub {
            has_amp_and_turbo_name($_[0]) ? 'desktop'
          : has_amp_name($_[0])           ? 'amp'
          : has_turbo_name($_[0])         ? 'turbo'
          :                                 no_applicable_format($_[0]);
    },

    # No applicable format
    '' => sub {no_applicable_format($_[0])},
);

my $chunk = 1000;

main();

sub main {

    my $app = Application->new();
    $app->pre_run();
    $app->set_cur_user({id => 0});

    {
        no warnings 'redefine';
        *QBit::Application::check_rights = sub {1};
    }

    my ($blocks, $pages) = infer_versions($app);

    edit_pages($app, $pages);

    edit_blocks($app, $blocks);

    set_need_update($app, $pages, $blocks);

    say "END";
}

sub infer_versions {
    my ($app) = @_;

    my $i = 0;
    my $blocks;
    my %pages;
    my %blocks;

    my ($versions_log_fh, $versions_log_file_name) = tempfile('PI_9828_versions_log_XXXXXX', DIR => './');
    binmode($versions_log_fh, ':utf8');

    say 'START GET BLOCKS';

    do {
        $blocks = $app->context_on_site_rtb->get_all(
            fields => [
                qw(campaign_id id site_version caption domain direct_block limit is_custom_format_direct page is_custom_bk_data)
            ],
            offset => $i * $chunk,
            limit  => $chunk,
        );

        for my $b (@$blocks) {
            next if $b->{id} == 100500;

            my $types = applicable_types($b);
            my $version = $conflicts{$types}->($b, $types) // 'UNRESLOVED';

            if ($version eq 'UNRESOLVED') {
                say
                  "UNRESOLVED $b->{campaign_id}-$b->{id}, direct_block - $b->{direct_block}, caption - '$b->{caption}'";
            } elsif (
                grep {
                    $version eq $_
                } (qw(mobile turbo amp))
              )
            {
                $pages{$version}{$b->{page}{id}}{page_id}    = $b->{page}{page_id};
                $pages{$version}{$b->{page}{id}}{multistate} = $b->{page}{multistate};

                push @{$pages{$version}{$b->{page}{id}}{blocks}}, $b;
                push @{$blocks{$version}}, $b;

                $b->{site_version} = $version;
                $versions_log_fh->print(to_json($b) . "\n");
            }
        }

        $i += 1;

    } while (@$blocks > 0);

    close($versions_log_fh);

    return (\%blocks, \%pages);
}

sub edit_pages {
    my ($app, $pages) = @_;

    for my $version (qw(turbo amp)) {
        say "START EDIT $version pages";
        $app->partner_db->context_on_site_campaign->edit(
            $app->partner_db->filter([OR => [map {[page_id => '=' => \$_->{page_id}]} values(%{$pages->{$version}})]]),
            {"allowed_$version" => 1},
        ) if $version ne 'mobile';
    }
}

sub edit_blocks {
    my ($app, $blocks) = @_;

    for my $version (qw(mobile turbo amp)) {
        say "START EDIT $version blocks";
        my $i = 0;

        do {
            $app->partner_db->context_on_site_rtb->edit(
                $app->partner_db->filter(
                    [
                        OR => [
                            map {[AND => [[campaign_id => '=' => \$_->{campaign_id}], [id => '=' => \$_->{id}]]]}
                              @{$blocks->{$version}}
                              [($i * $chunk) .. min($#{$blocks->{$version}}, $i * $chunk + $chunk)]
                        ]
                    ]
                ),
                {site_version => $version, form_factor => undef, smart_media => undef},
            );

            $i += 1;
        } while ($i * $chunk < @{$blocks->{$version}});

        if (grep {$version eq $_} (qw(turbo amp))) {
            my $opposite_version = $version eq 'turbo' ? 'amp' : 'turbo';
            for my $block (@{$blocks->{$version}}) {
                if ($block->{is_custom_bk_data}) {
                    my $bk_data = from_json($app->context_on_site_rtb->get($block, fields => [qw(bk_data)])->{bk_data});
                    $bk_data->{PageImpOptions} = {
                        Enabled => [$version],
                        Disable => [$opposite_version],
                    };
                    $app->partner_db->context_on_site_rtb->edit(
                        {campaign_id => $block->{campaign_id}, id     => $block->{id}},
                        {bk_data     => to_json($bk_data,      pretty => 1)},
                    );
                }
            }
        }
    }
}

sub set_need_update {
    my ($app, $pages, $blocks) = @_;

    my ($set_need_update_log_fh, $set_need_update_log_file_name) =
      tempfile('PI_9828_set_need_update_log_XXXXXX', DIR => './');
    my ($failed_blocks_fh, $failed_blocks_file_name) = tempfile('PI_9828_failed_blocks_XXXXXX', DIR => './');
    binmode($set_need_update_log_fh, ':utf8');
    binmode($failed_blocks_fh,       ':utf8');

    for my $version (qw(mobile turbo amp)) {
        say "START SET_NEED_UPDATE $version pages";
        for my $id (keys(%{$pages->{$version}})) {
            next
              if $app->context_on_site_campaign->check_multistate_flag($pages->{$version}{$id}{multistate},
                'need_approve')
              || $app->context_on_site_campaign->check_multistate_flag($pages->{$version}{$id}{multistate}, 'rejected')
              || $app->context_on_site_campaign->check_multistate_flag($pages->{$version}{$id}{multistate}, 'deleted');

            my $status = 'OK';

            try {
                $app->context_on_site_campaign->do_action($id, 'set_need_update');
            }
            catch {
                my ($e) = @_;
                $status = $e->message;
            };

            $set_need_update_log_fh->print("$pages->{$version}{$id}{page_id} - $status\n");

            if ($status ne 'OK') {
                say "SET_NEED_UPDATE FAIL page_id - $pages->{$version}{$id}{page_id}";
                for my $block (@{$pages->{$version}{$id}{blocks}}) {
                    $failed_blocks_fh->print(
                        to_json(
                            {
                                page_id      => $block->{campaign_id},
                                id           => $block->{id},
                                site_version => $block->{site_version},
                            }
                          )
                          . "\n"
                    );
                }
            }
        }
    }
}

sub has_amp_name {
    my ($block) = @_;

    $block->{caption} =~ m/amp|AMP/;
}

sub has_turbo_name {
    my ($block) = @_;

    $block->{caption} =~ m/турбо|turbo/i;
}

sub has_mobile_name {
    my ($block) = @_;

    $block->{caption} =~ m/мобильн|mobile/i
      || $block->{domain} =~ m/^m\./i;
}

sub has_amp_and_turbo_name {
    my ($block) = @_;

    if (has_amp_name($block) && has_turbo_name($block)) {
        say
"UNRESOLVABLE $block->{campaign_id}-$block->{id}, direct_block - $block->{direct_block}, caption - '$block->{caption}'";
        return 1;
    } else {
        return 0;
    }
}

sub has_amp_format {
    my ($block) = @_;

    $block->{direct_block}      eq '300x250'
      || $block->{direct_block} eq '300x300'
      || $block->{direct_block} eq 'adaptive';
}

sub has_turbo_format {
    my ($block) = @_;

         $block->{direct_block} eq 'posterHorizontal' && !defined($block->{limit})
      || $block->{direct_block} eq 'horizontal'       && !defined($block->{limit})
      || $block->{direct_block} eq 'extensibleMobile'
      || $block->{direct_block} eq '300x250'
      || $block->{direct_block} eq '300x300';
}

sub has_mobile_format {
    my ($block) = @_;

         $block->{is_custom_format_direct}
      || $block->{direct_block} eq 'posterHorizontal' && !defined($block->{limit})
      || $block->{direct_block} eq 'posterVertical'   && $block->{limit} <= 2
      || $block->{direct_block} eq 'horizontal'       && $block->{limit} <= 2
      || $block->{direct_block} eq 'vertical'         && $block->{limit} <= 2
      || grep {$block->{direct_block} eq $_} (
        qw(
          240x400
          200x300
          250x250
          240x400
          300x300
          300x250
          320x50
          320x100
          grid
          adaptive
          modernAdaptive
          extensibleMobile
          )
      );
}

sub has_desktop_format {
    my ($block) = @_;

    $block->{is_custom_format_direct}
      || grep {$block->{direct_block} eq $_} (
        qw(
          160x600
          200x300
          250x250
          240x400
          300x600
          300x500
          300x300
          300x250
          336x280
          1000x120
          728x90
          970x250
          970x90
          grid
          vertical
          horizontal
          adaptive
          modernAdaptive
          posterVertical
          posterHorizontal
          )
      );
}

sub no_applicable_format {
    my ($block) = @_;

    say
"NO APPLICABLE FORMAT $block->{campaign_id}-$block->{id}, direct_block - $block->{direct_block}, caption - '$block->{caption}'";
    return 'desktop';
}

sub applicable_types {
    my ($block) = @_;

    join(
        ',',
        (
            (has_amp_format($block)     ? ('amp')     : ()),
            (has_desktop_format($block) ? ('desktop') : ()),
            (has_mobile_format($block)  ? ('mobile')  : ()),
            (has_turbo_format($block)   ? ('turbo')   : ()),
        )
    );
}
