#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

  Скрипт копирует блоки под новые категории включая найтроки dsp и brands

  NOTE! Нужно только подключить прод базу

=head1 USAGE

    LAZY_LOAD=1 FORCE_LOGGER_TO_SCREEN=1 script.pl  --page_ids=172307,22  --debug

=head1 OPTIONS

  nodebug   - запустить копирование с подменой id (по дефольу копирует в те же строки)
  page_ids  - Список ID площадок через запятую (для дебага)


=cut

use feature 'say';
use lib::abs qw( ../../lib );

use File::Slurp qw( read_file  write_file );
use Pod::Usage;
use Getopt::Long qw();
use Term::ANSIColor qw(colored);

use qbit;
use Application;
use Utils::Logger qw( WARN WARNF );

my $CATEGORY_TRANSITION = {
    1011 => 1018,    # VH: Главная  =>  VH: Главная SmartTV
    1013 => 1019     # VH: Я.Видео  =>  VH: Я.Видео SmartTV
};

my $VIDEO_BLOCK_ACCESSORS = [
    qw(
      video_an_site_instream
      video_an_site_inpage
      video_an_site_fullscreen
      )
];

my $res_file_path  = './PI-12829_res_data.json';
my $DEBUG          = 1;
my $DEBUG_PAGE_IDS = [261491, 235026];          # перввый 2 блока, второй один с годмодом
my $STASH          = {};

my $VIDEO_CATEGORIES_MIGRATIONS = {
    table  => 'video_an_site_categories',
    filter => sub {
        return ['id', 'IN', \[keys %$CATEGORY_TRANSITION]];
    },
    change_row => sub {
        my ($app, $row, $accessor_data) = @_;

        my $new_category_id = $CATEGORY_TRANSITION->{$row->{id}};
        my $name            = $STASH->{category_names}->{$new_category_id};

        # WARNF "\t\t\t\t category_id: %s => %s", $row->{id}, $new_category_id;
        # WARNF "\t\t\t\t name_ru: %s => %s", $row->{name}  // '', $name->{ru} // '';

        unless ($DEBUG) {
            $row->{name} = $name;
            $row->{id}   = $new_category_id;
        }
    },
};

my $VIDEO_BLOCK_MIGRATION = {
    filter => sub {
        my ($app, $accessor_data) = @_;

        my $filters = [
            ['category_id', 'IN',     \[keys %$CATEGORY_TRANSITION]],
            ['multistate',  'NOT IN', \$accessor_data->{deleted_bits}]
        ];

        if ($DEBUG_PAGE_IDS && @$DEBUG_PAGE_IDS) {
            push @$filters, ['page_id', 'IN', \$DEBUG_PAGE_IDS];
        }

        return ['AND', $filters];
    },
    change_row => sub {
        my ($app, $row, $accessor_data) = @_;

        my $accessor = $STASH->{cur_video_block_accessor};

        my $new_block_id = $accessor_data->{pages}->{$row->{page_id}}->{$row->{id}}
          // $app->video_an_site->get_next_block_id($row->{'page_id'});

        # WARNF "\t\t\t\t block id: %s => %s", $row->{id}, $new_block_id;

        $accessor_data->{pages}->{$row->{page_id}}->{$row->{id}} = $new_block_id;

        my $new_category_id = $CATEGORY_TRANSITION->{$row->{'category_id'}};

        unless ($DEBUG) {
            $row->{category_id}   = $new_category_id;
            $row->{category_path} = $STASH->{category_names}->{$new_category_id}->{ru};
            $row->{id}            = $new_block_id;

            if (exists($row->{'bk_data'}) && $row->{'bk_data'}) {
                my $public_id     = $app->$accessor->public_id({page_id => $row->{page_id}, id => $row->{id}});
                my $new_public_id = $app->$accessor->public_id({page_id => $row->{page_id}, id => $new_block_id});

                $row->{bk_data} =~ s/$public_id/$new_public_id/g;

                my $bk_data = from_json($row->{bk_data});
                $bk_data->{Video}->{CategoryID} = $new_category_id;

                $row->{bk_data} = to_json($bk_data, pretty => TRUE);
            }
        }
    },
};

my $CONTEXT_MIGRATIONS = [
    {
        table  => 'video_an_site_block_dsps',
        filter => sub {
            my ($app, $accessor_data) = @_;
            return [
                'AND',
                [
                    ['page_id',  '=',  \$STASH->{curr_page_id}],
                    ['block_id', 'IN', \[keys %{$accessor_data->{pages}->{$STASH->{curr_page_id}}}]]
                ]
            ];

        },
        change_row => sub {
            my ($app, $row, $accessor_data) = @_;
            unless ($DEBUG) {
                $row->{block_id} = $accessor_data->{pages}->{$row->{page_id}}->{$row->{block_id}};
            }
        },
    },
    {
        table  => 'brands',
        filter => sub {
            my ($app, $accessor_data) = @_;
            return [
                'AND',
                [
                    ['page_id',  '=',  \$STASH->{curr_page_id}],
                    ['block_id', 'IN', \[keys %{$accessor_data->{pages}->{$STASH->{curr_page_id}}}]]
                ]
            ];
        },
        change_row => sub {
            my ($app, $row, $accessor_data) = @_;
            unless ($DEBUG) {
                $row->{block_id} = $accessor_data->{pages}->{$row->{page_id}}->{$row->{block_id}};
            }
        },
    },
    {
        table  => 'video_an_site',
        filter => sub {
            return ['id', '=', \$STASH->{curr_page_id}];
        },
        change_row => sub {
            my ($app, $row, $accessor_data) = @_;
            unless ($DEBUG) {
                $row->{multistate} = (0 + $row->{'multistate'}) | $STASH->{need_update_bit};
            }
        },
    },
];

######
main();
######

sub main {

    ($DEBUG_PAGE_IDS) = _get_args();
    my $app = _get_app();
    $STASH = get_stash();

    WARN '!!DEBUG!!' if $DEBUG;

    try {
        copy_blocks($app);
    }
    catch {
        my ($e) = @_;
        save_stash($STASH);
        say $@;
        throw $e;
    };

    $app->post_run();
    save_stash($STASH);

    say "see file ", $res_file_path;
    say "#END";
}

sub copy_blocks {
    my ($app) = @_;

    my $categories     = $app->video_an_site_categories->get_default_video_categories();
    my $h              = {map {$_ => 1} %$CATEGORY_TRANSITION, values %$CATEGORY_TRANSITION};
    my $category_names = {map {$_->{id} => $_->{name}} grep {$h->{$_->{id}}} @$categories};

    my $INIT_STASH = {
        need_update_bit          => 0 + $app->video_an_site->get_multistate_by_name('need_update'),
        category_names           => $category_names,
        cur_video_block_accessor => '',
        curr_page_id             => '',
        accessors                => {},
    };

    $STASH = {%$INIT_STASH, %$STASH};

    $app->partner_db->transaction(
        sub {
            copy_tables($app, [$VIDEO_CATEGORIES_MIGRATIONS]);
        }
    );

    foreach my $accessor (@$VIDEO_BLOCK_ACCESSORS) {
        WARN "\nstart MODEL: $accessor";

        $STASH->{cur_video_block_accessor} = $accessor;
        $STASH->{accessors}->{$accessor} = {
            deleted_bits => $app->$accessor->get_multistates_by_filter('deleted'),
            pages        => $STASH->{accessors}->{$accessor}->{pages} // {}
        };

        $VIDEO_BLOCK_MIGRATION->{table} = $app->$accessor->db_table_name();

        $app->partner_db->transaction(
            sub {
                copy_tables($app, [$VIDEO_BLOCK_MIGRATION]);

                my @page_ids = sort keys %{$STASH->{accessors}->{$accessor}->{pages}};
                if ($DEBUG_PAGE_IDS && @$DEBUG_PAGE_IDS) {
                    my %h = map {$_ => 1} @$DEBUG_PAGE_IDS;
                    @page_ids = grep {$h{$_}} @page_ids;
                }

                my $count = 0;
                foreach my $page_id (@page_ids) {
                    WARNF "\tstart %d/%d. PAGE_ID: %s", ++$count, scalar(@page_ids), $page_id;
                    $STASH->{curr_page_id} = $page_id;
                    copy_tables($app, $CONTEXT_MIGRATIONS);
                    WARNF "\tend PAGE_ID: %s", $page_id;
                }
            }
        );

        WARNF "end MODEL: %s", $accessor;
    }
}

sub copy_tables {
    my ($app, $migrations) = @_;

    my $accessor      = $STASH->{cur_video_block_accessor};
    my $accessor_data = $STASH->{accessors}->{$accessor};

    foreach my $migr (@$migrations) {
        my $table = $migr->{'table'};

        my $new_table = $migr->{'new_table'} // $table;

        my $data = $app->partner_db->$table->get_all(
            fields => [map                      {$_->name} @{$app->partner_db->$table->fields}],
            filter => $migr->{'filter'}->($app, $accessor_data)
        );

        WARNF "\t\tstart TABLE: %s (rows %s)", $new_table, scalar(@$data);

        my $fields = [map {$_->name} @{$app->partner_db->$new_table->fields}];

        foreach my $row (@$data) {
            $migr->{'change_row'}->($app, $row, $accessor_data) if $migr->{'change_row'};
            $row = {hash_transform($row, $fields)};
        }

        if (@$data) {
            say colored("\t\t\t insert data: " . to_json($data), 'yellow') unless $ENV{'FORCE_LOGGER_TO_SCREEN'};
            $app->partner_db->$new_table->add_multi($data, 'duplicate_update' => 1);
        }

        WARN "\t\tend TABLE: $table";
    }
}

sub get_stash {
    my $content = eval {read_file($res_file_path, binmode => ':utf8')};
    my $data = eval {from_json($content)} // {};
    return $data;
}

sub save_stash {

    map {delete $STASH->{$_}} qw( category_names  cur_video_block_accessor  curr_page_id );
    foreach my $accessor (keys %{$STASH->{accessors}}) {
        map {delete $STASH->{accessors}->{$accessor}->{$_}} qw( need_update_bit  deleted_bits );
    }
    my $content = eval {to_json($STASH)};
    write_file($res_file_path, {binmode => ':utf8'}, \$content);
}

sub _get_app {

    my $app = Application->new();
    $app->pre_run();

    $app->set_cur_user({id => 0});

    no strict 'refs';
    no warnings 'redefine';
    *QBit::Application::check_rights = sub {1};

    return $app;
}

sub _get_args {

    my $page_id_str = '';
    my $help        = 0;

    Getopt::Long::GetOptions(
        #--- Obligatory
        'page_ids:s' => \$page_id_str,
        'debug!'     => \$DEBUG,
        #---
        'help|?|h' => \$help,
    ) or pod2usage(1);

    pod2usage(-verbose => 2, -noperldoc => 1) if $help;

    my $page_ids = [split /,/, $page_id_str];

    return $page_ids;
}
