#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

  Скрипт добавляет новую категорию 'VH: Я.Музыка', копирую 'VH: Я.Видео'

  NOTE! Нужно только подключить прод базу

=head1 USAGE

    LAZY_LOAD=1 FORCE_LOGGER_TO_SCREEN=1 script.pl --page_ids=172307,22 --debug

=head1 OPTIONS

  nodebug   - запустить копирование с подменой id (по дефольу копирует в те же строки)

=cut

use feature 'say';
use lib::abs qw( ../../lib );

use Pod::Usage;
use Getopt::Long qw();
use Term::ANSIColor qw(colored);

use qbit;
use Application;
use Utils::Logger qw( WARN WARNF );

my $DEBUG          = 1;
my $DEBUG_PAGE_IDS = [261491, 235026];
my $STASH          = {};

my $CATEGORY_TRANSITION = {
    1013 => 1020,    # VH: Я.Видео  =>  VH: Я.Музыка
};

my $VIDEO_CATEGORIES_MIGRATIONS = [
    {
        table  => 'video_an_site_categories',
        filter => sub {
            my $filter = ['id', 'IN', \[keys %$CATEGORY_TRANSITION]];
            if ($DEBUG_PAGE_IDS && @$DEBUG_PAGE_IDS) {
                $filter = ['AND', [$filter, ['page_id', 'IN', \$DEBUG_PAGE_IDS]]];
            }
            return $filter;
        },
        change_row => sub {
            my ($app, $row) = @_;

            my $new_category_id = $CATEGORY_TRANSITION->{$row->{id}};
            my $name            = $STASH->{category_names}->{$new_category_id};

            $STASH->{pages}->{$row->{page_id}} = 1;

            unless ($DEBUG) {
                $row->{name} = $name;
                $row->{id}   = $new_category_id;
            }
          }
    },
    {
        table  => 'video_an_site',
        filter => sub {
            return ['id', '=', \[keys %{$STASH->{pages}}]];
        },
        change_row => sub {
            my ($app, $row, $accessor_data) = @_;
            unless ($DEBUG) {
                $row->{multistate} = (0 + $row->{'multistate'}) | $STASH->{need_update_bit};
            }
        },
    },
];

######
main();
######

sub main {

    _get_args();
    my $app = _get_app();

    WARN '!!DEBUG!!' if $DEBUG;

    copy_blocks($app);

    $app->post_run();

    say "#END";
}

sub copy_blocks {
    my ($app) = @_;

    my $categories     = $app->video_an_site_categories->get_default_video_categories();
    my $h              = {map {$_ => 1} %$CATEGORY_TRANSITION, values %$CATEGORY_TRANSITION};
    my $category_names = {map {$_->{id} => $_->{name}} grep {$h->{$_->{id}}} @$categories};

    $STASH = {
        category_names  => $category_names,
        need_update_bit => 0 + $app->video_an_site->get_multistate_by_name('need_update'),
    };

    $app->partner_db->transaction(
        sub {
            copy_tables($app, $VIDEO_CATEGORIES_MIGRATIONS);
        }
    );
}

sub copy_tables {
    my ($app, $migrations) = @_;

    foreach my $migr (@$migrations) {
        my $table = $migr->{'table'};

        my $new_table = $migr->{'new_table'} // $table;

        my $data = $app->partner_db->$table->get_all(
            fields => [map {$_->name} @{$app->partner_db->$table->fields}],
            filter => $migr->{'filter'}->($app)
        );

        WARNF "\t\tstart TABLE: %s (rows %s)", $new_table, scalar(@$data);

        my $fields = [map {$_->name} @{$app->partner_db->$new_table->fields}];

        foreach my $row (@$data) {
            $migr->{'change_row'}->($app, $row) if $migr->{'change_row'};
            $row = {hash_transform($row, $fields)};
        }

        if (@$data) {
            say colored("\t\t\t insert data: " . to_json($data), 'yellow') unless $ENV{'FORCE_LOGGER_TO_SCREEN'};
            $app->partner_db->$new_table->add_multi($data, 'duplicate_update' => 1);
        }

        WARN "\t\tend TABLE: $table";
    }
}

sub _get_app {

    my $app = Application->new();
    $app->pre_run();

    $app->set_cur_user({id => 0});

    no strict 'refs';
    no warnings 'redefine';
    *QBit::Application::check_rights = sub {1};

    return $app;
}

sub _get_args {

    my $page_id_str = '';
    my $help        = 0;

    Getopt::Long::GetOptions(
        #--- Obligatory
        'page_ids:s' => \$page_id_str,
        'debug!'     => \$DEBUG,
        #---
        'help|?|h' => \$help,
    ) or pod2usage(1);

    pod2usage(-verbose => 2, -noperldoc => 1) if $help;

    $DEBUG_PAGE_IDS = [split /,/, $page_id_str];
    $DEBUG_PAGE_IDS = [] if !$page_id_str && !$DEBUG;

    1;
}
