#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

Скрипт выставляет значения по умолчанию для горизонтальных блоков

=head1 PARAMS

  commit - внести изменения
  count  - проверить отсутствие установленных значений и посчитать изменяемые записи
  show   - показать sql-запросы
  model  - список моделей для работы(по умолчанию все)

=head1 USAGE

  bin/oneshots/PI-13391_restore_horizontal_blocks.pl  --count
  bin/oneshots/PI-13391_restore_horizontal_blocks.pl  --commit

=cut

use lib::abs qw(
  ../../lib
  );

use qbit;
use Application;
use Data::Dumper;
use Getopt::Long;
use Pod::Usage;

my $COUNT;
my $SHOW;
my $COMMIT;
my $MODEL;

my %skip_custom_data = map {$_ => 1} qw(
  bg_color
  );

my %skip_custom_format = map {$_ => 1} qw(
  callouts
  );

my %defaults = (
    bg_color         => \'FFFFFF',
    favicon          => \0,
    border_color     => {if => [['border_type', '<>', \'none'], \'DDDCDA', \undef]},
    callouts         => \0,
    font_size        => \'1.1',
    horizontal_align => \0,
    links_underline  => \1,
    text_color       => \'000000',
);

my @models = qw(
  context_on_site_direct
  internal_context_on_site_direct
  context_on_site_rtb
  internal_context_on_site_rtb
  mobile_app_rtb
  internal_mobile_app_rtb
  internal_search_on_site_direct
  internal_search_on_site_premium
  );

sub main {

    my $app = _get_app();

    if ($MODEL) {
        my %m = map {$_ => 1} split(/\s*,\s*/, $MODEL);
        @models = grep {$m{$_}} @models;
        warn sprintf("ONLY: %s\n", join(", ", @models));
    }

    if ($SHOW or !$COMMIT) {
        no warnings 'redefine';
        my $do = QBit::Application::Model::DB::mysql->can('_do');
        *QBit::Application::Model::DB::mysql::_do = sub {
            warn $_[1], "\n" if $SHOW;
            unless ($COMMIT) {
                return 0;
            } else {
                return $do->(@_);
            }
        };
    }

    my $block_type    = 'horizontal';
    my $count_changed = 0;
    my $count_fail    = {};
    for my $accessor (sort @models) {
        my $model = $app->$accessor;

        my ($type, $fields, $data) = get_model_fields($model);

        my $filter_horizontal = [$type, '=', \$block_type];

        my ($c, $u);
        if ($COUNT) {
            ($c, my $f) = check_rows($model, $filter_horizontal, $fields);
            $count_fail->{$accessor} = $f if $f;
        } else {
            ($c) = make_changes($model, $filter_horizontal, $data);
        }
        $count_changed += $c;

        warn sprintf "%s changed=%d\n", $accessor, $c;
    }

    if ($COUNT) {
        unless (%$count_fail) {
            warn "check - ok\n";
        } else {
            while (my ($m, $c) = each %$count_fail) {
                warn "fail: $m - $c\n";
            }
        }
    }

    warn sprintf "TOTAL: %s%s changed=%d\n", $COUNT ? "COUNT" : "MAKE", $COMMIT ? "" : "(DEBUG)", $count_changed;
}

sub get_model_fields {
    my ($model) = @_;
    require Role::Tiny;

    my $mf = $model->get_model_fields;

    my $has_custom_data   = exists $mf->{is_custom_bk_data};
    my $has_custom_format = exists $mf->{is_custom_format_direct};

    my $type =
      Role::Tiny::does_role($model, 'Application::Model::Block::MIXIN::RTB')
      ? "direct_block"
      : "type";

    my @fields = grep {exists $mf->{$_}} keys %defaults;
    my %data;
    for my $field (@fields) {
        my $value = $defaults{$field};
        if ($has_custom_data and $skip_custom_data{$field}) {
            $data{$field} = {if => [['is_custom_bk_data', '<>', \1], $value, \undef]};
        } elsif ($has_custom_format and $skip_custom_format{$field}) {
            $data{$field} = {if => [['is_custom_format_direct', '<>', \1], $value, \undef]};
        } else {
            $data{$field} = $value;
        }
    }

    return ($type, \@fields, \%data);
}

sub make_changes {
    my ($model, $filter_horizontal, $data) = @_;

    my $filter_edit     = $model->partner_db->filter($filter_horizontal);
    my $need_update_can = $model->get_multistate_by_action('set_need_update');
    my $filter_update =
      $model->partner_db->filter($filter_horizontal)->and(['multistate' => 'IN' => \$need_update_can]);

    my $pk         = $model->get_pk_fields;
    my $page_field = $model->get_page_id_field_name;
    my $rows       = query($model, [@$pk, $page_field], $filter_update);

    my ($changed) = $model->partner_db_table->edit($filter_edit, $data);

    my $page_ids = array_uniq map {$_->{$page_field}} @$rows;
    $model->app->all_pages->mark_pages_for_async_update(page_ids => $page_ids);

    return ($changed);
}

sub query {
    my ($model, $fields, $filter) = @_;

    my $query = $model->partner_db->query->select(
        table  => $model->partner_db_table(),
        fields => $model->_get_fields_obj($fields)->get_db_fields(),
        filter => $filter,
    );
    $SHOW && warn(($query->get_sql_with_data)[0] . "\n");
    return $query->get_all;
}

sub check_rows {
    my ($model, $filter_horizontal, $fields) = @_;

    my $filter = $model->partner_db->filter($filter_horizontal);
    my $rows = query($model, $fields, $filter);

    my $f = 0;
    for my $row (@$rows) {
        $f++ if grep {!exists $row->{$_} or defined $row->{$_}} @$fields;
    }
    my $c = @$rows;

    return ($c, $f);
}

sub _get_app {
    _get_args();

    my $app = Application->new();
    $app->pre_run();

    $app->set_cur_user({id => 0, login => 'system-cron'});
    my $curuser = $app->get_option('cur_user');

    no strict 'refs';
    no warnings 'redefine';
    *{'QBit::Application::check_rights'} = sub {TRUE};

    return $app;
}

sub _get_args {
    my $help;
    Getopt::Long::GetOptions(
        'count!'   => \$COUNT,
        'commit!'  => \$COMMIT,
        'show!'    => \$SHOW,
        'model:s'  => \$MODEL,
        'help|?|h' => \$help,
    ) or pod2usage(1);

    pod2usage(-verbose => 2, -noperldoc => 1) if $help;
}

main();
