#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION
    Миграция сравнивает articles в полях таблиц и из отдельной таблицы,
    выводит блоки, на которых articles отличаются,
    выводит sql, который удаляет и вставляет записи в таблицу articles, чтобы они соотвествовали articles из полей

=head1 USAGE
  ./migrations/after_release/PI-16001_compare_articles_after_release.pl
  ./migrations/after_release/PI-16001_compare_articles_after_release.pl --append_logs

=head1 OPTIONS
=cut

use lib::abs qw(
  ../../lib
  );
use qbit;
use Application;
use Utils::ScriptWrapper;

my $chunk_size = 5_000;

sub _sort_articles {
    return ( $a->{page_id} <=> $b->{page_id}
          || $a->{block_id} <=> $b->{block_id}
          || $a->{article_sid} <=> $b->{article_sid});
}

run(
    sub {
        my ($app, $opts) = @_;

        my @model_list =
          grep {$app->$_->DOES('Application::Model::Role::Has::Articles::FromField')}
          @{$app->product_manager->get_block_model_names()};

        # list of statements to exec to update articles after release if needed
        my @sql_list;

        for my $model (@model_list) {
            print logstr("starting [$model]");

            my $page_id_field = $app->$model->get_page_id_field_name;
            my @fields        = ('id', $page_id_field);
            my $is_opts_field = 'opts' eq $app->$model->storage_method_of_articles();
            push @fields, ($is_opts_field ? 'opts' : 'articles');
            my $filter =
              $is_opts_field
              ? [
                AND => [
                    {JSON_CONTAINS_PATH => ['opts', \'one', \'$.articles']},
                    [{JSON_UNQUOTE => [{JSON_EXTRACT => ['opts', \'$.articles']}]} => '<>' => \'null']
                ]
              ]
              : ['articles' => 'IS NOT' => \undef];

            my $chunk;
            my $i = 0;
            do {
                $chunk = $app->partner_db->$model->get_all(
                    fields => \@fields,
                    limit  => $chunk_size,
                    offset => $chunk_size * $i,
                    filter => $filter
                );

                print logstr("[$model] " . @$chunk . ' records to process') if @$chunk;

                for my $block (@$chunk) {
                    my $articles =
                      $is_opts_field
                      ? from_json($block->{opts})->{articles}
                      : from_json($block->{articles});

                    $articles = from_json($articles) unless 'ARRAY' eq ref($articles);

                    my $page_id  = $block->{$page_id_field};
                    my $block_id = $block->{id};

                    my @rows = map {
                        {page_id => $page_id, block_id => $block->{id}, article_sid => $_->{id}, cpm => 0 + $_->{cpm},}
                    } @{$articles};

                    my $rows_from_articles = $app->partner_db->articles->get_all(
                        filter => {
                            page_id  => $page_id,
                            block_id => $block->{id},
                        }
                    );

                    next if (0 == @$articles + @$rows_from_articles);

                    my $json_fields = to_json([sort _sort_articles @rows], canonical => 1);
                    my $json_table =
                      to_json([sort _sort_articles map {$_->{cpm} = 0 + $_->{cpm}; $_;} @$rows_from_articles],
                        canonical => 1);

                    if ((@rows != @$rows_from_articles) || ($json_fields ne $json_table)) {
                        my $sql =
                          sprintf("DELETE FROM articles WHERE page_id = %d AND block_id = %d;", $page_id, $block_id);
                        if (@$articles) {
                            $sql .= sprintf(
                                'INSERT IGNORE INTO articles (page_id, block_id, article_sid, cpm) VALUES %s;',
                                join(
                                    ',',
                                    map {
                                        sprintf('(%d, %d, %d, %d)',
                                            $_->{page_id}, $_->{block_id}, $_->{article_sid}, $_->{cpm})
                                      } @rows
                                    )
                            );
                        }

                        push @sql_list, $sql;

                        print logstr(
                            to_json(
                                {
                                    block => {
                                        page_id  => $page_id,
                                        block_id => $block->{id},
                                    },
                                    from_field => from_json($json_fields),
                                    from_table => from_json($json_table),
                                    sql        => $sql,
                                },
                                canonical => 1
                            )
                        );
                    }
                }

                $i += 1;

            } while (@$chunk);
        }

        print logstr(join("\n", @sql_list));

        print logstr('#END');
    }
   );
