#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

Скрипт правит блоки у которых разъехались direct_block и name

=head1 USAGE

  perl -I./lib ./bin/oneshots/PI-16607_fix_direct_block.pl
  perl -I./lib ./bin/oneshots/PI-16607_fix_direct_block.pl --page_ids=12312,534142 --dry_run

=head1 OPTIONS

  page_ids           - Список ID площадок через запятую (необязательный)
  check_after_update - Проверить блок после апдейта

=cut

use qbit;

use Utils::ScriptWrapper;

sub args {
    my ($opts) = @_;

    return ('page_ids:s' => \$opts->{'page_ids'}, 'check_after_update!' => \$opts->{'check_after_update'});
}

run(
    sub {
        my ($app, $opts) = @_;

        my $dry                = $opts->{'dry'};
        my $check_after_update = $opts->{'check_after_update'};

        my @page_ids = split(',', $opts->{'page_ids'} // '');

        my $rtb_blocks = get_blocks($app, \@page_ids);

        print logstr('COUNT:', scalar(@$rtb_blocks));

        my $model_fields = $app->context_on_site_rtb->get_model_fields();

        foreach my $public_id (@$rtb_blocks) {
            my $block = $app->context_on_site_rtb->get($public_id, fields => ['*']);

            print logstr('BLOCK:', $public_id, $block);

            my $design = $app->context_on_site_rtb->get_design_with_min_id($block->{'design_templates'});

            print logstr('DESIGN:', $public_id, $design);

            my $new_settings = $app->context_on_site_rtb->update_fields_by_design_template($block, $design);

            my $qv = $app->validator->check(
                data => $new_settings,
                app  => $app->context_on_site_rtb,
                $app->context_on_site_rtb->get_template(
                    'fix' => {
                        campaign_id => {type => 'int_un', optional => TRUE},
                        page_id     => {type => 'int_un', optional => TRUE}
                    }
                ),
            );

            if ($qv->has_errors) {
                print logstr('ERROR:', $public_id, $new_settings, [$qv->get_fields_with_error()]);

                next;
            }

            my $fields = $app->context_on_site_rtb->get_design_settings_fields();

            my $update_settings = {map {$_ => $new_settings->{$_}} keys(%$fields)};

            print logstr('UPDATE:', $public_id, $update_settings);

            next if $dry;

            $app->partner_db->context_on_site_rtb->edit(
                {campaign_id => $new_settings->{'page_id'}, id => $new_settings->{'id'}},
                $update_settings);

            if ($check_after_update) {
                $app->validator->check(
                    data => $app->context_on_site_rtb->get(
                        $public_id, fields => [grep {$model_fields->{$_}{'need_check'}} keys(%$model_fields)]
                    ),
                    app   => $app->context_on_site_rtb,
                    throw => TRUE,
                    $app->context_on_site_rtb->get_template(
                        'fix' => {
                            campaign_id => {type => 'int_un', optional => TRUE},
                            page_id     => {type => 'int_un', optional => TRUE}
                        }
                    ),
                );
            }
        }
    }
   );

sub get_blocks {
    my ($app, $page_ids) = @_;

    # select
    #     r.campaign_id, r.id, r.direct_block, j.opts->>"$.design_settings.name"
    # from
    #     context_on_site_rtb as r
    # join (
    #     select
    #         d.id, d.page_id, d.block_id, d.opts
    #     from
    #         design_templates as d
    #     join (
    #         select
    #             min(id) as min_id, page_id, block_id
    #         from
    #             design_templates
    #         where
    #             multistate = 0
    #         group by
    #             page_id, block_id
    #     ) as g
    #     on
    #         d.id = g.min_id
    # ) as j
    # on
    #     r.campaign_id = j.page_id and r.id = j.block_id
    # where
    #     r.direct_block <> j.opts->>"$.design_settings.name"

    my $design_templates = $app->partner_db->design_templates;

    my $q_with_min_id = $app->partner_db->query->select(
        table  => $design_templates,
        fields => {min_id => {min => ['id']}, page_id => '', block_id => ''},
        filter => ['AND', [['multistate', '=', \0], (@$page_ids ? ['page_id', 'IN', \$page_ids] : ())]],
    )->group_by(qw(page_id block_id));

    my $q_with_opts = $app->partner_db->query->select(
        table  => $design_templates,
        alias  => 'd',
        fields => [qw(id page_id block_id opts)],
    )->join(table => $q_with_min_id, alias => 'g', join_on => [{'id' => 'd'}, '=', {'min_id' => 'g'}]);

    my $query = $app->partner_db->query->select(
        table  => $app->partner_db->context_on_site_rtb,
        alias  => 'r',
        fields => {
            'campaign_id'  => {'campaign_id'  => 'r'},
            'id'           => {'id'           => 'r'},
            'direct_block' => {'direct_block' => 'r'},
            'name' => ['->>', [{'opts' => 'j'}, \'$.design_settings.name']]
        },
        filter => [{'direct_block' => 'r'}, '<>', ['->>', [{'opts' => 'j'}, \'$.design_settings.name']]],
      )->join(
        table => $q_with_opts,
        alias => 'j',
        join_on =>
          ['AND', [[{'campaign_id' => 'r'}, '=', {'page_id' => 'j'}], [{'id' => 'r'}, '=', {'block_id' => 'j'}],]]
      );

    return [map {$app->context_on_site_rtb->public_id($_)} @{$query->get_all()}];
}
