#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

Скрипт для переноса статистики нативных блоков, перемещенных из контентных в ClickHouse

=head1 USAGE

perl ./migrations/before_release/PI-17540_move_statistics_for_old_natural --stage=ts
perl ./migrations/before_release/PI-17540_move_statistics_for_old_natural --stage=ts --block_list=C-A-345678-9,C-I-123456-7
perl ./migrations/before_release/PI-17540_move_statistics_for_old_natural --stage=ts --rollback=old_log_file

=head1 OPTIONS

  stage - DB stage (ts / prod)

=cut

use qbit;

use Utils::ScriptWrapper;

sub args {
    my ($opts) = @_;

    return (
        'block_list:s' => \$opts->{block_list},
        'stage:s'      => \$opts->{stage},
        'rollback:s'   => \$opts->{rollback}
    );
}

my ($app, $opts);
run(
    sub {
        ($app, $opts) = @_;

        print "#START#\n";

        my $ch_conn = "clickhouse_mdb --server=$opts->{stage}";
        my ($block_list, $old_stat, $new_stat);

        if ($opts->{rollback}) {
            my $fh;
            if (open($fh, '<', $opts->{rollback})) {
                while (<$fh>) {
                    if (/Moved blocks:(.+)$/) {
                        $block_list = $1;

                        #Выбираем текущую статистику по блокам без product_id для последующего сравнения
                        $old_stat = get_stat($ch_conn, $block_list);
                        print logstr 'Old statistics:', $old_stat;
                    }
                    if (/Rollback SQL:\s+(.+)$/) {
                        my $execute = "echo \"$1\" | $ch_conn";
                        print logstr 'Executing', $execute;
                        `$execute`;
                    }
                }
                $new_stat = get_stat($ch_conn, $block_list);
                print logstr 'New statistics:', $new_stat;

                print logstr 'New and old statistics are ' . ($old_stat eq $new_stat ? '' : 'not ') . 'equal.';
                close $fh;
            } else {
                print logstr 'Can\'t open log-file for rollback: ', $!;
                return;
            }
        } else {

            if ($opts->{block_list}) {
                $block_list = "(" . $opts->{block_list} . ")";
                $block_list =~ s/,/),(/g;
                $block_list =~ s/C-(A|I)-//g;
                $block_list =~ s/-/,/g;
            } else {
                #Запрашиваем блоки, перенесенные в БД
                $block_list =
`echo "SELECT page_id, block_id from statistics where product_id='context_on_site_content' and dictGetString('blocks_dict', 'caption', tuple('context_on_site_natural',toUInt32(page_id),toUInt32(block_id))) != '' group by page_id, block_id FORMAT TabSeparated" | $ch_conn`;

                chomp($block_list);
                $block_list =~ s/^\n//;
                $block_list = join ',',
                  map {my @block = split "\t", $_; '(' . $block[0] . ',' . $block[1] . ')'} split "\n", $block_list;
            }
            print logstr 'Moved blocks:', $block_list;

            #Выбираем текущую статистику по блокам без product_id для последующего сравнения
            my $old_stat = get_stat($ch_conn, $block_list);
            print logstr 'Old statistics:', $old_stat;

            #Формируем данные для отката
            my $rollback_sql =
`echo "select concat('insert into statistics values(''',toString(dt),''',''',toString(product_id),''',',toString(page_id),',',toString(block_id),',',toString(dsp_id),',',toString(currency_id),',',toString(tag_id),',',toString(category_id),',',toString(monetizer_id),',',toString(shows),',',toString(clicks),',',toString(hits),',',toString(impressions),',',toString(direct_clicks),',',toString(direct_shows),',',toString(all_w_nds),',',toString(all_wo_nds),',',toString(partner_w_nds),',',toString(partner_wo_nds),',',toString(bad_shows),',',toString(bad_hits),',',toString(bad_win_price_w_nds),',',toString(bad_win_price_wo_nds),',',toString(bad_win_partner_price_w_nds),',',toString(bad_win_partner_price_wo_nds),',',toString(win_max_positions_count),',',toString(an_fraud_shows),',',toString(an_fraud_clicks),',',toString(an_fraud_hits),',',toString(an_cover_hits),',',toString(an_cover_direct_hits),',',toString(an_cover_market_hits),',',toString(an_cover_mcb_hits),',',toString(an_cover_senthits),',',toString(an_rtb_cover_hits),',',toString(an_rtb_cover_senthits),',',toString(an_rtb_cover_direct_hits),',',toString(direct_page_ad_shows),',',toString(premium_page_ad_shows),',',toString(direct_hits),',',toString(market_hits),',',toString(mcb_hits),',',toString(premium_hits),',',toString(view),',',toString(open_player),');') from statistics where (page_id,block_id) in ($block_list) and product_id = 'context_on_site_content' FORMAT TabSeparatedRaw" | clickhouse_mdb --server=$ch_conn`;
            chomp $rollback_sql;
            my @rollback_sql = split "\n", $rollback_sql;
            print logstr 'Rollback SQL:', $_ foreach @rollback_sql;
            $rollback_sql =
`echo "select concat('insert into statistics values(''',toString(dt),''',''context_on_site_natural'',',toString(page_id),',',toString(block_id),',',toString(dsp_id),',',toString(currency_id),',',toString(tag_id),',',toString(category_id),',',toString(monetizer_id),',',toString(-1*shows),',',toString(-1*clicks),',',toString(-1*hits),',',toString(-1*impressions),',',toString(-1*direct_clicks),',',toString(-1*direct_shows),',',toString(-1*all_w_nds),',',toString(-1*all_wo_nds),',',toString(-1*partner_w_nds),',',toString(-1*partner_wo_nds),',',toString(-1*bad_shows),',',toString(-1*bad_hits),',',toString(-1*bad_win_price_w_nds),',',toString(-1*bad_win_price_wo_nds),',',toString(-1*bad_win_partner_price_w_nds),',',toString(-1*bad_win_partner_price_wo_nds),',',toString(-1*win_max_positions_count),',',toString(-1*an_fraud_shows),',',toString(-1*an_fraud_clicks),',',toString(-1*an_fraud_hits),',',toString(-1*an_cover_hits),',',toString(-1*an_cover_direct_hits),',',toString(-1*an_cover_market_hits),',',toString(-1*an_cover_mcb_hits),',',toString(-1*an_cover_senthits),',',toString(-1*an_rtb_cover_hits),',',toString(-1*an_rtb_cover_senthits),',',toString(-1*an_rtb_cover_direct_hits),',',toString(-1*direct_page_ad_shows),',',toString(-1*premium_page_ad_shows),',',toString(-1*direct_hits),',',toString(-1*market_hits),',',toString(-1*mcb_hits),',',toString(-1*premium_hits),',',toString(-1*view),',',toString(-1*open_player),');') from statistics where (page_id,block_id) in ($block_list) and product_id = 'context_on_site_content' FORMAT TabSeparatedRaw" | clickhouse_mdb --server=$ch_conn`;
            chomp $rollback_sql;
            @rollback_sql = split "\n", $rollback_sql;
            print logstr 'Rollback SQL:', $_ foreach @rollback_sql;

            #Добавляем статистику для нативных блоков
            my $insert_stat =
`echo "insert into statistics select dt,'context_on_site_natural',page_id,block_id,dsp_id,currency_id,tag_id,category_id,monetizer_id, shows,clicks, hits,impressions,direct_clicks,direct_shows,all_w_nds,all_wo_nds,partner_w_nds,partner_wo_nds,bad_shows,bad_hits,bad_win_price_w_nds,bad_win_price_wo_nds,bad_win_partner_price_w_nds,bad_win_partner_price_wo_nds,win_max_positions_count,an_fraud_shows,an_fraud_clicks,an_fraud_hits,an_cover_hits,an_cover_direct_hits,an_cover_market_hits,an_cover_mcb_hits,an_cover_senthits,an_rtb_cover_hits,an_rtb_cover_senthits,an_rtb_cover_direct_hits,direct_page_ad_shows,premium_page_ad_shows,direct_hits,market_hits,mcb_hits,premium_hits,view,open_player from statistics where (page_id,block_id) in ($block_list) and product_id = 'context_on_site_content';" | $ch_conn`;
            print logstr 'Add statistics for natural blocks.', $insert_stat;
            #Вычитаем статистику для контентных блоков
            $insert_stat =
`echo "insert into statistics select dt,product_id,page_id,block_id,dsp_id,currency_id,tag_id,category_id,monetizer_id,-1*shows,-1*clicks,-1*hits,-1*impressions,-1*direct_clicks,-1*direct_shows,-1*all_w_nds,-1*all_wo_nds,-1*partner_w_nds,-1*partner_wo_nds,-1*bad_shows,-1*bad_hits,-1*bad_win_price_w_nds,-1*bad_win_price_wo_nds,-1*bad_win_partner_price_w_nds,-1*bad_win_partner_price_wo_nds,-1*win_max_positions_count,-1*an_fraud_shows,-1*an_fraud_clicks,-1*an_fraud_hits,-1*an_cover_hits,-1*an_cover_direct_hits,-1*an_cover_market_hits,-1*an_cover_mcb_hits,-1*an_cover_senthits,-1*an_rtb_cover_hits,-1*an_rtb_cover_senthits,-1*an_rtb_cover_direct_hits,-1*direct_page_ad_shows,-1*premium_page_ad_shows,-1*direct_hits,-1*market_hits,-1*mcb_hits,-1*premium_hits,-1*view,-1*open_player from statistics where (page_id,block_id) in ($block_list) and product_id = 'context_on_site_content';" | $ch_conn`;
            print logstr 'Remove statistics for content blocks.', $insert_stat;

            `echo "optimize table statistics" | $ch_conn`;

            #Выбираем текущую статистику по блокам без product_id для сравнения
            $new_stat = get_stat($ch_conn, $block_list);
            print logstr 'New statistics:', $new_stat;

            print logstr 'New and old statistics are ' . ($old_stat eq $new_stat ? '' : 'not ') . 'equal.';
        }
    }
   );

sub get_stat {
    my ($ch_conn, $block_list) = @_;
    my $stat =
`echo "select dt,page_id,block_id,dsp_id,currency_id,tag_id,category_id,monetizer_id,sum(shows),sum(clicks),sum(hits),sum(impressions),sum(direct_clicks),sum(direct_shows),sum(all_w_nds),sum(all_wo_nds),sum(partner_w_nds),sum(partner_wo_nds),sum(bad_shows),sum(bad_hits),sum(bad_win_price_w_nds),sum(bad_win_price_wo_nds),sum(bad_win_partner_price_w_nds),sum(bad_win_partner_price_wo_nds),sum(win_max_positions_count),sum(an_fraud_shows),sum(an_fraud_clicks),sum(an_fraud_hits),sum(an_cover_hits),sum(an_cover_direct_hits),sum(an_cover_market_hits),sum(an_cover_mcb_hits),sum(an_cover_senthits),sum(an_rtb_cover_hits),sum(an_rtb_cover_senthits),sum(an_rtb_cover_direct_hits),sum(direct_page_ad_shows),sum(premium_page_ad_shows),sum(direct_hits),sum(market_hits),sum(mcb_hits),sum(premium_hits),sum(view),sum(open_player) from statistics where (page_id,block_id) in ($block_list) group by dt,page_id,block_id,dsp_id,currency_id,tag_id,category_id,monetizer_id having sum(shows)<>0 or sum(clicks)<>0 or sum(hits)<>0 or sum(impressions)<>0 or sum(direct_clicks)<>0 or sum(direct_shows)<>0 or sum(all_w_nds)<>0 or sum(all_wo_nds)<>0 or sum(partner_w_nds)<>0 or sum(partner_wo_nds)<>0 or sum(bad_shows)<>0 or sum(bad_hits)<>0 or sum(bad_win_price_w_nds)<>0 or sum(bad_win_price_wo_nds)<>0 or sum(bad_win_partner_price_w_nds)<>0 or sum(bad_win_partner_price_wo_nds)<>0 or sum(win_max_positions_count)<>0 or sum(an_fraud_shows)<>0 or sum(an_fraud_clicks)<>0 or sum(an_fraud_hits)<>0 or sum(an_cover_hits)<>0 or sum(an_cover_direct_hits)<>0 or sum(an_cover_market_hits)<>0 or sum(an_cover_mcb_hits)<>0 or sum(an_cover_senthits)<>0 or sum(an_rtb_cover_hits)<>0 or sum(an_rtb_cover_senthits)<>0 or sum(an_rtb_cover_direct_hits)<>0 or sum(direct_page_ad_shows)<>0 or sum(premium_page_ad_shows)<>0 or sum(direct_hits)<>0 or sum(market_hits)<>0 or sum(mcb_hits)<>0 or sum(premium_hits)<>0 or sum(view)<>0 or sum(open_player)<>0 order by dt,page_id,block_id,dsp_id,currency_id,tag_id,category_id,monetizer_id FORMAT TabSeparated" | $ch_conn`;
    chomp($stat);
    return $stat;
}
