#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

Скрипт для раскладывания SSP продуктов в новые таблицы. Требует подключения ПРОДАКШЕН базы

=head1 USAGE

perl ./migrations/before_release/PI-10484_ssp.pl --rename_tables=0

perl ./migrations/before_release/PI-10484_ssp.pl --accessor=ssp_mobile_app --id=6

perl ./migrations/before_release/PI-10484_ssp.pl --accessor=ssp_context_rtb --page_id=23564

perl ./migrations/before_release/PI-10484_ssp.pl --diff_bk_data

=head1 OPTIONS

  rename_tables - делает rename таблиц (ssp_mobile_app_rtb / ssp_mobile_app_rtb_action_log)
  accessor      - Акксессор старой продуктовой модели (ssp_context_rtb / ssp_mobile_app / ssp_video_app)
  id            - ID площадки
  page_id       - Page ID площадки
  continue      - Продолжить с определенного id
  diff_bk_data  - приводит bk_data к одному виду и сравнивает

=cut

use lib::abs qw(../../lib/);

use feature say;

use Test::More;
use Test::Most;
use Test::Deep;

use qbit;
use Application;
use File::Temp qw(tempfile);
use Pod::Usage;
use Getopt::Long qw();

my $SCRIPT = lib::abs::path('../../bin/pt-online-schema-change.pl');

my $TABLES = [
    qw(
      ssp_context_on_site_campaign
      ssp_context_on_site_campaign_action_log
      ssp_context_on_site_rtb
      ssp_context_on_site_rtb_action_log
      ssp_context_on_site_rtb_dsp_blocks
      ssp_context_on_site_dsps

      ssp_mobile_app_settings
      ssp_mobile_app_settings_action_log
      ssp_mobile_app_rtb
      ssp_mobile_app_rtb_action_log
      ssp_mobile_app_rtb_dsp_blocks
      ssp_mobile_app_dsps

      ssp_video_an_site
      ssp_video_an_site_action_log
      ssp_video_an_site_rtb
      ssp_video_an_site_rtb_action_log
      ssp_video_an_site_rtb_dsp_blocks
      ssp_video_an_site_dsps

      ssp_context_on_site_campaign_mirrors
      ssp_mobile_app_settings_mirrors
      ssp_video_an_site_mirrors
      )
];

my %OLD_MODELS = (
    'ssp_context_rtb' => 'ssp_context_on_site_campaign',
    'ssp_mobile_app'  => 'ssp_mobile_app_settings',
    'ssp_video_app'   => 'ssp_video_an_site',
);

my %CHANGE_FIELDS = (
    pictures_enabled => 'view_images',
    data_brands      => 'brands',
);

my $CHANGE_FOREIGN_KEYS = {
    ssp_context_rtb => {
        table    => 'ssp_link_context_rtb',
        fk_field => 'context_rtb_id',
        sql =>
'DROP FOREIGN KEY `_ssp_link_context_rtb_ibfk_2`, ADD FOREIGN KEY `fk_ssp_link_context_rtb__context_rtb_id___ssp_context_on_site` (`context_rtb_id`) REFERENCES `ssp_context_on_site_campaign` (`id`) ON UPDATE RESTRICT ON DELETE RESTRICT',
    },
    ssp_mobile_app => {
        table    => 'ssp_link_mobile_app',
        fk_field => 'mobile_app_id',
        sql =>
'DROP FOREIGN KEY `_ssp_link_mobile_app_ibfk_2`, ADD FOREIGN KEY `fk_ssp_link_mobile_app__mobile_app_id___ssp_mobile_app_settings` (`mobile_app_id`) REFERENCES `ssp_mobile_app_settings` (`id`) ON UPDATE RESTRICT ON DELETE RESTRICT',
    },
    ssp_video_app => {
        table    => 'ssp_link_video_app',
        fk_field => 'video_app_id',
        sql =>
'DROP FOREIGN KEY `_ssp_link_video_app_ibfk_2`, ADD FOREIGN KEY `fk_ssp_link_video_app__video_app_id___ssp_video_an_site__id` (`video_app_id`) REFERENCES `ssp_video_an_site` (`id`) ON UPDATE RESTRICT ON DELETE RESTRICT',
    },
};

main();

sub main {
    my ($rename_tables, $accessor, $id, $page_id, $continue, $diff_bk_data) = _get_args();

    my $app = Application->new();

    $app->pre_run();

    $app->set_option('cur_user' => {id => 0});

    my $tmp_rights = $app->add_all_tmp_rights();

    {
        no strict 'refs';
        no warnings 'redefine';
        *{'Application::Model::Page::SSP::on_action_register_in_balance'} = sub { };
    }

    $app->partner_db->_do(
        'update ssp_mobile_app set owner_id = 412873060, seller_id = 13921737 where id in (
  select t.id from (
    select
        ssp.id
    from
        ssp_mobile_app as ssp
    left join
        ssp_link_mobile_app as l on ssp.id = l.mobile_app_id
    where l.seller_id <> ssp.seller_id and l.seller_id = 13921737
  ) as t
);'
    );

    $app->partner_db->_do(
        'RENAME TABLE
    ssp_mobile_app_rtb TO ssp_mobile_app_rtb_old,
    ssp_mobile_app_rtb_action_log TO ssp_mobile_app_rtb_old_action_log'
    ) if $rename_tables;

    my @fields_with_tmpl = qw(tmpl_ssp_mobile_app tmpl_ssp_context_rtb tmpl_ssp_video_app);

    my $ssp_sellers = $app->partner_db->ssp_seller->get_all(fields => ['id', @fields_with_tmpl]);

    foreach my $ssp_seller (@$ssp_sellers) {
        my $id = delete($ssp_seller->{'id'});

        foreach my $field (@fields_with_tmpl) {
            if (defined($ssp_seller->{$field}) && $ssp_seller->{$field} ne '') {
                my $tmpl = from_json($ssp_seller->{$field});

                if ($field eq 'tmpl_ssp_mobile_app') {
                    $tmpl->{'mobile_app_mode'} //= 0;
                }

                foreach (keys(%CHANGE_FIELDS)) {
                    $tmpl->{$CHANGE_FIELDS{$_}} = delete($tmpl->{$_});
                }
            }
        }

        $app->partner_db->ssp_seller->edit($id, $ssp_seller);
    }

    foreach my $table (reverse(@$TABLES)) {
        $app->partner_db->_do("DROP TABLE IF EXISTS $table");
    }

    $app->partner_db->init_db($TABLES);

    foreach my $old_model (sort keys(%OLD_MODELS)) {
        my $new_model = $OLD_MODELS{$old_model};

        next if defined($accessor) && $accessor ne $old_model;

        say "Move: $old_model";

        my $offset = 0;
        my $limit  = 1000;

        my $filter;
        $filter->{'id'}      = $id      if defined($id);
        $filter->{'page_id'} = $page_id if defined($page_id);

        my @need_remove_app = ();
        my $data;
        while (
            $data = $app->$old_model->get_all(
                fields => ['*'],
                ($filter ? (filter => $filter) : ()),
                order_by => ['id'],
                offset   => $offset,
                limit    => $limit
            )
          )
        {
            last unless @$data;

            my $page_add_fields = [keys(%{$app->$new_model->get_add_fields()}), 'id', 'seller_id', 'page_id'];

            my $block_model = $app->$new_model->get_block_model_names();

            my $block_add_fields = [keys(%{$app->$block_model->get_add_fields()})];

            foreach my $row (@$data) {
                if ($continue && $row->{'id'} < $continue) {
                    next;
                }

                say "Start work: $row->{id}";

                delete($row->{'page'});

                unless (defined($row->{'page_id'})) {
                    say "Page id not defined: " . to_json($row);

                    push(@need_remove_app, $row->{'id'});

                    next;
                }

                $row->{'mincpm'} //= '0';

                #fix db data...
                foreach (qw(articles font_family font_size geo)) {
                    $row->{$_} = undef if $row->{$_} && $row->{$_} eq 'NULL';
                }

                throw 'Bad articles'
                  if (@{$row->{'data_articles'}} && !$row->{'articles'})
                  || (!@{$row->{'data_articles'}} && $row->{'articles'});

                my $page = {hash_transform($row, $page_add_fields)};

                $page->{'source_id'} =
                  exists($row->{'link_mobile_app'})
                  ? $row->{'link_mobile_app'}{'application'}{'id'}
                  : $row->{'link_context_rtb'}{'site'}{'id'};

                unless (defined($page->{'source_id'})) {
                    say "Source id not found: " . to_json($row);

                    next;
                }

                $app->$new_model->add(%$page);

                my $block = {hash_transform($row, $block_add_fields)};
                $block->{'campaign_id'} = $page->{'page_id'};
                $block->{'id'}          = 1;

                $app->$block_model->add(%$block);

                if ($row->{'multistate'} & 4) {
                    say "Stop: $page->{id}";
                    $app->$new_model->do_action($page->{'id'}, 'stop');
                }

                if ($diff_bk_data) {
                    my $old_bk_data = $app->$old_model->get_bk_data({id => $row->{'id'}});
                    my $new_bk_data =
                      $app->$new_model->_data_for_bk($app->$new_model->get($row->{'id'}, fields => ['*']));

                    $new_bk_data->{'product_id'} = $old_model;

                    #новые опции
                    foreach (qw(business_unit is_yandex_page page_caption page_options)) {
                        delete($new_bk_data->{$_});
                    }

                    #теперь заданы, а раньше были такими
                    if ($old_model eq 'ssp_context_rtb') {
                        $new_bk_data->{'excluded_domains'} = [];
                    } else {
                        my %h = map {$_ => 1} @{$new_bk_data->{'excluded_domains'}};
                        my @not_exists = grep {!$h{$_}} @{$old_bk_data->{'excluded_domains'}};

                        throw sprintf('not exists domains: %s', join(', ', @not_exists)) if @not_exists;

                        $new_bk_data->{'excluded_domains'} = [];
                        $old_bk_data->{'excluded_domains'} = [];
                    }

                    my $block_settings = $new_bk_data->{"rtb_blocks"}{"1"};

                    #новые опции для блока
                    foreach (qw(BlockModel)) {
                        delete($block_settings->{$_});
                    }

                    #теперь заданы, а раньше были такими
                    $block_settings->{'BlockCaption'} = '';

                    #нормализуем RtbDesign
                    my $new_json = '{' . $block_settings->{"RtbDesign"} . '}';
                    my $old_json = '{' . $old_bk_data->{"rtb_blocks"}{"1"}{"RtbDesign"} . '}';

                    foreach ($new_json, $old_json) {
                        my $p = from_json($_);

                        delete($p->{'horizontalAlign'});
                        $p->{'limit'} = 1;

                        my $json = to_json($p);
                        $json =~ /^\{|\}$/g;

                        $_ = $json;
                    }

                    $block_settings->{"RtbDesign"} = $new_json;
                    $old_bk_data->{"rtb_blocks"}{"1"}{"RtbDesign"} = $old_json;

                    #нормализация Sizes
                    $block_settings->{"Sizes"} = [
                        sort {$a->{'Width'} . $a->{'Height'} cmp $b->{'Width'} . $b->{'Height'}}
                        grep {!($_->{'Width'} == 0 && $_->{'Height'} == 0)} @{$block_settings->{"Sizes"}}
                    ];
                    $old_bk_data->{"rtb_blocks"}{"1"}{"Sizes"} =
                      [sort {$a->{'Width'} . $a->{'Height'} cmp $b->{'Width'} . $b->{'Height'}}
                          @{$old_bk_data->{"rtb_blocks"}{"1"}{"Sizes"}}];

                    foreach ($new_bk_data, $old_bk_data) {
                        delete($_->{"rtb_blocks"}{"1"}{'MultiState'});
                        $_->{"rtb_blocks"}{"1"}{'DirectLimit'} = 1;
                    }

                    cmp_deeply($new_bk_data, $old_bk_data, "diff: $row->{'id'}");
                }

                say "End work: $row->{id}";
            }

            last if $filter;

            $offset += $limit;
        }

        my $old_pages = $app->partner_db->_get_all("select count(*) as cnt from $old_model")->[0];
        my $new_pages = $app->partner_db->_get_all("select count(*) as cnt from $new_model")->[0];

        my $old_working_pages = $app->partner_db->_get_all(
            "select count(*) as cnt from $old_model where (multistate & 1) or (multistate & 2)")->[0];
        my $new_working_pages = $app->partner_db->_get_all(
            "select count(*) as cnt from $new_model where (multistate & 1) or (multistate & 2)")->[0];

        printf(
            "\n\n----\n\n%s:\nall pages: %d\nworking pages: %d\n\n%s:\nall pages: %d\nworking pages: %d\n\n",
            $old_model, $old_pages->{'cnt'}, $old_working_pages->{'cnt'},
            $new_model, $new_pages->{'cnt'}, $new_working_pages->{'cnt'}
        );

        my $alter_data = $CHANGE_FOREIGN_KEYS->{$old_model};

        my ($table, $fk_field) = @$alter_data{qw(table fk_field)};

        if (@need_remove_app) {
            my @ids =
              map {$_->{'id'}}
              @{$app->partner_db->$table->get_all(fields => [qw(id)], filter => {$fk_field => \@need_remove_app})};

            my $filter = {elem_id => \@ids};

            my $log_table = $table . '_action_log';

            say "delete from $log_table: " . to_json($filter);

            my $deleted_logs = $app->partner_db->$log_table->get_all(filter => $filter);

            say "deleted rows: " . to_json($deleted_logs);

            $app->partner_db->$log_table->delete($app->partner_db->filter($filter));

            $filter = {id => @ids};

            say "delete from $table: " . to_json($filter);

            my $deleted_links = $app->partner_db->$table->get_all(filter => $filter);

            say "deleted rows: " . to_json($deleted_links);

            $app->partner_db->$table->delete($filter);
        }

        system($SCRIPT, "--table=$table", "--alter=$alter_data->{sql}");

        $app->partner_db->_do("ANALYZE TABLE `partner`.`$table`;");
    }

    done_testing();

    say '#END';
}

sub _get_args {
    my ($accessor, $id, $page_id, $continue, $diff_bk_data);
    my $help          = 0;
    my $rename_tables = 1;

    Getopt::Long::GetOptions(
        #--- Obligatory
        'accessor=s'      => \$accessor,
        'id=i'            => \$id,
        'page_id=i'       => \$page_id,
        'continue=i'      => \$continue,
        'rename_tables=i' => \$rename_tables,
        'diff_bk_data'    => \$diff_bk_data,
        #---
        'help|?|h' => \$help,
    ) or pod2usage(1);

    pod2usage(-verbose => 2, -noperldoc => 1) if $help;

    return ($rename_tables, $accessor, $id, $page_id, $continue, $diff_bk_data);
}
