#!/usr/bin/perl

use feature 'say';

use lib::abs qw(../../lib);

use qbit;

use Application;

use PiConstants qw($TECHNICAL_RTB_BLOCK_ID);

my @fields = qw(
  campaign_id
  id
  direct_block
  is_custom_format_direct
  form_factor
  smart_media
  horizontal_align
  limit
  );

my $direct_formats_by_version = {};

my @mobile_attributes = qw(
  мобильный
  mobile
  мобильной
  touch
  );

my @mobile_pages = qw(
  148790
  151547
  139196
  80007
  88848
  143156
  247357
  247361
  247362
  248072
  248375
  146642
  249073
  249076
  249077
  261276
  204310
  227232
  227511
  236159
  236315
  236329
  236330
  241835
  244655
  141769
  137990
  239082
  113011
  242027
  121319
  242961
  242961
  84375
  84912
  209597
  129744
  217994
  249430
  245352
  249431
  185103
  153638
  155707
  258356
  213088
  );

main();

sub main {
    my $app = Application->new();
    $app->pre_run();

    foreach (qw(desktop mobile turbo)) {
        $direct_formats_by_version->{$_} =
          {map {$_->{'id'} => $_} @{$app->internal_context_on_site_rtb->get_types_direct_blocks({site_version => $_})}};
    }

    _set_mobile($app);

    _set_turbo($app);

    _set_suitable($app);

    $app->post_run();
}

sub _set_mobile {
    my ($app) = @_;

    say 'SET mobile';

    my @mobile_page_ids = map {$_->{'id'}} @{
        $app->partner_db->query->select(
            table  => $app->partner_db->internal_context_on_site_campaign,
            fields => [qw(id)],
            filter => [
                'OR',
                [
                    ['id', 'IN', \\@mobile_pages],
                    [
                        'site_id',
                        'IN',
                        $app->partner_db->query->select(
                            table  => $app->partner_db->internal_site,
                            fields => [qw(id)],
                            filter => ['domain', 'LIKE', \'m.%']
                        )
                    ]
                ]
            ]
          )->get_all()
      };

    my $blocks = $app->partner_db->internal_context_on_site_rtb->get_all(
        fields => \@fields,
        filter => [
            'AND',
            [
                ['id', '<>', \$TECHNICAL_RTB_BLOCK_ID],
                [
                    'OR',
                    [
                        (map {['caption', 'LIKE', \"%$_%"]} @mobile_attributes),
                        ['campaign_id', 'IN', \\@mobile_page_ids]
                    ]
                ]
            ]
        ]
    );

    foreach my $block (@$blocks) {
        my $msg = $app->internal_context_on_site_rtb->public_id($block);

        if (_can_set_version('mobile', $block)) {
            $msg .= _set_version($app, $block, 'mobile');
        } else {
            $msg .= ' conflict: ' . to_json($block->{'direct_block'});
        }

        say $msg;
    }

    say 'END mobile';
}

sub _set_turbo {
    my ($app) = @_;

    say 'SET turbo';

    my $blocks = $app->partner_db->internal_context_on_site_rtb->get_all(
        fields => \@fields,
        filter => ['AND', [['id', '<>', \$TECHNICAL_RTB_BLOCK_ID], ['campaign_id', 'IN', \[260855, 264925]]]]
    );

    foreach my $block (@$blocks) {
        my $msg = $app->internal_context_on_site_rtb->public_id($block);

        if (_can_set_version('turbo', $block)) {
            $msg .= _set_version($app, $block, 'turbo');
        } else {
            $msg .= ' conflict: ' . to_json($block->{'direct_block'});
        }

        say $msg;
    }

    say 'END turbo';
}

sub _set_suitable {
    my ($app) = @_;

    say 'SET suitable';

    my $blocks = $app->partner_db->internal_context_on_site_rtb->get_all(
        fields => \@fields,
        filter => ['AND', [['id', '<>', \$TECHNICAL_RTB_BLOCK_ID], ['site_version', '=', \'desktop']]]
    );

    foreach my $block (@$blocks) {
        my $msg = $app->internal_context_on_site_rtb->public_id($block);

        if (_can_set_version('desktop', $block)) {
            my $res = _set_version($app, $block, 'desktop');
            $msg .= $res ? $res : ' ok';
        } else {
            my $update = FALSE;

            foreach (qw(mobile turbo)) {
                if (_can_set_version($_, $block)) {
                    $msg .= _set_version($app, $block, $_);

                    $update = TRUE;

                    last;
                }
            }

            $msg .= ' conflict: ' . to_json($block->{'direct_block'}) unless $update;
        }

        say $msg;
    }

    say 'END suitable';
}

sub _set_version {
    my ($app, $block, $version) = @_;

    my $msg          = '';
    my $new_settings = {};

    if ($version ne 'desktop') {
        $new_settings = {form_factor => undef, smart_media => undef};
    }

    if (exists($direct_formats_by_version->{$version}{$block->{'direct_block'}}{'banners'}) && $block->{'limit'}) {
        $new_settings->{'limit'} = undef;
    } elsif (
        exists($direct_formats_by_version->{$version}{$block->{'direct_block'}}{'max_banners'})
        && (  !$block->{'limit'}
            || $block->{'limit'} > $direct_formats_by_version->{$version}{$block->{'direct_block'}}{'max_banners'})
      )
    {
        $new_settings->{'limit'} = $direct_formats_by_version->{$version}{$block->{'direct_block'}}{'max_banners'};
    }

    if (%$new_settings) {
        $msg .= ' updated OLD SETTINGS: ' . to_json($block);

        $new_settings->{'site_version'} = $version;

        $app->partner_db->internal_context_on_site_rtb->edit($block, $new_settings);
    }

    return $msg;
}

sub _can_set_version {
    my ($version, $block) = @_;

    return TRUE if $block->{'is_custom_format_direct'};

    return exists($direct_formats_by_version->{$version}{$block->{'direct_block'}});
}
