#!/usr/bin/perl -w

use qbit;

use Test::Partner2::Simple;
use Test::Partner2::Mock;
use Test::Partner::Utils;

use Test::More;
use Test::Deep qw(cmp_deeply ignore);

sub can_create_vertical_with_url_background_color {
    my (%opts) = @_;

    my $app                 = $opts{app};
    my $model               = $opts{model};
    my $vertical_block_data = $opts{vertical_block_data};

    my $add_return_value = $app->$model->add(%$vertical_block_data);

    my $public_id;

    if (ref($add_return_value) eq '') {
        $public_id = $add_return_value;
    } elsif (ref($add_return_value) eq 'ARRAY' && @$add_return_value == 2) {
        $public_id =
          sprintf('%s%s-%s', $app->$model->public_id_prefix(), $add_return_value->[0], $add_return_value->[1]);
    } else {
        die "$model got unknown type of return value";
    }

    my $block_in_db =
      $app->$model->get($public_id,
        fields => [qw(text_color title_color url_background_color), $app->$model->get_page_id_field_name()],);

    my $page_id = delete($block_in_db->{$app->$model->get_page_id_field_name()});
    my ($block_id) = $public_id =~ /([0-9]+)\z/;

    cmp_deeply(
        $block_in_db,
        {
            text_color           => '0ABB01',
            title_color          => '0ABB02',
            url_background_color => '0ABB03',
        },
        $model . ' - Got expected block settings from database',
    );

    $Test::Partner2::Mock::RUN_IN_EDIT_PAGE = sub {
        my ($data) = @_;

        my $design;

        if ($opts{type} eq 'direct') {
            $design = $data->{0}->{DirectBlocks}->{$block_id}->{Design}{0}{design};
        } elsif ($opts{type} eq 'rtb') {
            my @designs =
              grep {($_->{type} // 'tga') eq 'tga'} values %{$data->{0}->{RtbBlocks}->{$block_id}->{RtbDesign}};
            $design = $designs[0]->{design};
        } else {
            die "$model has no type";
        }

        cmp_deeply(
            {
                textColor          => $design->{textColor},
                titleColor         => $design->{titleColor},
                urlBackgroundColor => $design->{urlBackgroundColor},
            },
            {
                textColor          => '0ABB01',
                titleColor         => '0ABB02',
                urlBackgroundColor => '0ABB03',
            },
            $model . ' - Got expected data in EditPage',
        );
    };

    my $get_campaign_model_name = $app->$model->get_campaign_model_name();
    $app->$get_campaign_model_name->update_in_bk({page_id => $page_id});

    return 1;
}

sub cant_create_vertical_without_url_background_color {
    my (%opts) = @_;

    my $app                 = $opts{app};
    my $model               = $opts{model};
    my $vertical_block_data = $opts{vertical_block_data};
    delete $vertical_block_data->{url_background_color};

    eval {my $add_return_value = $app->$model->add(%$vertical_block_data);};

    is(ref($@), 'Exception::Validator::Errors', "$model - Can't create vertical block without url_background_color");

    return 1;
}

sub cant_create_other_with_url_background_color {
    my (%opts) = @_;

    my $app              = $opts{app};
    my $model            = $opts{model};
    my $other_block_data = $opts{other_block_data};

    eval {my $add_return_value = $app->$model->add(%$other_block_data);};

    is(ref($@), 'Exception::Validator::Errors', "$model - Can't create other block with url_background_color");

    return 1;
}

sub can_create_other_without_url_background_color {
    my (%opts) = @_;

    my $app              = $opts{app};
    my $model            = $opts{model};
    my $other_block_data = $opts{other_block_data};
    $other_block_data->{url_background_color} = undef;

    try {
        $app->$model->add(%$other_block_data);
    }
    catch {
        my ($e) = @_;
        fail($model . ' - Created other block without url_background_color' . "\n" . $e->message());
    };

    pass($model . ' - Created other block without url_background_color');

    return 1;
}

run_tests(
    sub {
        my ($app) = @_;

        mock_bk($app);
        mock_yt_partner($app);
        mock_yt_picategory_mapping($app);

        my $models = {
            context_on_site_direct => {
                vertical_block_data => {
                    get_new_direct_context(
                        {
                            campaign_id => 153745,
                            type        => 'vertical',

                            text_color           => '0ABB01',
                            title_color          => '0ABB02',
                            url_background_color => '0ABB03',
                        }
                    )
                },
                other_block_data => {
                    get_new_direct_context(
                        {
                            campaign_id => 153745,
                            type        => '728x90',

                            title_font_size => undef,
                            font_family     => undef,
                            font_size       => undef,
                            limit           => undef,

                            text_color  => '0ABB01',
                            title_color => '0ABB02',
                        }
                    ),

                    url_background_color => '0ABB03',
                },
                type               => 'direct',
                has_add_validation => 1,
            },
            internal_context_on_site_direct => {
                vertical_block_data => {
                    get_new_internal_direct_context(
                        {
                            campaign_id => 88848,
                            type        => 'vertical',

                            text_color           => '0ABB01',
                            title_color          => '0ABB02',
                            url_background_color => '0ABB03',
                        }
                    )
                },
                other_block_data => {
                    get_new_internal_direct_context(
                        {
                            campaign_id => 88848,
                            type        => '728x90',

                            title_font_size => undef,
                            font_family     => undef,
                            font_size       => undef,
                            limit           => undef,

                            text_color  => '0ABB01',
                            title_color => '0ABB02',
                        }
                    ),
                    url_background_color => '0ABB03',
                },
                type               => 'direct',
                has_add_validation => 0,
            },
            internal_mobile_app_rtb => {
                vertical_block_data => {
                    get_new_internal_mobile_rtb(
                        {
                            application_id => 132439,
                            direct_block   => 'vertical',
                            callouts       => 0,

                            text_color           => '0ABB01',
                            title_color          => '0ABB02',
                            url_background_color => '0ABB03',
                        }
                    )
                },
                other_block_data => {

                    get_new_internal_mobile_rtb(
                        {
                            application_id => 132439,
                            direct_block   => '320x100',

                            text_color  => '0ABB01',
                            title_color => '0ABB02',
                        }
                    ),

                    url_background_color => '0ABB03',
                },
                type               => 'rtb',
                has_add_validation => 1,
            },
            search_on_site_direct => {
                vertical_block_data => {
                    get_new_direct_search(
                        {
                            campaign_id => 111161,
                            type        => 'vertical',

                            text_color           => '0ABB01',
                            title_color          => '0ABB02',
                            url_background_color => '0ABB03',
                        }
                    )
                },
                # Поля 'other_block_data' нет, так как в этом продукте может быть только vertical блок
                type               => 'direct',
                has_add_validation => 0,
            },
        };

        foreach my $model (grep {$_ ne 'internal_mobile_app_rtb'} sort keys %$models) {

            subtest $model => sub {
                note 'Start ' . $model;

                can_create_vertical_with_url_background_color(
                    app                 => $app,
                    model               => $model,
                    vertical_block_data => $models->{$model}->{vertical_block_data},
                    type                => $models->{$model}->{type},
                );

                if ($models->{$model}->{has_add_validation}) {
                    cant_create_vertical_without_url_background_color(
                        app                 => $app,
                        model               => $model,
                        vertical_block_data => $models->{$model}->{vertical_block_data},
                    );
                }

                if (   $models->{$model}->{other_block_data}
                    && $models->{$model}->{has_add_validation}
                    && !$models->{$model}->{has_fixing_values_in_add})
                {
                    cant_create_other_with_url_background_color(
                        app              => $app,
                        model            => $model,
                        other_block_data => $models->{$model}->{other_block_data},
                    );
                }

                if ($models->{$model}->{other_block_data}) {
                    can_create_other_without_url_background_color(
                        app              => $app,
                        model            => $model,
                        other_block_data => $models->{$model}->{other_block_data},
                        type             => $models->{$model}->{type},
                    );
                }
              }
        }
    },
    #    do_not_die_on_fail => 1,
    user => 'yndx-developer'
);
