#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use Test::Partner2::Simple;
use Test::Partner2::Mock qw(mock_format_system);

use Test::More;

use qbit;

run_tests(
    sub {
        my ($app) = @_;

        mock_format_system($app);
        my @accessors = get_tests_data($app);

        foreach my $model (@accessors) {
            my $block =
              $app->$model->get_all(limit => 1, fields => [$app->$model->get_page_id_field_name(), 'id'])->[0];

            if ($model eq 'context_on_site_rtb' || $model eq 'internal_context_on_site_rtb') {
                $app->$model->do_action($block, 'edit', show_video => 1);
                $block = $app->$model->get($block, fields => [qw(campaign_id page_id id show_video)]);
            }

            my @ad_type_list = @{$app->$model->get_ad_type_list(is_video_applicable => $block->{show_video})};
            my $is_sepcpm_video_available =
              $app->$model->DOES('Application::Model::Role::Has::Strategies::ShiftingFields')
              && (0 < grep {'video' eq $_} @ad_type_list);

            $app->$model->partner_db_table()
              ->edit($block, {multistate => $app->$model->get_multistate_by_action('edit')->[0]});

            foreach my $type (@ad_type_list) {
                invalid_cpm($app, $model, $block, $type, $is_sepcpm_video_available);

                correct_cpm($app, $model, $block, $type, $is_sepcpm_video_available);
            }
        }
    },
);

sub get_tests_data {
    my ($app) = @_;

    my @rtb_models_with_cpm_fields = ();

    foreach my $accessor (
        grep {$_ !~ /^ssp_/ && $app->$_->DOES('Application::Model::Role::Has::RTB') && $_ ne 'mobile_mediation_block'}
        keys(%{$app->get_models()})
      )
    {
        my $model_fields = $app->$accessor->get_model_fields();

        next unless $model_fields->{'text_cpm'} && $model_fields->{'media_cpm'};

        push(@rtb_models_with_cpm_fields, $accessor);
    }

    return @rtb_models_with_cpm_fields;
}

sub get_strategy {
    my $is_sepcpm_video_available = (0 != (scalar @_ % 2) ? shift @_ : 0);
    return (
        strategy      => 3,
        mincpm        => undef,
        media_active  => 0,
        media_blocked => undef,
        media_cpm     => undef,
        text_active   => 0,
        text_blocked  => undef,
        text_cpm      => undef,
        (
            $is_sepcpm_video_available
            ? (
                video_active  => 0,
                video_blocked => undef,
                video_cpm     => undef,
              )
            : ()
        ),
        @_
    );
}

sub invalid_cpm {
    my ($app, $model, $block, $type, $is_sepcpm_video_available) = @_;

    eval {
        $app->$model->do_action($block, 'edit',
            get_strategy($is_sepcpm_video_available, "${type}_active" => 1, "${type}_blocked" => 0, "${type}_cpm" => 0)
        );
    };

    ok($@, "$model - throw exception");

    is(
        $@->message,
        to_json(
            [
                {
                    messages => ["${type}_cpm value must be from '0.001' to '9999'"],
                    name     => ["${type}_cpm"]
                }
            ],
            pretty => TRUE
        ),
        "$model - correct message"
      );
}

sub correct_cpm {
    my ($app, $model, $block, $type, $is_sepcpm_video_available) = @_;

    $app->$model->do_action(
        $block, 'edit',
        get_strategy(
            $is_sepcpm_video_available,
            "${type}_active"  => 1,
            "${type}_blocked" => 0,
            "${type}_cpm"     => 32,
        )
    );

    is($app->$model->get($block, fields => ["${type}_cpm"])->{"${type}_cpm"} + 0, 32, "$model - ${type}_cpm = 32");
}
