use strict;
use warnings FATAL => 'all';

use lib::abs qw(
  ../lib
  ../t_lib
  );
use Test::Partner2::Simple;

use qbit;
use Test::More;
use Test::Deep qw(cmp_deeply isa);
use Test::Differences;
use Test::MockObject::Extends;
use Test::Partner::Utils qw(mock_curdate);

my $mock_date = '2020-02-18 18:28:12';
my $page_id   = 123;
my $client_id = 11009;

run_tests(
    sub {
        my ($app) = @_;

        mock_curdate($mock_date);

        $app->api_bk;
        $app->{'api_bk'} = Test::MockObject::Extends->new($app->{'api_bk'});

        my $som = SOAP::SOM->new();
        $app->{'__SOM__'} = Test::MockObject::Extends->new($som);

        create_or_update_campaign__ok($app);
        create_or_update_campaign__fail($app);
    },
);

sub create_or_update_campaign__ok {
    my ($app) = @_;

    _mock_api_bk(
        $app,
        bk_data => {
            0 => {
                IsYandexPage => 0,
                PageID       => $page_id,
                PartnerID    => $client_id,
                Places       => isa('SOAP::Data'),
                PPCTotal     => 9,
                ProductID    => 'context_on_site_campaign',
                Login        => 'mocked-yan-partner',
                UpdateTimePI => $mock_date,
                isPi2        => 1,
                TargetType   => 3,
                PPCTotal     => 9,
                Options      => ''
            },

        },
        response => {
            '0' => {
                'Error'    => '0',
                'PageID'   => $page_id,
                'ErrorStr' => ''
            }
        }
    );

    my $result = $app->api_bk->create_or_update_campaign(
        {
            page_id     => $page_id,
            client_id   => $client_id,
            product_id  => 'context_on_site_campaign',
            target_type => 3,
            Login       => 'mocked-yan-partner',
            PPCTotal    => 9,
            Places      => {},
            isPi2       => 1,
        }
    );

    is($result, $page_id);

    return 1;
}

sub create_or_update_campaign__fail {
    my ($app) = @_;

    subtest 'TargetType error' => sub {

        my $error_message = q[TargetType can't be changed from 3 to 0];

        _mock_api_bk(
            $app,
            bk_data => {
                0 => {
                    IsYandexPage => 0,
                    PageID       => $page_id,
                    PartnerID    => $client_id,
                    Places       => isa('SOAP::Data'),
                    PPCTotal     => 9,
                    ProductID    => 'context_on_site_campaign',
                    Login        => 'mocked-yan-partner',
                    UpdateTimePI => $mock_date,
                    isPi2        => 1,
                    TargetType   => 3,
                    Options      => ''
                },
            },
            response => {
                '0' => {
                    'PageID'   => $page_id,
                    'Error'    => 12,
                    'ErrorStr' => $error_message,
                }
            }
        );

        my $error;

        try {
            my $result = $app->api_bk->create_or_update_campaign(
                {
                    page_id     => $page_id,
                    client_id   => $client_id,
                    product_id  => 'context_on_site_campaign',
                    target_type => 3,
                    Login       => 'mocked-yan-partner',
                    PPCTotal    => 9,
                    Places      => {},
                    isPi2       => 1,
                    Options     => ''
                }
            );
        }
        catch {
            $error = $@;
        };

        is(ref($error), 'Exception::BK::EditPage', 'Exception name correct');
        eq_or_diff(
            $@->message,
            "Got error from banner system. (page_id=123) TargetType can\'t be changed from 3 to 0",
            'Exception message correct'
        );
    };

    subtest 'Lock error' => sub {
        _mock_api_bk(
            $app,
            bk_data  => undef,
            response => {
                'Error'    => '1',
                'ErrorStr' => 'Lock'
            }
        );

        eval {
            $app->api_bk->create_or_update_campaign(
                {
                    page_id     => $page_id,
                    client_id   => $client_id,
                    product_id  => 'context_on_site_campaign',
                    target_type => 3,
                    Login       => 'mocked-yan-partner',
                    PPCTotal    => 9,
                    Places      => {},
                    isPi2       => 1,
                }
            );
        };

        is(ref($@), 'Exception::BK::EditPage', 'Exception name correct');
        is(
            $@->message,
            'Got error from banner system. (page_id=123) {"Error":"1","ErrorStr":"Lock"}',
            'Exception message correct',
          );
    };

    subtest 'Protected error' => sub {

        _mock_api_bk(
            $app,
            bk_data  => undef,
            response => {
                '0' => {
                    'PageID'   => $page_id,
                    'Error'    => '100',
                    'ErrorStr' => 'Protected'
                }
            }
        );

        eval {
            $app->api_bk->create_or_update_campaign(
                {
                    page_id     => $page_id,
                    client_id   => $client_id,
                    product_id  => 'context_on_site_campaign',
                    target_type => 3,
                    Login       => 'mocked-yan-partner',
                    PPCTotal    => 9,
                    Places      => {},
                    isPi2       => 1,
                }
            );
        };

        is(ref($@), 'Exception::BK::Protected', 'Exception name correct for protected page');
    };

    return TRUE;
}

sub _mock_api_bk {
    my ($app,     %opts)     = @_;
    my ($bk_data, $response) = @opts{qw( bk_data  response )};

    $app->{'__SOM__'}->mock('paramsall', sub {$response});

    $app->api_bk->mock(
        'call',
        sub {
            my ($self, $method, @opts) = @_;
            is($method, 'EditPage');

            delete($_->{'0'}->{Slots}) foreach @opts;

            $self->{__SOAP__}->{__OPTS__} = \@opts;

            local $Test::Builder::Level = $Test::Builder::Level + 4;

            cmp_deeply(\@opts, [$bk_data]) if defined $bk_data;

            $self->{__SOAP__}->{__CUR_SOM__} = $app->{'__SOM__'};

            return [$response];
        }
    );
}
