#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use lib::abs qw(
  ./lib
  ../lib
  ../t_lib
  );
use Test::Partner2::Simple;

use Test::Exception;
use Test::Most tests => 37;
use Test::More;
use Test::Deep;
use Test::MockObject::Extends;
use Test::Partner::Utils qw(mock_curdate);

use qbit;

my %_COOKIEMATCH_OPTS;

run_tests(
    sub {
        my ($app) = @_;

        mock_app($app);

        cmp_deeply(
            $app->cookie_match->get_model_fields,
            {
                fields_depends => {
                    'api'      => 1,
                    depends_on => ['id'],
                    get        => ignore(),
                    type       => 'complex'
                },
                status => {
                    'api'      => 1,
                    depends_on => ['id'],
                    get        => ignore(),
                    type       => 'string'
                },
                public_id => {
                    'api'       => 1,
                    db          => TRUE,
                    db_expr     => 'id',
                    type        => 'string',
                    adjust_type => 'str',
                },

                'owners' => {
                    'api'        => 1,
                    'get'        => ignore(),
                    'depends_on' => ['id'],
                    type         => 'array',
                    'sub_type'   => 'users',
                    need_check   => {
                        optional => TRUE,
                        type     => 'array',
                        check    => ignore(),
                    },
                },
                'create_date' => {
                    'api'        => 1,
                    'get'        => ignore(),
                    'label'      => ignore(),
                    'depends_on' => ['id'],
                    type         => 'string',
                },
                'actions' => {
                    'api'        => 1,
                    'get'        => ignore(),
                    'label'      => ignore(),
                    'depends_on' => ['id', 'multistate', 'owners'],
                    type         => 'complex',
                    fix_type     => ignore(),
                },
                'multistate_name' => {
                    'api'        => 1,
                    'get'        => ignore(),
                    'label'      => ignore(),
                    'depends_on' => ['multistate'],
                    type         => 'string',
                },
                'page_ids' => {
                    'api'        => 1,
                    'get'        => ignore(),
                    'label'      => ignore(),
                    'depends_on' => ['id', 'owners'],
                    type         => 'array',
                    sub_type     => 'number',
                },
                'skip_data_key' => {
                    'api'      => 1,
                    'db'       => 1,
                    'label'    => ignore(),
                    type       => 'boolean',
                    need_check => {
                        type     => 'boolean',
                        optional => TRUE,
                    },
                    adjust_type => 'str',
                },
                'multistate' => {
                    'api'       => 1,
                    'db'        => 1,
                    'label'     => ignore(),
                    type        => 'number',
                    adjust_type => 'str',
                },
                'js_redir' => {
                    'api'      => 1,
                    'db'       => 1,
                    'label'    => ignore(),
                    type       => 'boolean',
                    need_check => {
                        type     => 'boolean',
                        optional => TRUE,
                    },
                    adjust_type => 'str',
                },
                'comment' => {
                    'api'          => 1,
                    'check_rights' => 'cookie_match_view_field__comment',
                    'db'           => 1,
                    'label'        => ignore(),
                    type           => 'string',
                    need_check     => {
                        len_max  => 255,
                        optional => TRUE,
                    },
                },
                'data_key' => {
                    'api'      => 1,
                    'db'       => 1,
                    'label'    => ignore(),
                    'default'  => 1,
                    type       => 'string',
                    need_check => {check => ignore(),},
                },
                'is_deleted' => {
                    'api'        => 1,
                    'get'        => ignore(),
                    'depends_on' => ['multistate'],
                    type         => 'boolean',
                },
                'available_fields' => {
                    'api'        => 1,
                    'get'        => ignore(),
                    'label'      => ignore(),
                    'depends_on' => ['id', 'multistate'],
                    type         => 'complex',
                    fix_type     => ignore(),
                },
                'id' => {
                    'api'       => 1,
                    'pk'        => 1,
                    'db'        => 1,
                    'label'     => ignore(),
                    'default'   => 1,
                    type        => 'number',
                    adjust_type => 'str',
                },
                'tag' => {
                    'api'      => 1,
                    'db'       => 1,
                    'label'    => ignore(),
                    'default'  => 1,
                    type       => 'string',
                    need_check => {check => ignore(),}
                },
                'editable_fields' => {
                    'api'        => 1,
                    'get'        => ignore(),
                    'depends_on' => ['id', 'multistate'],
                    type         => 'complex',
                    fix_type     => ignore()
                },
                trace_back_reference => {
                    'api'      => 1,
                    'db'       => 1,
                    'label'    => ignore(),
                    type       => 'boolean',
                    need_check => {
                        type     => 'boolean',
                        optional => TRUE,
                    },
                    adjust_type => 'str',
                }
            },
            'get_model_fields'
        );

        cmp_deeply(
            $app->cookie_match->get_db_filter_fields,
            {
                'owners' => {
                    'subfields' => ignore(),
                    'label'     => ignore(),
                    'type'      => 'subfilter'
                },
                'comment' => {
                    'label' => ignore(),
                    'type'  => 'text'
                },
                'data_key' => {
                    'label' => ignore(),
                    'type'  => 'text'
                },
                'tag' => {
                    'label' => ignore(),
                    'type'  => 'text'
                },
                'id' => {
                    'label' => ignore(),
                    'type'  => 'number'
                },
                'multistate' => {
                    'values' => {
                        'deleted' => ignore(),
                        'new'     => ignore(),
                    },
                    'label' => ignore(),
                    'type'  => 'multistate'
                },
                'page_id' => {
                    'label' => ignore(),
                    'type'  => 'number'
                },
                'pi_page' => {
                    'subfields' => ignore(),
                    'label'     => ignore(),
                    'type'      => 'subfilter'
                },
            },
            'get_db_filter_fields'
        );

        cmp_deeply(
            $app->cookie_match->get_db_filter_simple_fields,
            [
                [
                    {
                        'name'  => 'tag',
                        'label' => ignore(),
                    }
                ],
                [
                    {
                        'name'  => 'page_id',
                        'label' => ignore(),
                    },
                    {
                        'name'  => 'pi_page.page_id',
                        'label' => ignore(),
                    }
                ]
            ],
            'get_db_filter_simple_fields'
        );

        cmp_deeply(
            $app->cookie_match->get_add_fields(),
            {
                'js_redir'             => 1,
                'comment'              => 1,
                'owners'               => 1,
                'tag'                  => 1,
                'skip_data_key'        => 1,
                'data_key'             => 1,
                'trace_back_reference' => 1,
            },
            'get_add_fields'
        );

        # checking tag validation
        throws_ok(
            sub {
                $app->cookie_match->add(
                    tag           => 'TAG',
                    data_key      => '42idxavj7m8eiuqybaqmsrn21s6sz0y3',
                    comment       => 'Comment',
                    owners        => [1009],
                    skip_data_key => 1,
                    js_redir      => 0,
                );
            },
            'Exception::Validator::Errors',
            'validation fails on tag=TAG'
        );
        throws_ok(
            sub {
                $app->cookie_match->add(
                    tag           => '1234567890123456789012345678901234567890',
                    data_key      => '42idxavj7m8eiuqybaqmsrn21s6sz0y3',
                    comment       => 'Comment',
                    owners        => [1009],
                    skip_data_key => 1,
                    js_redir      => 0,
                );
            },
            'Exception::Validator::Errors',
            'validation fails on tag=123456789012345678901234567890123'
        );
        throws_ok(
            sub {
                $app->cookie_match->add(
                    tag           => 'фыва',
                    data_key      => '42idxavj7m8eiuqybaqmsrn21s6sz0y3',
                    comment       => 'Comment',
                    owners        => [1009],
                    skip_data_key => 1,
                    js_redir      => 0,
                );
            },
            'Exception::Validator::Errors',
            'validation fails on tag=фыва'
        );
        throws_ok(
            sub {
                $app->cookie_match->add(
                    tag           => '',
                    data_key      => '42idxavj7m8eiuqybaqmsrn21s6sz0y3',
                    comment       => 'Comment',
                    owners        => [1009],
                    skip_data_key => 1,
                    js_redir      => 0,
                );
            },
            'Exception::Validator::Errors',
            'validation fails on tag=""'
        );
        # checking comment validation
        throws_ok(
            sub {
                $app->cookie_match->add(
                    tag           => 'tag0',
                    data_key      => '42idxavj7m8eiuqybaqmsrn21s6sz0y3',
                    comment       => 'c' x 256,
                    owners        => [1009],
                    skip_data_key => 1,
                    js_redir      => 0,
                );
            },
            'Exception::Validator::Errors',
            'validation fails on comment > 255 chars'
        );
        ok($@->message =~ /length .+ more than/igm, 'ok validation err msg on too long comment');
        # checking data_key validation
        throws_ok(
            sub {
                $app->cookie_match->add(
                    tag           => 'tag0',
                    data_key      => '42idxavj7m8eiuqybaqmsrn21s6sz0y',
                    comment       => 'Comment',
                    owners        => [1009],
                    skip_data_key => 1,
                    js_redir      => 0,
                );
            },
            'Exception::Validator::Errors',
            'validation fails on data_key < 32 chars'
        );
        ok($@->message =~ /and must be 32 characters in length/igm, 'ok validation err msg on too short data_key');
        throws_ok(
            sub {
                $app->cookie_match->add(
                    tag           => 'tag0',
                    data_key      => '42idxavj7m8eiuqybaqmsrn21s6sz0y30',
                    comment       => 'Comment',
                    owners        => [1009],
                    skip_data_key => 1,
                    js_redir      => 0,
                );
            },
            'Exception::Validator::Errors',
            'validation fails on data_key > 32 chars'
        );
        ok($@->message =~ /and must be 32 characters in length/igm, 'ok validation err msg on too long data_key');
        throws_ok(
            sub {
                $app->cookie_match->add(
                    tag           => 'tag0',
                    data_key      => '42idxavj7m8eiuqybaqmsrn21s6sz0yZ',
                    comment       => 'Comment',
                    owners        => [1009],
                    skip_data_key => 1,
                    js_redir      => 0,
                );
            },
            'Exception::Validator::Errors',
            'validation fails on data_key with UPPERCASE'
        );
        ok($@->message =~ /must consist of latin lowercase letters and numbers/igm,
            'ok validation err msg on UPPERCASE');
        # checking owners validation
        throws_ok(
            sub {
                $app->cookie_match->add(
                    tag           => 'tag0',
                    data_key      => '42idxavj7m8eiuqybaqmsrn21s6sz0y3',
                    comment       => 'Comment',
                    owners        => [11, 'ggwp'],
                    skip_data_key => 1,
                    js_redir      => 0,
                );
            },
            'Exception::Validator::Errors',
            'validation fails on owners=["ggwp"]'
        );
        ok($@->message =~ /must be numbers/igm, 'ok validation err msg on not number owner id');
        throws_ok(
            sub {
                $app->cookie_match->add(
                    tag           => 'tag0',
                    data_key      => '42idxavj7m8eiuqybaqmsrn21s6sz0y3',
                    comment       => 'Comment',
                    owners        => [2**32],
                    skip_data_key => 1,
                    js_redir      => 0,
                );
            },
            'Exception::Validator::Errors',
            'validation fails on owners=[2**32]'
        );
        ok($@->message =~ /These id`s not found/igm, 'ok validation err msg on not found owner id');

        %_COOKIEMATCH_OPTS = (
            tag           => 'tag',
            data_key      => '42idxavj7m8eiuqybaqmsrn21s6sz0y3',
            comment       => 'Comment',
            owners        => [1009],
            skip_data_key => 1,
            js_redir      => 0,
        );

        my $id = $app->cookie_match->add(%_COOKIEMATCH_OPTS);

        cmp_deeply(
            $app->cookie_match->get($id, fields => ['*']),
            {
                'public_id'      => $id,
                'fields_depends' => ignore(),
                'status'         => 'sinc',
                'owners'         => [
                    {
                        'client_id' => '11009',
                        'id'        => '1009',
                        'login'     => 'mocked-yan-partner'
                    }
                ],
                'actions' => {
                    'delete' => ignore(),
                    'edit'   => ignore(),
                },
                'create_date'          => '2016-02-04 19:32:13',
                'multistate_name'      => ignore(),
                'skip_data_key'        => '1',
                'multistate'           => '1',
                'page_ids'             => [],
                'js_redir'             => '0',
                'trace_back_reference' => '0',
                'comment'              => 'Comment',
                'data_key'             => '42idxavj7m8eiuqybaqmsrn21s6sz0y3',
                'is_deleted'           => '',
                'available_fields'     => {
                    'owners'               => 1,
                    'actions'              => 1,
                    'multistate_name'      => 1,
                    'skip_data_key'        => 1,
                    'multistate'           => 1,
                    'js_redir'             => 1,
                    'comment'              => 1,
                    'data_key'             => 1,
                    'available_fields'     => 1,
                    'is_deleted'           => 1,
                    'tag'                  => 1,
                    'id'                   => 1,
                    'editable_fields'      => 1,
                    'page_ids'             => 1,
                    'create_date'          => 1,
                    'public_id'            => 1,
                    'fields_depends'       => 1,
                    'status'               => 1,
                    'trace_back_reference' => 1,
                },
                'id'              => $id,
                'tag'             => 'tag',
                'editable_fields' => {
                    'js_redir'             => 1,
                    'comment'              => 1,
                    'skip_data_key'        => 1,
                    'owners'               => 1,
                    'trace_back_reference' => 1,
                }
            },
            'add'
        );

        %_COOKIEMATCH_OPTS = (
            js_redir             => 1,
            trace_back_reference => 1,
            skip_data_key        => 0,
            comment              => 'Fix comment',
            owners               => [1009, 1016, 1010]
        );

        $app->cookie_match->do_action($id, 'edit', %_COOKIEMATCH_OPTS);

        cmp_deeply(
            $app->cookie_match->get($id, fields => ['*']),
            {
                'public_id'      => $id,
                'fields_depends' => ignore(),
                'status'         => 'sinc',
                'owners'         => [
                    {
                        'client_id' => '11009',
                        'id'        => '1009',
                        'login'     => 'mocked-yan-partner'
                    },
                    {
                        'client_id' => '11010',
                        'id'        => '1010',
                        'login'     => 'mocked-video-partner'
                    },
                    {
                        'client_id' => '11016',
                        'id'        => '1016',
                        'login'     => 'mocked-context-adblock-partner'
                    }
                ],
                'actions' => {
                    'delete' => ignore(),
                    'edit'   => ignore(),
                },
                'create_date'          => '2016-02-04 19:32:13',
                'multistate_name'      => ignore(),
                'skip_data_key'        => '0',
                'multistate'           => '1',
                'page_ids'             => [],
                'js_redir'             => '1',
                'trace_back_reference' => '1',
                'comment'              => 'Fix comment',
                'data_key'             => '42idxavj7m8eiuqybaqmsrn21s6sz0y3',
                'is_deleted'           => '',
                'available_fields'     => {
                    'owners'               => 1,
                    'actions'              => 1,
                    'multistate_name'      => 1,
                    'skip_data_key'        => 1,
                    'multistate'           => 1,
                    'js_redir'             => 1,
                    'comment'              => 1,
                    'data_key'             => 1,
                    'available_fields'     => 1,
                    'is_deleted'           => 1,
                    'tag'                  => 1,
                    'id'                   => 1,
                    'editable_fields'      => 1,
                    'page_ids'             => 1,
                    'create_date'          => 1,
                    'public_id'            => 1,
                    'fields_depends'       => 1,
                    'status'               => 1,
                    'trace_back_reference' => 1,
                },
                'id'              => $id,
                'tag'             => 'tag',
                'editable_fields' => {
                    'js_redir'             => 1,
                    'comment'              => 1,
                    'skip_data_key'        => 1,
                    'owners'               => 1,
                    'trace_back_reference' => 1,
                }
            },
            'edit'
        );

        $app->cookie_match->do_action($id, 'delete');

        is(
            $app->cookie_match->check_multistate_flag(
                $app->cookie_match->get($id, fields => ['multistate'])->{'multistate'}, 'deleted'
            ),
            TRUE,
            'deleted'
          );

        $app->cookie_match->do_action($id, 'restore');

        is(
            $app->cookie_match->check_multistate_flag(
                $app->cookie_match->get($id, fields => ['multistate'])->{'multistate'}, 'deleted'
            ),
            FALSE,
            'restored'
          );

        cmp_deeply(
            $app->cookie_match->get_by_user_id(1009),
            [
                {
                    'js_redir'      => '1',
                    'data_key'      => '42idxavj7m8eiuqybaqmsrn21s6sz0y3',
                    'tag'           => 'tag',
                    'id'            => $id,
                    'skip_data_key' => '0'
                }
            ],
            'get_by_user_id(1009)'
        );

        cmp_deeply(
            $app->cookie_match->get_by_user_id(1010),
            [
                {
                    'js_redir'      => '1',
                    'data_key'      => '42idxavj7m8eiuqybaqmsrn21s6sz0y3',
                    'tag'           => 'tag',
                    'id'            => $id,
                    'skip_data_key' => '0'
                }
            ],
            'get_by_user_id(1010)'
        );

        my $test_data = [
            {
                model => 'context_on_site_campaign',
                id    => 1,
            },
            {
                model => 'search_on_site_campaign',
                id    => 111161,
            },
            {
                model => 'video_an_site',
                id    => 154689,
            }
        ];

        cmp_deeply(
            [map {$_->{'model'}} @$test_data],
            [$app->cookie_match->pages_with_user_synchronization],
            'check pages with user_synchronization'
        );

        foreach my $test (@$test_data) {
            my ($model, $model_id) = @$test{qw(model id)};

            $app->$model->do_action($model_id, 'edit', user_synchronization => 1, tag_id => $id);
            $app->$model->update_in_bk({id => $model_id});

            cmp_deeply(
                $app->$model->get($model_id, fields => [qw(user_synchronization tag_id cookie_match)]),
                {
                    'user_synchronization' => '1',
                    'tag_id'               => $id,
                    'cookie_match'         => {
                        'js_redir'      => '1',
                        'data_key'      => '42idxavj7m8eiuqybaqmsrn21s6sz0y3',
                        'tag'           => 'tag',
                        'id'            => $id,
                        'skip_data_key' => '0'
                    }
                },
                "added tag for $model"
            );
        }
    },
    init => [qw(cookie_match)]
);

sub mock_app {
    my ($app) = @_;

    mock_curdate('2016-02-04 19:32:13');

    no strict 'refs';
    no warnings 'redefine';
    *{'Application::Model::CookieMatch::_generate_data_key'} = sub {'42idxavj7m8eiuqybaqmsrn21s6sz0y3'};

    $app->api_bk;
    $app->{'api_bk'} = Test::MockObject::Extends->new($app->{'api_bk'});
    $app->api_bk->mock(
        'call',
        sub {
            is($_[2]->{0}{CookieMatchTag}, 'tag', 'check update in bk');
            return [{'0' => {'Error' => 0, 'PageID' => 1}}];
        }
    );

    $app->api_http_bk;
    $app->{'api_http_bk'} = Test::MockObject::Extends->new($app->{'api_http_bk'});
    $app->api_http_bk->mock(
        'call',
        sub {
            my ($self, $url, %opts) = @_;
            if ('export/import-cookie-match-settings.cgi' eq $url) {
                is_deeply(
                    from_json($opts{':content'}),
                    [
                        {
                            'CookieMatchTag' => 'tag',
                            'DataKey'        => '42idxavj7m8eiuqybaqmsrn21s6sz0y3',
                            'InitURL'        => '',
                            'Options'        => {
                                'jsredir'              => $_COOKIEMATCH_OPTS{js_redir}             // '',
                                'skipdatakey'          => $_COOKIEMATCH_OPTS{skip_data_key}        // '',
                                'trace-back-reference' => $_COOKIEMATCH_OPTS{trace_back_reference} // '',
                            }
                        }
                    ],
                    'ok create_or_update_in_bk'
                );
            }
            return '[{}]';
        }
    );
}
