#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use FindBin qw($Bin);
use lib "$Bin/lib";
use lib "$Bin/../lib";
use lib "$Bin/../utils/lib";

use Test::Partner2::Simple;

use Test::More;
use Test::Deep;
use Test::Partner2::Mock;
use Test::Partner::Utils qw(set_test_partner_utils_app get_hash_values restore_table get_new_user add_bookmaker_filter);

use qbit;
use Utils::PublicID;

my @LOCALES              = ();
my $DEFAULT_QUALITY_COEF = 80;

$Test::Partner2::Mock::PAGE_ID_START = 87940;

run_tests(
    sub {
        my ($app) = @_;
        mock_utils_partner2($app);

        set_test_partner_utils_app($app);
        my $dbh = $app->partner_db->get_dbh();

        @LOCALES = keys %{$app->get_option('locales')};

        my $table_dumps_dir = $app->get_option('ApplicationPath') . 't_old_test_data/statistics/2014_03_01/table_dumps';
        restore_table($dbh, $table_dumps_dir, 'tns_dict_article');

        restore_table($dbh, $table_dumps_dir, 'tns_dict_article');

        $dbh->do('DELETE FROM users_action_log');
        $dbh->do('DELETE FROM users');

        my $manager = get_new_user();

        my @rights = qw(
          do_internal_context_on_site_add
          internal_context_on_site_campaign_view_all
          internal_context_on_site_stripe_view_all
          do_internal_context_on_site_stripe_add
          internal_context_on_site_stripe_view_field__comment
          internal_context_on_site_campaign_view_field__banner_lang
          do_internal_context_on_site_register_in_balance
          do_internal_context_on_site_start
          internal_context_on_site_stripe_edit
          internal_context_on_site_campaign_edit_all
          );

        $app->{'__CURRENT_USER_RIGHTS__'} = {};
        $app->set_cur_user({id => $manager->{'id'}});
        $app->{'__CURRENT_USER_RIGHTS__'} = {map {$_ => TRUE} @rights};

        $dbh->do('DELETE FROM quality_def_coef');
        $app->partner_db->quality_def_coef->add({k => $DEFAULT_QUALITY_COEF, dt => curdate(oformat => 'db')});

        $app->api_bk;
        $app->{api_bk} = Test::MockObject::Extends::Easy->new($app->{api_bk});

        $app->users;
        $app->{users} = Test::MockObject::Extends::Easy->new($app->{users});
        $app->users->set_always('get_login_by_client_id', 'adinside');

        $app->api_balance;
        $app->{api_balance} = Test::MockObject::Extends::Easy->new($app->{api_balance});

        my $campaign_domain = 'auto.yandex.ru';

        $app->api_bk->mock_get_return(
            'create_or_update_campaign',
            mock_create_or_update_campaigns_method(
                page_id   => 87940,
                domain    => $campaign_domain,
                client_id => 944028,
            ),
        );

        add_bookmaker_filter($app);

        $app->api_balance->set_always('create_or_update_place', TRUE);

        {
            my $campaign_id = $app->internal_context_on_site_campaign->add(domain => $campaign_domain);
            $app->internal_context_on_site_campaign->do_action($campaign_id, 'register_in_balance');
            $app->partner_db->managers->add(
                {page_id => $campaign_id, manager_id => $app->get_option('cur_user')->{'id'}});

            my $public_id = $app->internal_context_on_site_stripe->add(campaign_id => $campaign_id);
            is(scalar(@{$app->internal_context_on_site_stripe->get_all(fields => [qw(campaign_id id)])}),
                1, 'One and only one stripe is added.');

            my $stripe = {};
            @$stripe{qw(campaign_id id)} = (split_block_public_id($public_id))[1, 2];
            $stripe->{'domain'} = $campaign_domain;

            my $multistate = 18;    # working, need_update

            my @fields =
              qw(campaign_id id domain multistate title disposition_id interaction_id type_id animation comment public_id editable_fields available_fields actions);
            my @fields_default_values = (
                campaign_id     => $stripe->{'campaign_id'},
                id              => $stripe->{'id'},
                domain          => $campaign_domain,
                disposition_id  => any(0, 1),
                interaction_id  => any(0, 1),
                type_id         => any(1, 4),
                animation       => any(0, 1),
                title           => re(qr(.)),
                comment         => undef,
                public_id       => public($stripe->{'campaign_id'}, $stripe->{'id'}),
                editable_fields => {
                    title          => TRUE,
                    comment        => TRUE,
                    disposition_id => $app->internal_context_on_site_stripe->get_disposition_ids(),
                    interaction_id => $app->internal_context_on_site_stripe->get_interaction_with_content_ids(),
                    type_id        => $app->internal_context_on_site_stripe->get_type_ids(),
                    animation      => $app->internal_context_on_site_stripe->get_animation_ids(),
                },
                available_fields => ignore(),
                actions          => {
                    delete               => re(qr(\w)),
                    delete_with_page     => re(qr(\w)),
                    stop                 => re(qr(\w)),
                    edit                 => re(qr(\w)),
                    set_check_statistics => re(qr(\w)),
                },
                multistate => $multistate,
            );
            cmp_deeply(
                $app->internal_context_on_site_stripe->get($stripe, fields => \@fields),
                {@fields_default_values,},
                'The added stripe is correct.'
            );

            $app->internal_context_on_site_stripe->do_action($stripe, 'edit');

            cmp_deeply(
                $app->internal_context_on_site_stripe->get($stripe, fields => \@fields),
                {@fields_default_values,},
                'The started stripe is correct.'
            );

            my $title   = 'New title';
            my $comment = 'Commented';
            my %changes = (
                title          => "   $title  ",
                disposition_id => 0,
                interaction_id => 0,
                type_id        => 1,
                animation      => 0,
                comment        => "   $comment   "
            );
            $app->internal_context_on_site_stripe->do_action($stripe, 'edit', %changes);

            cmp_deeply(
                $app->internal_context_on_site_stripe->get($stripe, fields => \@fields),
                {
                    @fields_default_values,
                    %changes,
                    title   => $title,
                    comment => $comment,
                },
                'The edited stripe is correct.'
            );

            is(scalar(@{$app->partner_db->internal_context_on_site_stripe_action_log->get_all()}),
                4, 'Actions are logged.');

            $app->api_bk->mock_get_return(
                'create_or_update_campaign',
                mock_create_or_update_campaigns_method(
                    page_id   => 87941,
                    domain    => $campaign_domain,
                    client_id => 944028,
                ),
            );

            $app->internal_context_on_site_campaign->add(domain => $campaign_domain);
            $app->partner_db->managers->add(
                {page_id => $campaign_id + 1, manager_id => $app->get_option('cur_user')->{'id'}});

            is(scalar(@{$app->internal_context_on_site_stripe->get_all_campaigns_for_adding()}),
                2, 'get_all_campaigns_for_adding()');

            #    subtest(
            #        'Trying to stop not existing stripe.' => sub {
            #            eval {$app->internal_context_on_site_stripe->do_action({campaign_id => 60, id => 10}, 'stop');};
            #            ok($@, 'Exception is throwed');
            #            is(ref($@),       'Exception::Validation::BadArguments',    'Exception is correct object');
            #            is($@->message(), 'Invalid primary key fields', 'Exception message is correct');
            #        }
            #    );
            subtest(
                'Trying to fly existing stripe.' => sub {
                    is($app->internal_context_on_site_stripe->check_action($stripe, 'fly'), FALSE, "Stripe can't fly!");
                    eval {$app->internal_context_on_site_stripe->do_action($stripe, 'fly');};
                    ok($@, 'Exception is throwed');
                    is(ref($@), 'Exception::Multistate::BadAction', 'Exception is correct object');
                    my $exception_message = $@->message();
                    like(
                        $exception_message,
qr/Cannot do action "fly" \(login "unknown", multistate $multistate, pk \{("id":"1")?,?("campaign_id":"87940"){1},?("id":"1")?\}, model "internal_context_on_site_stripe", action not exist, right "not defined"\)\./,
                        'Exception message is correct'
                    );
                }
            );
        }
    },
    fill_databases => 0,
);

sub public {
    return 'S-I-' . $_[0] . '-' . $_[1];
}

sub mock_create_or_update_campaigns_method {
    my (%opts) = @_;
    return

      get_create_or_update_campaigns_general_data(%opts, addition => TRUE),   => [$opts{page_id}],
      get_create_or_update_campaigns_general_data(%opts, state    => 'work'), => [$opts{page_id}],
      get_create_or_update_campaigns_general_data(%opts, with_stripe => TRUE, state => 'work') => [$opts{page_id}],
      get_create_or_update_campaigns_general_data(%opts, state => 'work'), => [$opts{page_id}],
      get_create_or_update_campaigns_general_data(%opts, with_stripe => TRUE, state => 'work') => [$opts{page_id}],
      get_create_or_update_campaigns_general_data(
        %opts,
        with_stripe => TRUE,
        animation   => 0,
        type_id     => 1,
        state       => 'work'
      ) => [$opts{page_id}],
      get_create_or_update_campaigns_general_data(%opts, addition => TRUE) => [$opts{page_id} + 1];
}

sub get_create_or_update_campaigns_general_data {
    my (%opts) = @_;

    my %data = (
        client_id       => $opts{'client_id'},
        state           => $opts{'state'} ? $opts{'state'} : 'stopped',
        domain          => $opts{'domain'},
        name            => $opts{'domain'},
        description     => $opts{'domain'},
        target_type     => 3,
        context         => 1,
        fast_context    => 1,
        only_picture    => 0,
        mobile_app_mode => 0,
        is_yandex_page  => 1,
    );

    return [\%data] if $opts{addition};

    %data = (
        %data,
        banner_language  => bag(@LOCALES),
        page_id          => $opts{'page_id'},
        cpa              => $DEFAULT_QUALITY_COEF,
        mirrors          => [],
        excluded_domains => ['auto.yandex.ru'],
        options          => {
            BlockTitle       => undef,
            ReloadTimeout    => 30,
            dontshowbehavior => 0,
            dontshowsex      => 0
        },
        pictures_enabled => 1,
        disabled_flags   => [],
        product_type     => 'context',
    );
    if ($opts{with_stripe}) {
        %data = (
            %data,

            banners_count => 9,
            places        => {
                956 => {
                    StripeAnimation => 0,
                    StripeType      => 1,
                },
            },
        );
        $data{places}{956}{StripeAnimation} = $opts{animation} if defined($opts{animation});
        $data{places}{956}{StripeType}      = $opts{type_id}   if defined($opts{type_id});
    } else {
        $data{places} = {};
    }
    return [{%data}];
}
