use Test::More;
use Test::MockObject::Extends;

use lib::abs qw(
  ../utils/lib
  ./lib
  ../lib
  ../t_lib
  );

use qbit;
use Application;
use Test::Partner::DB::Mock;
use Test::Partner2::Mock qw(mock_subs mock_sort_sql);

use constant DEFAULT_DATE        => '2013-08-08';
use constant DEFAULT_COEFFICIENT => 80;

my $app = Application->new();
$app->pre_run();

$app->api_conv_db;
$app->{api_conv_db} = Test::MockObject::Extends->new($app->{api_conv_db});

$app->product_manager;
$app->{product_manager} = Test::MockObject::Extends->new($app->{product_manager});

$app->api_bk;
$app->{api_bk} = Test::MockObject::Extends->new($app->{api_bk});

my $dbh = Test::Partner::DB::Mock->connect($app);
$dbh->mock($app);
$dbh->create_tables(partner_db => [qw(quality_def_coef quality_pages_coef)]);

mock_sort_sql();

subtest(
    'QualityCoef->get_def_coef' => sub {
        my $sub_get_sql_queries_expected = sub {
            [
                [
                        "SELECT\n    `quality_def_coef`.`k` AS `k`\nFROM `quality_def_coef`\n"
                      . "WHERE (\n    `quality_def_coef`.`dt` <= '$_[0]'\n)\nORDER BY `dt` DESC\nLIMIT 0, 1"
                ],
            ];
        };

        my $def_coef_expected  = DEFAULT_COEFFICIENT;
        my $sql_queries_gotten = [];
        $app->{partner_db}
          ->mock(_get_all => sub {push @$sql_queries_gotten, [@_[1 .. $#_]]; [{k => $def_coef_expected}]});

        subtest(
            "QualityCoef->get_def_coef()" => sub {
                my $sql_queries_expected = $sub_get_sql_queries_expected->(curdate(oformat => 'db'));
                $sql_queries_gotten = [];

                my $def_coef_gotten = $app->quality_coef->get_def_coef();
                &is($def_coef_gotten, $def_coef_expected, 'QualityCoef->get_def_coef() returns right coefficient.');
                &is_deeply($sql_queries_gotten, $sql_queries_expected, 'Right sql queries are generated.');
            }
        );
        subtest(
            "QualityCoef->get_def_coef(date => '" . DEFAULT_DATE . "')" => sub {
                my $sql_queries_expected = $sub_get_sql_queries_expected->(DEFAULT_DATE);
                $sql_queries_gotten = [];

                my $def_coef_gotten = $app->quality_coef->get_def_coef(date => DEFAULT_DATE);
                &is($def_coef_gotten, $def_coef_expected, 'QualityCoef->get_def_coef() returns right coefficient.');
                &is_deeply($sql_queries_gotten, $sql_queries_expected, 'Right sql queries are generated.');
            }
        );
    }
);
subtest(
    'QualityCoef->get_pages_coef' => sub {
        my $sub_get_sql_queries_expected = sub {
            [
                [
                        "SELECT\n"
                      . "    `quality_pages_coef`.`dt` AS `dt`\n"
                      . "FROM `quality_pages_coef`\n"
                      . "WHERE (\n"
                      . "    `quality_pages_coef`.`dt` <= '$_[0]'\n"
                      . (
                        $_[1] ? "    AND `quality_pages_coef`.`page_id` IN ('" . join("', '", sort @{$_[1]}) . "')\n"
                        : ""
                      )
                      . ")\n"
                      . "ORDER BY `dt` DESC\n"
                      . "LIMIT 0, 1"
                ],
                [
                        "SELECT\n"
                      . "    `quality_pages_coef`.`k` AS `k`,\n"
                      . "    `quality_pages_coef`.`page_id` AS `page_id`\n"
                      . "FROM `quality_pages_coef`\n"
                      . "WHERE (\n"
                      . "    `quality_pages_coef`.`dt` = '$_[0]'\n"
                      . (
                        $_[1] ? "    AND `quality_pages_coef`.`page_id` IN ('" . join("', '", sort @{$_[1]}) . "')\n"
                        : ""
                      )
                      . ")"
                ],
                (
                    $_[1]
                    ? [
                            "SELECT\n"
                          . "    `quality_def_coef`.`k` AS `k`\n"
                          . "FROM `quality_def_coef`\n"
                          . "WHERE (\n"
                          . "    `quality_def_coef`.`dt` <= '$_[0]'\n" . ")\n"
                          . "ORDER BY `dt` DESC\n"
                          . "LIMIT 0, 1"
                      ]
                    : ()
                )
            ];
        };
        my $sub_get_sql_query_results = sub {
            (
                [{dt      => $_[0]}],
                [{page_id => 69037, k => 83}, {page_id => 72521, k => 92}],
                ($_[1] ? [{k => DEFAULT_COEFFICIENT}] : ()),
            );
        };

        my $sql_queries_gotten = [];
        my @sql_query_results  = [];
        $app->{partner_db}->mock(_get_all => sub {push @$sql_queries_gotten, [@_[1 .. $#_]]; shift @sql_query_results});

        my $pages_coef_expected = {69037 => 83, 72521 => 92};

        subtest(
            'QualityCoef->get_pages_coef()' => sub {
                my $date = curdate(oformat => "db");
                $sql_queries_gotten = [];
                @sql_query_results  = $sub_get_sql_query_results->($date);

                my $pages_coef_gotten = $app->quality_coef->get_pages_coef();
                &is_deeply($pages_coef_gotten, $pages_coef_expected,
                    'QualityCoef->get_pages_coef returns right coefficients.');
                &is_deeply(
                    $sql_queries_gotten,
                    $sub_get_sql_queries_expected->($date),
                    'Right sql queries are generated.'
                );
            }
        );
        subtest(
            "QualityCoef->get_pages_coef(date => '" . DEFAULT_DATE . "')" => sub {
                my $date = DEFAULT_DATE;
                $sql_queries_gotten = [];
                @sql_query_results  = $sub_get_sql_query_results->($date);

                my $pages_coef_gotten = $app->quality_coef->get_pages_coef(date => DEFAULT_DATE);
                &is_deeply($pages_coef_gotten, $pages_coef_expected,
                    'QualityCoef->get_pages_coef returns right coefficients.');
                &is_deeply(
                    $sql_queries_gotten,
                    $sub_get_sql_queries_expected->($date),
                    'Right sql queries are generated.'
                );
            }
        );
        subtest(
            "QualityCoef->get_pages_coef(date => '" . DEFAULT_DATE . "', pages => [...])" => sub {
                my $date = DEFAULT_DATE;
                my $pages = [69037, 72521, 11111, 22222];
                my $pages_coef_expected =
                  {%$pages_coef_expected, 11111 => DEFAULT_COEFFICIENT, 22222 => DEFAULT_COEFFICIENT};
                $sql_queries_gotten = [];
                @sql_query_results = $sub_get_sql_query_results->($date, $pages);

                my $pages_coef_gotten = $app->quality_coef->get_pages_coef(date => DEFAULT_DATE, pages => $pages);
                &is_deeply($pages_coef_gotten, $pages_coef_expected,
                    'QualityCoef->get_pages_coef returns right coefficients.');
                is_deeply(
                    $sql_queries_gotten,
                    $sub_get_sql_queries_expected->($date, $pages),
                    'Right sql queries are generated.'
                );
                use Test::Differences;

                my $sql_queries_expected = $sub_get_sql_queries_expected->($date, $pages);
                while (@$sql_queries_gotten) {
                    my $gotten   = shift @$sql_queries_gotten;
                    my $expected = shift @$sql_queries_expected;
                    while (@$gotten) {
                        my $gotten_item   = shift @$gotten;
                        my $expected_item = shift @$expected;
                        eq_or_diff($gotten_item, $expected_item);
                    }
                }
            }
        );
    }
);

$dbh->disconnect();
done_testing();
