#!/usr/bin/perl -w

use Test::More;
use Test::Deep;

use lib::abs qw(
  ./lib
  ../lib
  ../t_lib
  ../utils/lib
  );

use qbit;

use Test::Partner2::Simple;

use Application;
use Test::Partner::Utils qw(get_hash_values get_new_user get_new_rtb);

use Exception::Queue;

{
    no strict 'refs';
    no warnings 'redefine';
    *{'QBit::Application::check_rights'} = sub {TRUE};
}

run_tests(
    sub {
        my ($app) = @_;

        my $milestone_number = 10;
        my $product          = 'dsp';
        my %params           = (
            start    => '2014-01-01',
            end      => "2014-01-$milestone_number",
            products => [$product],
        );

        my $method_name = 'statistics_intake';

        subtest 'Queue action execution is correct.' => sub {
            $app->queue->add(
                method_name => $method_name,
                params      => \%params,
            );

            my $id = $app->queue->grab_next_task()->{id};
            ok($id, 'Task is added to queue');

            subtest 'Added task properties are correct.' => sub {
                my $task = $app->queue->get(
                    $id,
                    fields => [
                        qw(
                          method_name
                          milestone_number
                          params_ref
                          error_data
                          time_eta_ref
                          completed_percent
                          start_dt end_dt
                          )
                    ]
                );

                is($task->{'method_name'}, $method_name, 'Method name is correct.');
                is_deeply($task->{'params_ref'}, \%params, 'Params are correct.');
                is($task->{'error_data'},       undef,             'Error data is undefined.');
                is($task->{'milestone_number'}, $milestone_number, 'Milestone number is correct.');
                ok(defined($task->{'time_eta_ref'}),      'Time ETA is defined.');
                ok(defined($task->{'completed_percent'}), 'Completed percent is defined.');
                ok(defined($task->{'start_dt'}),          'Start date is defined.');
                ok(!defined($task->{'end_dt'}),           'End date is undefined.');
            };

            my $completed_percent = 0;

            subtest 'Started task properties are correct.' => sub {
                my $task =
                  $app->queue->get($id,
                    fields => [qw(method_name params_ref error_data time_eta_ref completed_percent start_dt end_dt)]);

                is($task->{'method_name'}, $method_name, 'Method name is correct.');
                is_deeply($task->{'params_ref'}, \%params, 'Params are correct.');
                is($task->{'error_data'}, undef, 'Error data is undefined.');
                ok(defined($task->{'time_eta_ref'}), 'Time ETA is defined.');
                is($task->{'completed_percent'}, $completed_percent, "Completed percent is $completed_percent.");
                ok(defined($task->{'start_dt'}), 'Start date is defined.');
                ok(!defined($task->{'end_dt'}),  'End date is undefined.');
            };

            $app->queue->do_action($id, 'pass_milestone');
            $completed_percent = 100 / $milestone_number * 1;

            subtest 'One milestone passed task properties are correct.' => sub {
                my $task =
                  $app->queue->get($id,
                    fields => [qw(method_name params_ref error_data time_eta_ref completed_percent start_dt end_dt)]);

                is($task->{'method_name'}, $method_name, 'Method name is correct.');
                is_deeply($task->{'params_ref'}, \%params, 'Params are correct.');
                is($task->{'error_data'}, undef, 'Error data is undefined.');
                ok(defined($task->{'time_eta_ref'}), 'Time ETA is defined.');
                is($task->{'completed_percent'}, $completed_percent, "Completed percent is $completed_percent.");
                ok(defined($task->{'start_dt'}), 'Start date is defined.');
                ok(!defined($task->{'end_dt'}),  'End date is undefined.');
            };

            my $error_data_ref = {date => '2014-01-02', level_name => $product};
            eval {throw Exception::Queue 'Breaken bad', $error_data_ref};
            my $exception = $@;

            $app->queue->do_action($id, 'need_restart', exception => $exception);

            subtest 'Finished with error task properties are correct.' => sub {
                my $task = $app->queue->get($id,
                    fields => [qw(method_name params_ref error_data time_eta_ref completed_percent log start_dt end_dt)]
                );

                is($task->{'method_name'}, $method_name, 'Method name is correct.');
                is_deeply($task->{'params_ref'}, \%params, 'Params are correct.');
                is($task->{'error_data'}, to_json($error_data_ref),
                    "Error data is '" . to_json($error_data_ref) . "'.");
                ok(defined($task->{'time_eta_ref'}), 'Time ETA is defined.');
                is($task->{'completed_percent'}, $completed_percent, "Completed percent is $completed_percent.");
                is($task->{'log'},               'Breaken bad',      "log is corrected.");
                ok(defined($task->{'start_dt'}),         'Start date is defined.');
                ok(!defined($task->{'end_dt'}),          'End date is not defined.');
                ok($task->{'time_eta_ref'}->is_paused(), 'Time ETA object is paused.');
            };

            $app->queue->restart_lost_tasks();
            $app->queue->do_action($id, 'start');

            subtest 'Resumed task properties are correct.' => sub {
                my $task = $app->queue->get($id,
                    fields => [qw(method_name params_ref error_data time_eta_ref completed_percent log start_dt end_dt)]
                );

                is($task->{'method_name'}, $method_name, 'Method name is correct.');
                is_deeply($task->{'params_ref'}, \%params, 'Params are correct.');
                is($task->{'error_data'}, undef, 'Error data is undefined.');
                ok(!defined($task->{'log'}),         'Log is undefined.');
                ok(!defined($task->{'end_dt'}),      'End date is undefined.');
                ok(defined($task->{'time_eta_ref'}), 'Time ETA is defined.');
                is($task->{'completed_percent'}, $completed_percent, "Completed percent is $completed_percent.");
                ok(defined($task->{'start_dt'}),          'Start date is defined.');
                ok(!$task->{'time_eta_ref'}->is_paused(), 'Time ETA object is resumed.');
            };

            $app->queue->do_action($id, 'pass_milestone') for (1 .. ($milestone_number - 1));
            $completed_percent = 100;

            subtest 'All milestones passed task properties are correct.' => sub {
                my $task =
                  $app->queue->get($id,
                    fields => [qw(method_name params_ref error_data time_eta_ref completed_percent start_dt end_dt)]);

                is($task->{'method_name'}, $method_name, 'Method name is correct.');
                is_deeply($task->{'params_ref'}, \%params, 'Params are correct.');
                is($task->{'error_data'}, undef, 'Error data is undefined.');
                ok(defined($task->{'time_eta_ref'}), 'Time ETA is defined.');
                is($task->{'completed_percent'}, $completed_percent, "Completed percent is $completed_percent.");
                ok(defined($task->{'start_dt'}), 'Start date is defined.');
                ok(!defined($task->{'end_dt'}),  'End date is undefined.');
            };

            $app->queue->do_action($id, 'finish_with_success');

            subtest 'Successive finished task properties are correct.' => sub {
                my $task =
                  $app->queue->get($id,
                    fields => [qw(method_name params_ref error_data time_eta_ref completed_percent start_dt end_dt)]);

                is($task->{'method_name'}, $method_name, 'Method name is correct.');
                is_deeply($task->{'params_ref'}, \%params, 'Params are correct.');
                is($task->{'error_data'}, undef, 'Error data is undefined.');
                ok(defined($task->{'time_eta_ref'}), 'Time ETA is defined.');
                is($task->{'completed_percent'}, $completed_percent, "Completed percent is $completed_percent.");
                ok(defined($task->{'start_dt'}), 'Start date is defined.');
                ok(defined($task->{'end_dt'}),   'End date is defined.');
            };
        };

        subtest 'Task execution is correct.' => sub {
            my $call_number = 0;

            $app->statistics_dsp;
            $app->{'statistics_dsp'} = Test::MockObject::Extends->new($app->{statistics_dsp});
            $app->statistics_dsp->mock(
                'update_statistics',
                sub {
                    my %opts = @_[1 .. $#_];

                    $call_number++;
                    die 'Breaking bad.' if $opts{'from'} eq '2014-01-06';
                }
            );

            $app->queue->add(
                method_name => $method_name,
                params      => \%params,
            );

            my $id = $app->queue->grab_next_task();
            $app->queue->start($id);

            subtest 'Breaked task properties are correct.' => sub {
                my $task = $app->queue->get($id, fields => [qw(error_data_ref completed_percent)]);

                is($task->{'completed_percent'}, 50, 'Task completed percent is correct after finish with error.');
                ok(defined($task->{'error_data_ref'}), 'Error data is defined.');
                is($call_number, 6, 'Update statistics method call number is correct.');
            };

            $app->statistics_dsp->mock('update_statistics', sub {$call_number++;});

            $app->queue->restart_lost_tasks();
            $id = $app->queue->grab_next_task();
            $app->queue->start($id);

            subtest 'Resumed and successfully finished task properties are correct.' => sub {
                my $task = $app->queue->get($id, fields => [qw(error_data_ref completed_percent)]);

                is($task->{'completed_percent'}, 100,   'Task completed percent is correct after finish with success.');
                is($task->{'error_data_ref'},    undef, 'Error data is undefined.');
                is($call_number,                 11,    'Update statistics method call number is correct.');
            };
        };

        subtest 'Two product task execution is correct.' => sub {
            my $call_number = 0;
            my $product_2   = 'dsp';

            $app->statistics_dsp;
            $app->{'statistics_dsp'} = Test::MockObject::Extends->new($app->{statistics_dsp});
            $app->statistics_dsp->mock('update_statistics', sub {$call_number++;});

            $app->statistics_dsp;
            $app->{'statistics_dsp'} = Test::MockObject::Extends->new($app->{statistics_dsp});
            $app->statistics_dsp->mock(
                'update_statistics',
                sub {
                    my ($self, %opts) = @_;

                    $call_number++;
                    die 'Breaking bad.' if $opts{'from'} eq '2014-01-06';
                }
            );

            my %params = (
                start    => '2014-01-01',
                end      => "2014-01-$milestone_number",
                products => [$product, $product_2],
            );

            $app->queue->add(
                method_name => $method_name,
                params      => \%params,
            );

            my $id = $app->queue->grab_next_task();
            $app->queue->start($id);

            subtest 'Breaked task properties are correct.' => sub {
                my $task = $app->queue->get($id, fields => [qw(error_data_ref completed_percent)]);

                is($task->{'completed_percent'}, 50, 'Task completed percent is correct after finish with error.');
                ok(defined($task->{'error_data_ref'}), 'Error data is defined.');
                is($call_number, 11, 'Update statistics method call number is correct.');
            };

            $app->statistics_dsp->mock('update_statistics', sub {$call_number++;});

            $app->queue->restart_lost_tasks();
            $id = $app->queue->grab_next_task();
            $app->queue->start($id);

            subtest 'Resumed and successfully finished task properties are correct.' => sub {
                my $task = $app->queue->get($id, fields => [qw(error_data_ref completed_percent log)]);

                is($task->{'completed_percent'}, 100,   'Task completed percent is correct after finish with success.');
                is($task->{'error_data_ref'},    undef, 'Error data is undefined.');
                is($call_number,                 21,    'Update statistics method call number is correct.');
            };
        };
    }
);
