#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use FindBin qw($Bin);
use lib "$Bin/lib";
use lib "$Bin/../lib";
use lib "$Bin/../utils/lib";

use Test::Most;
use Test::More;
use Test::Deep;
use Test::Partner::DB::Mock;
use Test::Partner::Utils qw(set_test_partner_utils_app get_new_user default_mocks);

use qbit;
use utf8;
use open qw(:std :utf8);

use Application;

die_on_fail();

my $app = Application->new();
$app->pre_run();
my @locales = keys %{$app->get_option('locales')};

set_test_partner_utils_app($app);

my $dbh = Test::Partner::DB::Mock->connect($app);

$dbh->mock($app);

$dbh->create_tables(
    partner_db => [
        qw(
          users
          user_features
          users_action_log
          user_role
          text_template
          text_template_action_log
          user_adfox

          context_on_site_campaign
          context_on_site_direct
          assistants
          search_on_site_campaign
          dsp
          video_stat_files
          context_on_site_stripe
          video_an_site
          internal_search_on_site_campaign
          managers
          internal_context_on_site_campaign
          ssp_context_on_site_campaign
          ssp_context_on_site_rtb
          ssp_mobile_app_settings
          ssp_mobile_app_rtb
          ssp_video_an_site
          ssp_video_an_site_rtb
          ssp_seller
          mobile_app_settings
          internal_mobile_app
          common_offer_allowed_users
          )
    ]
);

default_mocks();
# Разаработчик
$app->partner_db->_do(
q[INSERT INTO users(`id`, `uid`, `login`, `client_id`, `email`, `country_id`, `create_date`, `opts`) VALUES (1022, 1022, "yndx-developer", 110221, "partner2-backend@yandex-team.ru", 225, "000-00-00 00:00:00", '{"has_tutby_agreement":0, "has_common_offer":0, "has_rsya":1, "has_mobile_mediation":0, "has_approved":0}')]
);
$app->partner_db->_do('INSERT INTO user_role(`user_id`, `role_id`) VALUES (1022, 2)');

# Партнёр
$app->partner_db->_do(
q[INSERT INTO users(`id`, `uid`, `login`, `client_id`, `email`, `country_id`, `create_date`, `opts`) VALUES (1009, 1009, "mocked-yan-partner", 11009, "devnull@yandex.ru", 225, "000-00-00 00:00:00", '{"has_tutby_agreement":0, "has_common_offer":0, "has_rsya":1, "has_mobile_mediation":0, "has_approved":0}')]
);
$app->partner_db->_do('INSERT INTO user_role(`user_id`, `role_id`) VALUES (1009, 9)');

my $developer = {id => 1022};
my $partner   = {id => 1009};

ok(ref($app->text_template), 'text_template accessor works.');

subtest(
    'add' => sub {
        my %text_template = (
            key     => 'Template toolkit key',
            caption => 'Template toolkit caption',
            content => {map {$_ => "Template content $_"} @locales},
        );

        eval {$app->text_template->add()};
        is(ref($@), 'Exception::Denied', "Add throws exception with correct type when user hasn't the permission.");

        $app->{'__CURRENT_USER_RIGHTS__'} = {};
        $app->set_cur_user($developer);
        eval {$app->text_template->add()};
        is(
            ref($@),
            'Exception::Validation::BadArguments',
            "Add throws exception with correct type when no data provided."
          );

        subtest(
            'No caption in add' => sub {
                eval {$app->text_template->add(hash_transform(\%text_template, [qw(content)]));};
                is(ref($@), 'Exception::Validation::BadArguments', 'Exception is throwed.');
                is(
                    ref($@) ? $@->message() : $@,
                    gettext("Missed required field '%s'", 'caption'),
                    'Exception message is correct.'
                  );
            }
        );
        subtest(
            'No content in add' => sub {
                eval {$app->text_template->add(hash_transform(\%text_template, [qw(key caption)]));};
                is(ref($@), 'Exception::Validation::BadArguments', 'Exception is throwed.');
                is(
                    ref($@) ? $@->message() : $@,
                    gettext("Missed required field '%s'", 'content'),
                    'Exception message is correct.'
                  );
            }
        );
        subtest(
            'No ru content in add' => sub {
                eval {
                    $app->text_template->add(hash_transform(\%text_template, [qw(key caption)]),
                        content => {hash_transform($text_template{'content'}, [grep {$_ ne 'ru'} @locales])});
                };
                is(ref($@), 'Exception::Validation::BadArguments', 'Exception is throwed.');
                is(
                    ref($@) ? $@->message() : $@,
                    gettext("Missed field '%s' in locale '%s'", 'content', 'ru'),
                    'Exception message is correct.'
                  );
            }
        );

        subtest(
            'Breaken ru content in add' => sub {
                eval {
                    $app->text_template->add(%text_template,
                        content =>
                          {hash_transform($text_template{'content'}, [grep {$_ ne 'ru'} @locales]), ru => '[% =-= %]'});
                };
                is(ref($@), 'Exception::TextTemplate', 'Exception is throwed.');
            }
        );

        subtest(
            'No en content in add' => sub {
                eval {
                    $app->text_template->add(hash_transform(\%text_template, [qw(key caption)]),
                        content => {hash_transform($text_template{'content'}, [grep {$_ ne 'en'} @locales])});
                };
                is(ref($@), 'Exception::Validation::BadArguments', 'Exception is throwed.');
                is(
                    ref($@) ? $@->message() : $@,
                    gettext("Missed field '%s' in locale '%s'", 'content', 'en'),
                    'Exception message is correct.'
                  );
            }
        );

        subtest(
            'Empty string en content in add' => sub {
                eval {
                    $app->text_template->add(
                        hash_transform(\%text_template, [qw(key caption)]),
                        key     => $text_template{'key'} . ' other key tail',
                        content => {%{$text_template{'content'}}, en => ''}
                    );
                };
                is(ref($@), 'Exception::Validation::BadArguments', 'Exception is throwed.');
                is(
                    ref($@) ? $@->message() : $@,
                    gettext("Missed field '%s' in locale '%s'", 'content', 'en'),
                    'Exception message is correct.'
                  );
            }
        );

        subtest(
            'Breaken en content in add' => sub {
                eval {
                    $app->text_template->add(%text_template,
                        content =>
                          {hash_transform($text_template{'content'}, [grep {$_ ne 'en'} @locales]), en => '[% =-= %]'});
                };
                is(ref($@), 'Exception::TextTemplate', 'Exception is throwed.');
            }
        );

        $text_template{'id'} = $app->text_template->add(%text_template);
        is($text_template{'id'}, 1, 'text_template->add returned correct template toolkit id.');

        cmp_deeply(
            $app->text_template->get_all(fields => [qw(id key caption content)]),
            [
                {
                    id      => $text_template{'id'},
                    key     => $text_template{'key'},
                    caption => $text_template{'caption'},
                    content => $text_template{'content'}{'ru'},
                }
            ],
            'The template toolkit is added.'
        );

        subtest(
            'No key in add' => sub {
                eval {$app->text_template->add(hash_transform(\%text_template, [qw(caption content)]),);};
                is(ref($@), '', 'Exception is not throwed.');
            }
        );

    }
);

subtest(
    'edit' => sub {
        my %text_template = (
            key     => 'Template toolkit key 2',
            caption => 'Template toolkit caption',
            content => {map {$_ => "Template content $_"} @locales},
        );

        $text_template{'id'} = $app->text_template->add(%text_template);

        $app->{'__CURRENT_USER_RIGHTS__'} = {};
        $app->set_cur_user($partner);
        cmp_deeply(
            $app->text_template->get($text_template{'id'}, fields => [qw(actions)]),
            {actions => {}},
            'Actions are not available without rights.'
        );

        $app->set_cur_user($developer);
        cmp_deeply(
            $app->text_template->get($text_template{'id'}, fields => [qw(actions)]),
            {actions => {map {$_ => ignore()} qw(edit)}},
            'edit action is given by right.'
        );

        $app->text_template->do_action($text_template{'id'}, 'edit', key => 'New template key');
        cmp_deeply(
            $app->text_template->get($text_template{'id'}, fields => [qw(id key caption content actions)]),
            {
                id      => $text_template{'id'},
                key     => $text_template{'key'},
                caption => $text_template{'caption'},
                content => $text_template{'content'}{'ru'},
                actions => {map {$_ => ignore()} qw(edit)}
            },
            'The template key is edited.'
        );

        $text_template{caption} = 'New template caption';
        $app->text_template->do_action($text_template{'id'}, 'edit', hash_transform(\%text_template, [qw(caption)]));
        cmp_deeply(
            $app->text_template->get($text_template{'id'}, fields => [qw(id key caption content actions)]),
            {
                id      => $text_template{'id'},
                key     => $text_template{'key'},
                caption => $text_template{'caption'},
                content => $text_template{'content'}{'ru'},
                actions => {map {$_ => ignore()} qw(edit)}
            },
            'The template caption is edited.'
        );

        $text_template{'content'}{'ru'} = 'New content content ru';
        $app->text_template->do_action($text_template{'id'}, 'edit', hash_transform(\%text_template, [qw(content)]));
        cmp_deeply(
            $app->text_template->get($text_template{'id'}, fields => [qw(id key caption content actions)]),
            {
                id      => $text_template{'id'},
                key     => $text_template{'key'},
                caption => $text_template{'caption'},
                content => $text_template{'content'}{'ru'},
                actions => {map {$_ => ignore()} qw(edit)}
            },
            'The template content is edited.'
        );

        subtest(
            'edit key to undef' => sub {
                eval {$app->text_template->do_action($text_template{'id'}, 'edit', key => undef)};
                is(ref($@), '', 'Exception is not throwed.');
            }
        );
        subtest(
            'edit caption to empty string' => sub {
                eval {$app->text_template->do_action($text_template{'id'}, 'edit', caption => '')};
                is(ref($@), '', 'Exception is not throwed.');
            }
        );
        subtest(
            'edit content to undef' => sub {
                eval {$app->text_template->do_action($text_template{'id'}, 'edit', content => undef)};
                is(ref($@), 'Exception::Validation::BadArguments::EmptyValue', 'Exception is throwed.');
                is(
                    ref($@) ? $@->message() : $@,
                    gettext("Missed required field '%s'", 'content'),
                    'Exception message is correct.'
                  );
            }
        );
        subtest(
            'edit content ru to undef' => sub {
                eval {
                    $app->text_template->do_action($text_template{'id'}, 'edit',
                        content =>
                          {ru => undef, hash_transform($text_template{'content'}, [grep {$_ ne 'ru'} @locales])});
                };
                is(ref($@), 'Exception::Validation::BadArguments', 'Exception is throwed.');
                is(
                    ref($@) ? $@->message() : $@,
                    gettext("Missed field '%s' in locale '%s'", 'content', 'ru'),
                    'Exception message is correct.'
                  );
            }
        );
        subtest(
            'edit content ru to empty string' => sub {
                eval {
                    $app->text_template->do_action($text_template{'id'}, 'edit',
                        content => {ru => '', hash_transform($text_template{'content'}, [grep {$_ ne 'ru'} @locales])});
                };
                is(ref($@), 'Exception::Validation::BadArguments', 'Exception is throwed.');
                is(
                    ref($@) ? $@->message() : $@,
                    gettext("Missed field '%s' in locale '%s'", 'content', 'ru'),
                    'Exception message is correct.'
                  );
            }
        );
        subtest(
            'edit content en to undef' => sub {
                eval {
                    $app->text_template->do_action($text_template{'id'}, 'edit',
                        content =>
                          {en => undef, hash_transform($text_template{'content'}, [grep {$_ ne 'en'} @locales])});
                };
                is(ref($@), 'Exception::Validation::BadArguments', 'Exception is throwed.');
                is(
                    ref($@) ? $@->message() : $@,
                    gettext("Missed field '%s' in locale '%s'", 'content', 'en'),
                    'Exception message is correct.'
                  );
            }
        );
        subtest(
            'edit content en to empty string' => sub {
                eval {
                    $app->text_template->do_action($text_template{'id'}, 'edit',
                        content => {en => '', hash_transform($text_template{'content'}, [grep {$_ ne 'en'} @locales])});
                };
                is(ref($@), 'Exception::Validation::BadArguments', 'Exception is throwed.');
                is(
                    ref($@) ? $@->message() : $@,
                    gettext("Missed field '%s' in locale '%s'", 'content', 'en'),
                    'Exception message is correct.'
                  );
            }
        );

        subtest(
            'Breaken ru content in editing' => sub {
                eval {
                    $app->text_template->do_action($text_template{'id'}, 'edit',
                        content => {%{$text_template{'content'}}, ru => '[% =-= %]'});
                };
                is(ref($@), 'Exception::TextTemplate', 'Exception is throwed.');
            }
        );
    }
);

subtest(
    'process_content' => sub {
        my %text_template = (
            key     => 'Template toolkit key 3',
            caption => 'Template toolkit caption',
            content => {map {$_ => "Template content [% variable %] $_"} @locales},
        );

        $text_template{'id'} = $app->text_template->add(%text_template);

        subtest(
            'No key given' => sub {
                eval {$app->text_template->process_content();};
                is(ref($@), 'Exception::Validation::BadArguments', 'Exception is throwed.');
                is(
                    ref($@) ? $@->message() : $@,
                    gettext("Missed required field '%s'", 'key'),
                    'Exception message is correct.'
                  );
            }
        );

        subtest(
            'Not existing key given' => sub {
                eval {$app->text_template->process_content('This key does not exist');};
                is(ref($@), 'Exception::Validation::BadArguments', 'Exception is throwed.');
                is(
                    ref($@) ? $@->message() : $@,
                    gettext('Template with key "This key does not exist" does not exist.'),
                    'Exception message is correct.'
                  );
            }
        );

        my $variable = 'is here';
        is(
            $app->text_template->process_content($text_template{'key'}, {variable => $variable}),
            "Template content $variable ru",
            'Correct template processing result is gotten.'
          );
    }
);

done_testing();
exit;
