use strict;
use warnings FATAL => 'all';
use utf8;
use open qw(:std :utf8);

use Test::Differences;
use Test::More;
use Path::Tiny;
use Cpanel::JSON::XS;
use Encode;

use Data::Dumper;

BEGIN {
    $ENV{FORM_PARTNER2_SERVER} = 'mocked_partner2_hostname';
}

use Partner2;

sub mock_partner2 {
    my ($user_id) = @_;
    no warnings 'redefine';

    *Partner2::list_attachments = sub {
        return [
            "partner_form_${user_id}_checksum1_timestamp1", "partner_form_${user_id}_checksum2_timestamp2",
            "partner_form_${user_id}_checksum3_timestamp3",
        ];
    };

    *Partner2::query_fias = sub {
        my (%opts) = @_;
        my %fias = (
            'bf400e6b-ff57-4764-a176-fbbbdde472ca' => {obj_level => 7,},
            'c2deb16a-0330-4f05-821f-1d09c93331e6' => {obj_level => 4,},
        );
        return [$fias{$opts{guid} // 'unknown'} // (),];
    };

    *Partner2::get_bank_by_bik = sub {
        my ($bik) = @_;

        if ($bik eq '044525700') {
            return {
                found => 1,
                info  => {
                    bank_address => undef,
                    bank_city    => 'МОСКВА',
                    bank_name    => 'АО "РАЙФФАЙЗЕНБАНК"',
                    bik          => '044525700',
                    city         => 'МОСКВА',
                    cor_acc      => 30101810200000000700,
                    corr_account => 30101810200000000700,
                    hidden       => 0,
                    id           => 1120,
                    info         => undef,
                    name         => 'АО "РАЙФФАЙЗЕНБАНК"',
                    swift        => 'RZBMRUMMXXX',
                    update_dt    => '20180625T04:00:49',
                },
            };
        } elsif ($bik eq '044525225') {
            return {
                found => 1,
                info  => {
                    bank_address => undef,
                    bank_city    => 'МОСКВА',
                    bank_name    => 'ПАО СБЕРБАНК',
                    bik          => '044525225',
                    city         => 'МОСКВА',
                    cor_acc      => 30101810400000000225,
                    corr_account => 30101810400000000225,
                    hidden       => 0,
                    id           => 1103,
                    info         => undef,
                    name         => 'ПАО СБЕРБАНК',
                    swift        => 'SABRRUMMXXX',
                    update_dt    => '20180625T04:00:49',
                },
            };
        } elsif ($bik eq '123123123') {
            return {found => 0,};
        } else {
            die "Need to mock data for bik '$bik'";
        }

        return 1;
    };

    *Partner2::get_bank_by_swift = sub {
        my ($swift) = @_;

        if ($swift eq 'INGBNL2A') {
            return {
                found => 1,
                info  => {
                    address => 'BIJLMERDREEF 106',
                    bicint  => 'INGBNL2AXXX',
                    country => 'NETHERLANDS',
                    name    => 'ING BANK N.V.',
                    place   => 'AMSTERDAM',
                    zipcode => '1102 CT AMSTERDAM',
                },
            };
        } elsif ($swift eq 'MOMENTUMXXX' || $swift eq 'SPDDCNSH630') {
            return {found => 0,};
        } else {
            die "Need to mock data for swift '$swift'";
        }

        return 1;
    };

    *Partner2::is_yamoney_account_identificated = sub {
        my (%opts) = @_;
        if ($opts{account} == 4100143322850) {
            return 1;
        } elsif ($opts{account} == 4100143322840) {
            return 0;
        } else {
            die "Need to mock data for yandex money account '$opts{account}'";
        }

        return;
    };

    *Partner2::check_pi_adfox_contracts = sub {
        my (%opts) = @_;
        eq_or_diff(
            [sort keys(%opts)],
            [
                qw(
                  adfox_login
                  adfox_password
                  allow_paid_services
                  branch_id
                  lang
                  user_id
                  )
            ],
            'opts keys for check_pi_adfox_contracts'
        );
        if ($opts{adfox_login} eq 'good-adfox-login') {
            return {contracts => ['EXTERNAL_CONTRACT_ID_1',],};
        } elsif ($opts{adfox_login} eq 'bad-adfox-login') {
            return {error_message => 'Bad login',};
        } else {
            die 'Unexpected case';
        }
    };

    *Partner2::check_bik_and_account = sub {
        my ($bik, $account) = @_;
        if ($bik eq '044525225' && $account eq '30101810400000000225') {
            return 0;
        }
        return 1;
    };

    return 1;
}

sub main_in_test {

    my $mocked_user_id = 123;

    mock_partner2($mocked_user_id);

    my %jsons;
    @jsons{glob './api/t_data/fields/*'} = undef;
    foreach my $file_name (glob './api/lib/Field/*') {
        next if $file_name =~ m|/Groups$|;
        my ($field_id) = $file_name =~ /^.*\/(.*)\.pm\z/;

        next if $ARGV[0] && $ARGV[0] ne $field_id;

        my $json_file_name = sprintf('./api/t_data/fields/%s.json', $field_id);
        delete $jsons{$json_file_name};
        my $test_data = decode_json encode_utf8 path($json_file_name)->slurp_utf8;
        fail(sprintf('Empty test data for "%s"', $field_id))
          if (ref($test_data) ne 'HASH' && ref($test_data) ne 'ARRAY');
        $test_data = [$test_data] if (ref($test_data) ne 'ARRAY');

        require "Field/${field_id}.pm";
        my $p = "Field::$field_id";
        my $f = $p->new(language => 'ru');
        is($f->_get_id(), $field_id, 'correct field id');

        my $cjx = Cpanel::JSON::XS->new->allow_nonref;

        foreach my $test_set (@$test_data) {
            foreach my $value (@{$test_set->{valid}}) {
                ok(
                    $f->is_valid(
                        $value,
                        {
                            user_id      => $mocked_user_id,
                            dependencies => $test_set->{dependencies},
                        }
                      )->{is_valid},
                    sprintf('in field "%s" value %s is valid', $field_id, $cjx->encode($value))
                  );
            }
            foreach my $value (@{$test_set->{not_valid}}) {
                ok(
                    !$f->is_valid(
                        $value,
                        {
                            user_id      => $mocked_user_id,
                            dependencies => $test_set->{dependencies},
                        }
                      )->{is_valid},
                    sprintf('in field "%s" value %s is not valid', $field_id, $cjx->encode($value))
                  );
            }
        }
    }

    eq_or_diff([keys %jsons], [], 'No extra jsons') unless $ARGV[0];

    done_testing();
}
main_in_test();
