#!/usr/bin/perl

use strict;
use warnings FATAL => 'all';
use utf8;
use open qw(:std :utf8);

use Test::Differences;
use Test::Exception;
use Test::More;

use Branches;

my %TEST_DATA = (
    1 => {
        ag_ph           => undef,
        ag_ur           => undef,
        belarus_ip      => 'BYN',
        belarus_ph      => undef,
        belarus_ur      => 'BYN',
        russia_ip       => 'RUB',
        russia_ph_part1 => 'RUB',
        russia_ph_part2 => 'RUB',
        russia_ur       => 'RUB',
        ukraine_ph      => '',
    },
    2 => {
        ag_ph_part1      => undef,
        ag_ph_part2      => undef,
        ag_ur_part1      => undef,
        ag_ur_part2      => undef,
        belarus_ip_part1 => 'BYN',
        belarus_ip_part2 => 'BYN',
        belarus_ph_part1 => undef,
        belarus_ph_part2 => undef,
        belarus_ur_part1 => 'BYN',
        belarus_ur_part2 => 'BYN',
        russia_ip_part1  => 'RUB',
        russia_ip_part2  => 'RUB',
        russia_ph_part1  => 'RUB',
        russia_ph_part2  => 'RUB',
        russia_ur_part1  => 'RUB',
        russia_ur_part2  => 'RUB',
        ukraine_ph_part1 => '',
        ukraine_ph_part2 => '',
    },
);

sub main_in_test {
    plan tests => scalar keys %TEST_DATA;

    foreach my $version (sort keys %TEST_DATA) {
        subtest "version $version" => sub {
            plan tests => 5 + scalar(keys %{$TEST_DATA{$version}});

            my $branches = Branches->new(version => $version);

            throws_ok {
                my $bool = $branches->get_balance_contract_currency();
            }
            qr/Must specify branch_id/, 'dies in case branch_id is not specified';

            throws_ok {
                my $bool = $branches->get_balance_contract_currency('XXX');
            }
            qr/Unknown branch_id XXX/, 'dies in case of unknown branch_id';

            eq_or_diff($branches->get_all_branches(), [sort keys %{$TEST_DATA{$version}}], 'all branches');

            foreach my $branch_id (sort keys %{$TEST_DATA{$version}}) {
                my $currency = $branches->get_balance_contract_currency($branch_id);
                unless (defined($TEST_DATA{$version}{$branch_id})) {
                    subtest $branch_id => sub {
                        my $is_part2 = $branches->is_branch_part2($branch_id);
                        plan tests => ($is_part2 ? 2 : 3);
                        is($currency, undef, sprintf('currency is not specified (%s)', $branch_id));
                        ok(scalar(grep {$_ eq 'currency'} $branches->get_fields($branch_id)), 'has currency field')
                          unless $is_part2;
                        ok(scalar(grep {$_ eq 'currency'} values %{$branches->get_converter_form($branch_id)}),
                            'has currency converter');
                    };
                } elsif ($TEST_DATA{$version}{$branch_id}) {
                    subtest $branch_id => sub {
                        plan tests => 1;
                        is(
                            $currency,
                            $TEST_DATA{$version}{$branch_id},
                            sprintf('currency is %s', $TEST_DATA{$version}{$branch_id})
                          );
                    };
                } else {
                    subtest $branch_id => sub {
                        plan tests => 1;
                        is($currency, undef, 'currency is not specified');
                    };
                }
            }

            # specific branches
            my $specific_branches = Branches->new(project => 'assessor', version => $version);
            eq_or_diff($specific_branches->get_all_branches(), ['assessor'], 'all specific branches');
            foreach my $branch_id (qw(assessor)) {
                is($specific_branches->get_balance_contract_currency($branch_id),
                    'RUB', "currency is not specified ($branch_id)");
            }
        };
    }
}
main_in_test();
