use strict;
use warnings FATAL => 'all';
use utf8;
use open qw(:std :utf8);

use Test::More;
use Test::Exception;

BEGIN {
    $ENV{FORM_PARTNER2_SERVER} = 'mocked_partner2_hostname';
}

use Partner2;

use Cpanel::JSON::XS;
use URI;
use URI::QueryParam;

my %cases = (
    adfox_contracts_are_ok => {
        args => {
            user_id             => 123,
            lang                => 'ru',
            adfox_login         => 'good_adfox_login',
            adfox_password      => 'good_adfox_password',
            allow_paid_services => '',
        },
        response => {
            reason  => 'OK',
            success => 1,
            status  => 200,
            content => {
                result => 'ok',
                data   => {contracts => ["EXTERNAL_CONTRACT_ID_1",],},
            },
        },
    },
    problem_with_adfox_contracts => {
        args => {
            user_id             => 321,
            lang                => 'ru',
            adfox_login         => 'плохой логин',
            adfox_password      => 'плохой пароль',
            allow_paid_services => '',
        },
        response => {
            reason  => 'OK',
            success => 1,
            status  => 200,
            content => {
                result => 'ok',
                data   => {error_message => 'Something is wrong with your adfox account',},
            },
        }
    },
    wrong_response_content => {
        args => {
            user_id             => 321,
            lang                => 'ru',
            adfox_login         => 'login',
            adfox_password      => 'pass',
            allow_paid_services => '',
        },
        response => {
            reason  => 'OK',
            success => 1,
            status  => 200,
            content => {
                result => 'ok',
                data   => {some_irrelevant_key => 'We forgot to send you contracts or error_message',},
            },
        },
        dies => 1,
    },
    internal_error => {
        reason  => 'Internal Error',
        success => 0,
        status  => 500,
        content => {
            "error_type" => "Internal",
            "message"    => "Internal Error",
            "result"     => "error"
        },
        dies => 1,
    },
    no_user_id => {
        args => {
            lang                => 'ru',
            adfox_login         => 'good_adfox_login',
            adfox_password      => 'good_adfox_password',
            allow_paid_services => '',
        },
        dies => 1,
    },
    no_lang => {
        args => {
            user_id             => 123,
            adfox_login         => 'good_adfox_login',
            adfox_password      => 'good_adfox_password',
            allow_paid_services => '',
        },
        dies => 1,
    },
    no_adfox_login => {
        args => {
            user_id             => 123,
            lang                => 'ru',
            adfox_password      => 'good_adfox_password',
            allow_paid_services => '',
        },
        dies => 1,
    },
    no_adfox_password => {
        args => {
            user_id             => 123,
            lang                => 'ru',
            adfox_login         => 'good_adfox_login',
            allow_paid_services => '',
        },
        dies => 1,
    },
    no_adfox_password => {
        args => {
            user_id        => 123,
            lang           => 'ru',
            adfox_login    => 'good_adfox_login',
            adfox_password => 'good_adfox_password',
        },
        dies => 1,
    },
);

sub main_in_test {

    my $case;
    {
        no warnings 'redefine';

        *HTTP::Tiny::request = sub {
            my ($self, $method, $url, $args) = @_;

            my $uri  = URI->new($url);
            my $body = decode_json $args->{content};
            my $lang = delete $case->{args}{lang};

            is($method, 'POST', 'HTTP::Tiny::request method');
            is(
                $uri->path,
                'mocked_partner2_hostname/intapi/form/check_pi_adfox_contracts.json',
                'HTTP::Tiny::request path'
              );
            is_deeply($body, {%{$case->{args}}, from_offer => 0}, 'HTTP::Tiny::request content');
            is($args->{headers}{'Accept-Language'}, $lang, 'HTTP::Tiny::request Accept-Language');

            return {
                'headers' => {
                    'server'            => 'nginx',
                    'date'              => 'Mon, 26 Mar 2018 10:48:37 GMT',
                    'transfer-encoding' => 'chunked',
                    'content-type'      => 'application/json; charset=UTF-8',
                    'connection'        => 'keep-alive'
                },
                'protocol' => 'HTTP/1.1',
                'url'      => $url,
                %{$case->{response}},
                'content' => Cpanel::JSON::XS->new->utf8->canonical->encode($case->{response}{content}),
            };
        };
    }

    for my $case_name (sort keys(%cases)) {
        subtest $case_name => sub {
            $case = $cases{$case_name};
            if ($case->{dies}) {
                dies_ok {
                    check_pi_adfox_contracts(%{$case->{args}});
                }
                'dies';
            } else {
                lives_ok {
                    is_deeply(
                        check_pi_adfox_contracts(%{$case->{args}}),
                        $case->{response}{content}{data},
                        'return value'
                    );
                }
                'lives';
            }
        };
    }

    done_testing();
}

main_in_test();
