use strict;
use warnings FATAL => 'all';
use utf8;
use open qw(:std :utf8);

use Test::More;
use Test::Exception;

use URI::Escape;

BEGIN {
    $ENV{FORM_PARTNER2_SERVER} = 'mocked_partner2_hostname';
}

use Partner2;

use Data::Dumper;
use Cpanel::JSON::XS;

my %cases = (
    account_identificated => {
        opts => {
            account    => '1234567890',
            lastname   => 'identificated',
            firstname  => 'account',
            middlename => 'credentials',
            passport   => '123456',
        },
        response => {
            reason  => 'OK',
            success => 1,
            status  => 200,
            content => {
                result => 'ok',
                data   => {is_identificated => 1,},
            },
        }
    },
    account_not_identifcated => {
        # no content needed. dies before http request
        opts => {
            account    => '1234567890',
            lastname   => 'not identificated',
            firstname  => 'account',
            middlename => 'credentials',
            passport   => '123456',
        },
        response => {
            reason  => 'OK',
            success => 1,
            status  => 200,
            content => {
                result => 'ok',
                data   => {is_identificated => 0,},
            },
        }
    },
    missing_opts => {
        opts     => {},
        response => {},
        dies     => 1,
    },
    internal_error => {
        opts => {
            account    => '1234567890',
            lastname   => 'error',
            firstname  => 'account',
            middlename => 'credentials',
            passport   => '123456',
        },
        response => {
            reason  => 'Internal Error',
            success => 0,
            status  => 500,
            content => {
                "error_type" => "Internal",
                "message"    => "Internal Error",
                "result"     => "error"
            },
        },
        dies => 1,
    },
);

sub main_in_test {

    my $case;
    {
        no warnings 'redefine';

        *HTTP::Tiny::request = sub {
            my ($self, $method, $url, $args) = @_;

            is($method, 'GET', 'HTTP::Tiny::request method');
            my $query_args = join('&',
                map {$_ . '=' . uri_escape_utf8($case->{opts}{$_})}
                  (qw(account firstname lastname middlename passport)));
            is(
                $url,
                "mocked_partner2_hostname/intapi/form/is_yamoney_account_identificated.json?lang=en&$query_args",
                'HTTP::Tiny::request url'
              );
            is_deeply($args, {}, 'HTTP::Tiny::request args');

            return {
                'headers' => {
                    'server'            => 'nginx',
                    'date'              => 'Mon, 26 Mar 2018 10:48:37 GMT',
                    'transfer-encoding' => 'chunked',
                    'content-type'      => 'application/json; charset=UTF-8',
                    'connection'        => 'keep-alive'
                },
                'protocol' => 'HTTP/1.1',
                'url'      => $url,
                %{$case->{response}},
                'content' => Cpanel::JSON::XS->new->utf8->canonical->encode($case->{response}{content}),
            };
        };
    }

    for my $case_name (sort keys(%cases)) {
        subtest $case_name => sub {
            $case = $cases{$case_name};
            if ($case->{dies}) {
                dies_ok {
                    is_yamoney_account_identificated(%{$case->{opts}});
                }
                'dies';
            } else {
                lives_ok {
                    is(
                        is_yamoney_account_identificated(%{$case->{opts}}),
                        $case->{response}{content}{data}{is_identificated},
                        'return value'
                      );
                }
                'lives';
            }
        };
    }

    done_testing();
}

main_in_test();
