use strict;
use warnings FATAL => 'all';

use utf8;
use open qw(:std :utf8);

use Carp;
use Cpanel::JSON::XS;
use Test::More;
use FormConstants qw($PARTNER2_CRON_MAIL);

BEGIN {
    $ENV{'FORM_PARTNER2_SERVER'} = 'mocked_partner2_hostname';
}

use Submit;

my $branch_id  = 'test';
my $country_id = 225;
my $user_id    = 607369863;
my $user_login = 'Test Login';
my $client_id  = 9876543210;

sub mock_all {
    no strict 'refs';
    no warnings 'redefine';

    *{'Branches::branch_ids2country_ids'}       = sub {return +{$branch_id => [$country_id,],}};
    *{'Branches::get_converter_form'}           = sub {return +{}};
    *{'Branches::get_converter_partner2'}       = sub {return +{}};
    *{'Branches::get_fields'}                   = sub { };
    *{'Branches::is_oferta'}                    = sub {return undef};
    *{'Branches::is_valid_branch_id'}           = sub {return 1};
    *{'Branches::working_with_balance'}         = sub {return undef};
    *{'Branches::working_with_contract'}        = sub {return undef};
    *{'Countries::is_valid_working_country_id'} = sub {return 1};
    *{'JSV::Validator::validate'}               = sub {return 1};
    *{'Submit::create_offer'}                   = sub { };
    *{'Submit::create_or_update_partner'}       = sub { };
    *{'Submit::get_client_id'}                  = sub {return $client_id};
    *{'Submit::get_partner2_user_data'}         = sub {
        return +{
            'db' => {
                'form_data' => {
                    'data' => encode_json(
                        {
                            'fields'   => {},
                            'language' => 'ru',
                        }
                    ),
                },
            },
        };
    };
    *{'Submit::save_data'} = sub { };
    *{'Submit::send_email'} = sub {
        is_deeply(
            \@_,
            ['bcc', $PARTNER2_CRON_MAIL, 'user_id', $user_id, 'template', 'CDQDUR54-VZD1', 'transport', 'sender'],
            'check send_email'
        );
    };
}

sub normal_get_client_id {
    plan tests => 4;

    no strict 'refs';
    no warnings 'redefine';

    mock_all();
    local *{'Submit::save_data'} = sub {
        my ($opts) = @_;
        is($$opts{'client_id'}, $client_id, 'save_data client_id');
    };

    my $s = Submit->new(
        {
            'branch_id'     => $branch_id,
            'country_id'    => $country_id,
            'user_id'       => $user_id,
            'display_login' => $user_login,
        },
        'ru',
        'default',
        1,
    );
    my ($ok, $err) = $s->submit();
    is($ok, 1, 'valid submit');
    is_deeply($err, {}, 'Error structure');
}

sub error_get_client_id {
    plan tests => 2;

    no strict 'refs';
    no warnings 'redefine';

    mock_all();
    *{'Submit::get_client_id'} = sub {return (undef, 1)};

    my $s = Submit->new(
        {
            'branch_id'     => $branch_id,
            'country_id'    => $country_id,
            'user_id'       => $user_id,
            'display_login' => $user_login,
        },
        'ru',
    );
    my ($ok, $err) = $s->submit();
    is($ok, 0, 'valid submit');
    is_deeply(
        $err,
        {
            'global' => [
                {
                    'description' =>
'Пользователь с логином Test Login не может быть зарегистрирован. Пожалуйста, выберите другой логин',
                }
            ],
        },
        'Error structure'
    );
}

sub exception_get_client_id {
    plan tests => 2;

    no strict 'refs';
    no warnings 'redefine';

    mock_all();
    *{'Submit::get_client_id'} = sub {croak 'test exception'};

    my $s = Submit->new(
        {
            'branch_id'  => $branch_id,
            'country_id' => $country_id,
            'user_id'    => $user_id,
        },
        'ru',
    );
    ok(
        !eval {
            $s->submit();
            return 1;
        },
        'Catch exception'
      );
    ok($@ =~ /^test exception/, 'Exception message');
}

sub check_get_client_id {
    plan tests => 3;

    subtest 'normal'    => \&normal_get_client_id;
    subtest 'error'     => \&error_get_client_id;
    subtest 'exception' => \&exception_get_client_id;
}

sub main_in_test {
    plan tests => 1;

    subtest 'Check behaviour with get_client_id' => \&check_get_client_id;
}

main_in_test();
