use strict;
use warnings FATAL => 'all';
use utf8;
use open qw(:std :utf8);

use List::Util;
use Test::More;
use Utils;
use Data::Dumper;

sub all_cases {
    my %croaks;
    @croaks{
        'no branch_id',
        'no canonical_login',
        'no language',
        'no partner2_db',
        'no can_fill_part2',
        'no representative',
        'no version',
        'Must specify language',
        'unknown opts',
        'impossible',
      }
      = ();
    my @names = qw( branch_id canonical_login can_fill_part2 language partner2_db representative version extra_field );
    my %sets  = (
        branch_id       => ['__EMPTY__', 'any branch', 'any_branch_part2', 'any_branch_part1',],
        canonical_login => ['__EMPTY__', '',           'some-login',       'yndx-login',],
        can_fill_part2  => ['__EMPTY__', '',           1,],
        language        => ['__EMPTY__', '',           'ru',               'en',],
        partner2_db     => [
            '__EMPTY__', undef,
            {form_data => {branch_id => '',},           user => {},},
            {form_data => {branch_id => 'any branch',}, user => {},},
        ],
        representative => ['__EMPTY__', undef, 1,],
        version        => ['__EMPTY__', 1,],
        extra_field    => ['__EMPTY__', undef, 1,],
    );
    my %results;

    my $iterator = sub {
        my ($cb) = @_;

        if (ref($cb) eq 'CODE') {
            my @vector = (0) x scalar @names;

            while (scalar @vector == scalar @names) {
                &$cb(@vector);

                my $i = 0;
                $vector[$i]++;
                while ($i < scalar @names && $vector[$i] > $#{$sets{$names[$i]}}) {
                    $vector[$i] = 0;
                    $i++;
                    $vector[$i]++;
                }
            }
        }
    };

    my $json_coder = JSON::PP->new->pretty(0)->canonical(1);

    my $val2name = sub {
        my (@vals) = @_;

        return
          join(', ', map {defined($_) ? (ref($_) ? $json_coder->encode($_) : ($_ eq '' ? '""' : $_)) : 'undef'} @vals);
    };

    my $vec2name = sub {
        my (@vector) = @_;

        return &$val2name(map {$sets{$names[$_]}[$vector[$_]]} 0 .. $#vector);
    };

    my $fill = sub {
        my (@vector) = @_;

        my $name  = &$vec2name(@vector);
        my $extra = pop @vector;
        if ($extra) {    # Лишнее поле не '__EMPTY__'
            $results{$name} = 'croak';    # Лишние поля не допустимы
        } elsif (
            grep {
                !$_
            } @vector
          )
        {                                 # 0 = '__EMPTY__'
            $results{$name} = 'croak';    # Поля обязательны
        } elsif ($vector[2] == 2 && $vector[4] == 1) {    # can_fill_part2 && !partner2_db->{user}
            $results{$name} = 'croak'
              ; # Пользователя нет, но почему-то разрешено заполнять вторую часть анкеты
        } elsif ($vector[3] == 1) {    # неизвестный язык
            $results{$name} = 'croak';
        } else {
            $results{$name} = 0;       # По умолчанию запрещено
        }
    };

    &$iterator($fill);

    # Новый пользователь с разрешённым логином и которого ещё нет
    $results{&$val2name('any_branch_part1', 'some-login', '', 'ru', undef, undef, 1, '__EMPTY__',)} = 1;
    $results{&$val2name('any_branch_part1', 'some-login', '', 'en', undef, undef, 1, '__EMPTY__',)} = 1;
    $results{&$val2name('any branch',       'some-login', '', 'ru', undef, undef, 1, '__EMPTY__',)} = 1;
    $results{&$val2name('any branch',       'some-login', '', 'en', undef, undef, 1, '__EMPTY__',)} = 1;

    # Пользователь, который уже есть, которому разрешено заполнять вторую часть анкеты и он пытается это сделать
    $results{
        &$val2name('any_branch_part2', 'some-login', 1, 'ru', {form_data => {branch_id => '',}, user => {},},
            undef, 1, '__EMPTY__',)
      }
      = 1;
    $results{
        &$val2name('any_branch_part2', 'some-login', 1, 'en', {form_data => {branch_id => '',}, user => {},},
            undef, 1, '__EMPTY__',)
      }
      = 1;
    $results{
        &$val2name('any_branch_part2', 'some-login', 1, 'ru', {form_data => {branch_id => 'any branch',}, user => {},},
            undef, 1, '__EMPTY__',)
      }
      = 1;
    $results{
        &$val2name('any_branch_part2', 'some-login', 1, 'en', {form_data => {branch_id => 'any branch',}, user => {},},
            undef, 1, '__EMPTY__',)
      }
      = 1;

    # Пользователь, который уже есть, которому разрешено заполнять вторую часть анкеты, а он снова пытается заполнить первую
    $results{
        &$val2name('any_branch_part1', 'some-login', 1, 'ru', {form_data => {branch_id => '',}, user => {},},
            undef, 1, '__EMPTY__',)
      }
      = -1;
    $results{
        &$val2name('any_branch_part1', 'some-login', 1, 'en', {form_data => {branch_id => '',}, user => {},},
            undef, 1, '__EMPTY__',)
      }
      = -1;
    $results{
        &$val2name('any_branch_part1', 'some-login', 1, 'ru', {form_data => {branch_id => 'any branch',}, user => {},},
            undef, 1, '__EMPTY__',)
      }
      = -1;
    $results{
        &$val2name('any_branch_part1', 'some-login', 1, 'en', {form_data => {branch_id => 'any branch',}, user => {},},
            undef, 1, '__EMPTY__',)
      }
      = -1;

    # Пользователь, который уже есть, а он снова пытается заполнить эту же ветку анкеты
    $results{
        &$val2name('any branch', 'some-login', '', 'ru', {form_data => {branch_id => 'any branch',}, user => {},},
            undef, 1, '__EMPTY__',)
      }
      = -1;
    $results{
        &$val2name('any branch', 'some-login', '', 'en', {form_data => {branch_id => 'any branch',}, user => {},},
            undef, 1, '__EMPTY__',)
      }
      = -1;
    $results{
        &$val2name('any branch', 'some-login', '', 'ru', {form_data => {branch_id => 'any branch',}, user => {},},
            1, 1, '__EMPTY__',)
      }
      = -1;
    $results{
        &$val2name('any branch', 'some-login', '', 'en', {form_data => {branch_id => 'any branch',}, user => {},},
            1, 1, '__EMPTY__',)
      }
      = -1;

    my $check = sub {
        my (@vector) = @_;

        my $name = &$vec2name(@vector);
        my @aa = map {$vector[$_] ? ($names[$_] => $sets{$names[$_]}[$vector[$_]]) : ()} 0 .. $#vector;
        unless (
            eval {
                my ($can_fill_form, $reason, $message) =
                  can_fill_branch(map {$vector[$_] ? ($names[$_] => $sets{$names[$_]}[$vector[$_]]) : ()}
                      0 .. $#vector);
                unless (is($can_fill_form, $results{$name}, $name)) {
                    note '     Got: ', $can_fill_form;
                    note 'Expected: ', $results{$name};
                }
                return 1;
            }
          )
        {
            my $msg = $@ // '';
            $msg =~ s/ at .*//;
            $msg =~ s/:.*//;
            $msg =~ s/\s+$//;
            unless (ok(exists $croaks{$msg} && $results{$name} eq 'croak', $name)) {
                note '     Got: ', $msg;
                note 'Expected: ', $results{$name};
            }
        }
    };
    &$iterator($check);
    done_testing(List::Util::product(map {scalar @$_} values %sets));
}

all_cases();
