#!/usr/bin/perl

use strict;
use warnings FATAL => 'all';
use utf8;
use open qw(:std :utf8);

use Data::Dumper;
use Test::Differences;
use Test::More tests => 6;

use Utils;

BEGIN {
    use_ok('Mock::Subs');
}

sub local_test_sub {
    return 'original';
}

subtest 'features' => sub {
    plan tests => 4;

    subtest 'unmock' => sub {
        plan tests => 6;

        local $Data::Dumper::Deparse = 1;    # For diff purposes

        my $mock = Mock::Subs->new();
        eq_or_diff({%{$mock}}, {'original' => {},}, 'structure before mock');
        is(local_test_sub(), 'original', 'call before mock');
        my $original = \&local_test_sub;
        $mock->mock('local_test_sub', 'return' => 'mocked');
        my $mocked = \&local_test_sub;
        eq_or_diff(
            {%{$mock}},
            {
                'mocked'   => {'main::local_test_sub' => $mocked,},
                'original' => {'main::local_test_sub' => $original,},
            },
            'structure after mock'
        );
        is(local_test_sub(), 'mocked', 'call after mock');
        $mock->unmock('local_test_sub');
        eq_or_diff(
            {%{$mock}},
            {
                'mocked'   => {},
                'original' => {'main::local_test_sub' => $original,},
                'unmocked' => {'main::local_test_sub' => $mocked,},
            },
            'structure after unmock'
        );
        is(local_test_sub(), 'original', 'call after unmock');
    };

    subtest 'calling mocked sub is fatal' => sub {
        plan tests => 3;

        my $mock = Mock::Subs->new();
        is(local_test_sub(), 'original', 'call before mock');
        $mock->mock('local_test_sub', 'fatal' => 'mocked exception', 'return' => 'mocked');
        like(!eval {local_test_sub()} && $@, qr/mocked exception/, 'call after mock');
        $mock->unmock('local_test_sub');
        is(local_test_sub(), 'original', 'call after unmock');
    };

    subtest 'calling mocked pass test' => sub {
        plan tests => 4;

        my $mock = Mock::Subs->new();
        is(local_test_sub(), 'original', 'call before mock');
        $mock->mock('local_test_sub', 'test_pass' => 'mocked test', 'return' => 'mocked');
        is(local_test_sub(), 'mocked', 'call after mock');
        $mock->unmock('local_test_sub');
        is(local_test_sub(), 'original', 'call after unmock');
    };
    subtest 'calling mocked fail test' => sub {
        plan tests => 4;

        my $mock = Mock::Subs->new();
        is(local_test_sub(), 'original', 'call before mock');
        $mock->mock('local_test_sub', 'test_fail' => 'mocked fail test', 'return' => 'mocked');
        my $result;
      TODO: {
            local $Mock::Subs::TODO =
              'planned fail test';    # According to documentation should be $TODO, but test failes in Mock::Subs scope
            $result = local_test_sub();
        }
        is($result, 'mocked', 'call after mock');
        $mock->unmock('local_test_sub');
        is(local_test_sub(), 'original', 'call after unmock');
    };
};

subtest 'local sub' => sub {
    plan tests => 4;

    is(local_test_sub(), 'original', 'call before scope of mock');
    {    # block for scope purpose
        my $mock = Mock::Subs->new();
        is(local_test_sub(), 'original', 'call before mock');
        $mock->mock('local_test_sub', 'return' => 'mocked');
        is(local_test_sub(), 'mocked', 'call after mock');
    }
    is(local_test_sub(), 'original', 'call after scope of mock');
};

subtest 'exported sub' => sub {
    plan tests => 8;

    is(is_yndx_login('yndx-some-login'),        1, 'call before scope of mock');
    is(Utils::is_yndx_login('yndx-some-login'), 1, 'full-name call before scope of mock');
    {    # block for scope purpose
        my $mock = Mock::Subs->new();
        is(is_yndx_login('yndx-some-login'),        1, 'call before mock');
        is(Utils::is_yndx_login('yndx-some-login'), 1, 'full-name call before mock');
        $mock->mock('is_yndx_login', 'return' => 'mocked');
        is(Utils::is_yndx_login('yndx-some-login'), 1,        'full-name call after mock');
        is(is_yndx_login('yndx-some-login'),        'mocked', 'call after mock');
    }
    is(is_yndx_login('yndx-some-login'),        1, 'call after scope of mock');
    is(Utils::is_yndx_login('yndx-some-login'), 1, 'full-name call after scope of mock');
};

subtest 'module sub with export' => sub {
    plan tests => 8;

    is(is_yndx_login('yndx-some-login'),        1, 'call before scope of mock');
    is(Utils::is_yndx_login('yndx-some-login'), 1, 'full-name call before scope of mock');
    {    # block for scope purpose
        my $mock = Mock::Subs->new();
        is(is_yndx_login('yndx-some-login'),        1, 'call before mock');
        is(Utils::is_yndx_login('yndx-some-login'), 1, 'full-name call before mock');
        $mock->mock('Utils::is_yndx_login', 'return' => 'mocked');
        is(Utils::is_yndx_login('yndx-some-login'), 'mocked', 'full-name call after mock');
        is(is_yndx_login('yndx-some-login'),        1,        'call after mock');
    }
    is(is_yndx_login('yndx-some-login'),        1, 'call after scope of mock');
    is(Utils::is_yndx_login('yndx-some-login'), 1, 'full-name call after scope of mock');
};

subtest 'unknown sub' => sub {
    plan tests => 4;

    like(
        !eval {unknown_test_sub(); return 1;} && $@,
        qr/Undefined subroutine &main::unknown_test_sub called/,
        'call before scope of mock'
    );
    {    # block for scope purpose
        my $mock = Mock::Subs->new();
        like(
            !eval {unknown_test_sub(); return 1;} && $@,
            qr/Undefined subroutine &main::unknown_test_sub called/,
            'call before mock'
        );
        $mock->mock('unknown_test_sub', 'return' => 'mocked');
        is(unknown_test_sub(), 'mocked', 'call after mock');
    }
    like(
        !eval {unknown_test_sub(); return 1;} && $@,
        qr/Undefined subroutine &main::unknown_test_sub called/,
        'call after scope of mock'
    );
  }
