use strict;
use warnings FATAL => 'all';

use qbit;

use Test::Partner2::Simple;
use Test::Partner::Utils;

use Test::More;
use Test::Deep qw(cmp_deeply cmp_bag);
use Data::Dumper;

use PiConstants qw($MIDROLL);

sub create_with_undef {
    my ($app) = @_;

    my $public_id = $app->video_an_site_instream->add(
        get_new_video_instream(),
        page_id => 128972,
        type    => $MIDROLL,

        repeat            => 0,
        broadcast_replace => 0,
        start_time        => 10,

        max_repeat_count => undef,
        pip              => undef,
        repeat_after     => undef,
        server_side      => undef,
        try_client       => undef,
    );

    like($public_id, qr'^R-V-', 'can create midroll block with undef-s');

    cmp_deeply(
        $app->video_an_site_instream->get($public_id, fields => [qw(editable_fields)]),
        {
            editable_fields => {
                allow_multiple_dsp_ads    => 1,
                alternative_code          => 1,
                articles                  => 1,
                blind                     => 1,
                brands                    => 1,
                call_promo_code           => 1,
                caption                   => 1,
                comment                   => 1,
                count_positions           => 1,
                custom_bk_options         => 1,
                dsps                      => 1,
                dsp_mode                  => 1,
                geo                       => 1,
                is_custom_bk_data         => 1,
                jingle_code_at_end        => 1,
                jingle_code_at_start      => 1,
                jingle_code_before_promo  => 1,
                max_duration              => 1,
                mincpm                    => 1,
                moneymap                  => 1,
                order_tags                => 1,
                picategories              => 1,
                stopped_in_vmap           => 1,
                strategy                  => 1,
                target_tags               => 1,
                video_active              => 1,
                video_blocked             => 1,
                video_cpm                 => 1,
                video_motion              => 1,
                video_performance_active  => 1,
                video_performance_blocked => 1,
                video_performance_cpm     => 1,

                repeat            => 1,
                broadcast_replace => 1,
                max_repeat_count  => 1,
                repeat_after      => 1,

                # только если broadcast_replace = 1
                # pip => 1,

                # только если repeat = 1 ("Способ вызова" = 'Повтор по времени') или repeat = 0 ("Способ вызова" = 'Без повтора').
                start_time => 1,

                # только если repeat = 1 ("Способ вызова" = 'Повтор по времени').
                # repeat_after => 1,

                # только если repeat = 1 ("Способ вызова" = 'Повтор по времени').
                # max_repeat_count => 1,

                # только если repeat = 2 ("Способ вызова" = 'Строго по метке'')
                # HACK - этого поля быть не должно
                server_side => 1,

                # только если repeat = 2 ("Способ вызова" = 'Строго по метке'') и server_side = 1 ( "Тип запроса" = "Серверный).
                # try_client => 1,

                readonly => 1
            },
        },
        'check block data',
    );

    return 1;
}

sub create_without_values {
    my ($app) = @_;

    my $public_id = $app->video_an_site_instream->add(
        get_new_video_instream(),
        page_id => 128972,
        type    => $MIDROLL,

        repeat            => 0,
        broadcast_replace => 0,
    );

    like($public_id, qr'^R-V-', 'can create midroll block without undef-s');

    return 1;
}

sub pip {
    my ($app) = @_;

    my $public_id_1 = $app->video_an_site_instream->add(
        get_new_video_instream(),
        page_id => 128972,
        type    => $MIDROLL,

        repeat            => 0,
        broadcast_replace => 0,
        start_time        => 10,
        pip               => 1,
    );

    cmp_deeply(
        $app->video_an_site_instream->get($public_id_1, fields => [qw(page_id repeat broadcast_replace pip)]),
        {
            page_id           => 128972,
            repeat            => 0,
            broadcast_replace => 0,
            pip =>
              undef
            ,    # в add() указали 1, но pip может быть только если broadcast_replace = 1
        },
        'pip is undef',
    );

    my $public_id_2 = $app->video_an_site_instream->add(
        get_new_video_instream(),
        page_id => 128972,
        type    => $MIDROLL,

        repeat            => 0,
        broadcast_replace => 1,
        start_time        => 10,
        pip               => 1,
    );

    cmp_deeply(
        $app->video_an_site_instream->get($public_id_2, fields => [qw(page_id repeat broadcast_replace pip)]),
        {
            page_id           => 128972,
            repeat            => 0,
            broadcast_replace => 1,
            pip               => 1,
        },
        'pip is 1',
    );

    my $public_id_3 = $app->video_an_site_instream->add(
        get_new_video_instream(),
        page_id => 128972,
        type    => $MIDROLL,

        repeat            => 0,
        broadcast_replace => 1,
        start_time        => 10,
        pip               => 0,
    );

    cmp_deeply(
        $app->video_an_site_instream->get($public_id_3, fields => [qw(page_id repeat broadcast_replace pip)]),
        {
            page_id           => 128972,
            repeat            => 0,
            broadcast_replace => 1,
            pip               => 0,
        },
        'pip is 0',
    );

    return 1;
}

sub time_repeat {
    my ($app) = @_;

    my @missing_fields;

    try {
        my $public_id_1 = $app->video_an_site_instream->add(
            get_new_video_instream(),
            page_id => 128972,
            type    => $MIDROLL,

            repeat            => 1,    # Повтор по времени
            broadcast_replace => 0,
        );
    }
    catch {
        my $e = shift;
        @missing_fields = map {$_->{name}->[0]} @{from_json($e->{text})};
    };

    cmp_bag(
        \@missing_fields,
        [
            qw(
              repeat_after
              max_repeat_count
              )
        ],
        'got expected missing fields for time repeat',
    );

    my $public_id_2 = $app->video_an_site_instream->add(
        get_new_video_instream(),
        page_id => 128972,
        type    => $MIDROLL,

        repeat            => 1,    # Повтор по времени
        broadcast_replace => 0,

        start_time       => 10,
        repeat_after     => 20,
        max_repeat_count => 3,
    );

    cmp_deeply(
        $app->video_an_site_instream->get(
            $public_id_2,
            fields => [
                qw(
                  page_id repeat broadcast_replace start_time repeat_after max_repeat_count)
            ]
        ),
        {
            page_id           => 128972,
            repeat            => 1,
            broadcast_replace => 0,

            start_time       => 10,
            repeat_after     => 20,
            max_repeat_count => 3,
        },
        'got expected data',
    );

}

sub label_repeat {
    my ($app) = @_;

    my @missing_fields;

    try {
        my $public_id_1 = $app->video_an_site_instream->add(
            get_new_video_instream(),
            page_id => 128972,
            type    => $MIDROLL,

            repeat            => 2,    # Строго по метке
            broadcast_replace => 0,
        );
    }
    catch {
        my $e = shift;
        @missing_fields = map {$_->{name}->[0]} @{from_json($e->{text})};
    };

    cmp_bag(
        \@missing_fields,
        [
            qw(
              server_side
              try_client
              )
        ],
        'got expected missing fields for time repeat',
    );

    my $public_id_2 = $app->video_an_site_instream->add(
        get_new_video_instream(),
        page_id => 128972,
        type    => $MIDROLL,

        repeat            => 2,    # Строго по метке
        broadcast_replace => 0,

        server_side => 1,
        try_client  => 1,
    );

    cmp_deeply(
        $app->video_an_site_instream->get(
            $public_id_2,
            fields => [
                qw(
                  page_id repeat broadcast_replace server_side try_client)
            ]
        ),
        {
            page_id           => 128972,
            repeat            => 2,
            broadcast_replace => 0,

            server_side => 1,
            try_client  => 1,
        },
        'got expected data',
    );

    my $public_id_3 = $app->video_an_site_instream->add(
        get_new_video_instream(),
        page_id => 128972,
        type    => $MIDROLL,

        repeat            => 2,    # Строго по метке
        broadcast_replace => 0,

        server_side => 0,
        try_client  => 1
        , # это неправильное значение - try_client может быть только если server_side = 1
    );

    cmp_deeply(
        $app->video_an_site_instream->get(
            $public_id_3,
            fields => [
                qw(
                  page_id repeat broadcast_replace server_side try_client)
            ]
        ),
        {
            page_id           => 128972,
            repeat            => 2,
            broadcast_replace => 0,

            server_side => 0,
            try_client  => undef,    # исправлен
        },
        'got expected data',
    );

}

sub create_with_start_time {
    my ($app) = @_;

    my $public_id = $app->video_an_site_instream->add(
        get_new_video_instream(),
        page_id => 128972,
        type    => $MIDROLL,

        repeat            => 0,
        broadcast_replace => 0,
        start_time        => 11,
    );

    like($public_id, qr'^R-V-', 'created midroll with repeat => 0, start_time => 1');

    cmp_deeply(
        $app->video_an_site_instream->get($public_id, fields => [qw(public_id repeat start_time)]),
        {
            public_id  => $public_id,
            repeat     => 0,
            start_time => 11,
        },
        'get expected block data',
    );

    return 1;
}

sub create_and_edit {
    my ($app) = @_;

    my $public_id = $app->video_an_site_instream->add(
        get_new_video_instream(),
        page_id => 128972,
        type    => $MIDROLL,

        repeat            => 0,
        broadcast_replace => 0,
    );

    like($public_id, qr'^R-V-', 'created midroll with repeat => 0, broadcast_replace => 0');

    cmp_deeply(
        $app->video_an_site_instream->get($public_id, fields => [qw(public_id repeat broadcast_replace)]),
        {
            public_id         => $public_id,
            repeat            => 0,
            broadcast_replace => 0,
        },
        'get expected block data',
    );

    $app->video_an_site_instream->do_action(
        $public_id,
        'edit',
        repeat      => 2,
        server_side => 1,
        try_client  => 0,
    );

    return 1;
}

run_tests(
    sub {
        my ($app) = @_;

        create_with_undef($app);
        create_without_values($app);
        pip($app);

        time_repeat($app);
        label_repeat($app);

        create_with_start_time($app);

        create_and_edit($app);

        my ($is_ok, $details) =
          $app->qbit_validator_checker->check_all_elements_in_model(accessor => 'video_an_site_instream');
        ok($is_ok, 'all instream blocks in db pass qbit validation');

        diag(Dumper($details)) unless $is_ok;

    },
    do_not_die_on_fail => 1,
);
