#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use Test::Partner2::Simple;
use Test::Partner2::Mock qw(mock_subs);
use Test::More;
use Test::Differences qw(eq_or_diff);

use Test::Partner::Utils qw(request_to_api eq_or_diff_http_message);

use qbit;

my $MOCKED_BK_ANSWER = q({
   "data" : {
      "currencies" : [
         {
            "code" : "USD",
            "id" : "1"
         },
         {
            "code" : "RUB",
            "id" : "2"
         },
         {
            "code" : "EUR",
            "id" : "3"
         }
      ],
      "dimensions" : {
         "domain" : {
            "index" : 1,
            "title" : "Домен или Bundle ID",
            "type" : "categories"
         },
         "page_id" : {
            "index" : 2,
            "title" : "Page ID",
            "type" : "page_id"
         }
      },
      "is_last_page" : true,
      "measures" : {
         "rtb_block_all_hits" : {
            "index" : 3,
            "title" : "Запросы рекламы в блоках",
            "type" : "number",
            "unit" : "count"
         },
         "rtb_block_direct_shows" : {
            "index" : 4,
            "title" : "Показы Директа в RTB-блоках",
            "type" : "number",
            "unit" : "count"
         }
      },
      "periods" : [
         [
            "2020-09-10",
            "2020-09-10"
         ]
      ],
      "points" : [
         {
            "dimensions" : {
               "domain" : "kinopoisk.ru",
               "page_id" : 53100
            },
            "measures" : [
               {
                  "rtb_block_all_hits" : 124980,
                  "rtb_block_direct_shows" : 44
               }
            ]
         }
      ],
      "report_title" : "Отчет за период 10.09.2020 - 10.09.2020",
      "totals" : {
         "2" : [
            {
               "rtb_block_all_hits" : 124980,
               "rtb_block_direct_shows" : 44
            }
         ]
      }
   },
   "result" : "ok"
});

my $STATUS_EXPECTED_BAD_REQ = "Status: 400 Bad request";
my @HEADERS_EXPECTED_BAD_REQ =
  sort +('X-Content-Type-Options: nosniff', 'Content-Type: application/json; charset=UTF-8',);

sub split_response_stdout {
    my ($stdout) = @_;
    my ($headers_string, $body) = split("\n\n", $stdout);
    my @headers = split("\n", $headers_string);
    my $status = shift @headers;
    return ($status, \@headers, $body);
}

run_tests(
    sub {
        my ($app) = @_;

        my $check_bkstatistics_opts_sub;
        $app->bk_statistics;
        mock_subs(
            {
                'Application::Model::BKStatistics::get_statistics2' => sub {
                    my ($self, %opts) = @_;
                    $check_bkstatistics_opts_sub->(\%opts) if defined $check_bkstatistics_opts_sub;
                    return from_json($MOCKED_BK_ANSWER)->{data};
                  }
            }
        );

        $check_bkstatistics_opts_sub = sub {
            my ($opts) = @_;
            ok(!exists $opts->{currency}, 'no fake currency passed');
        };
        my ($stdout, $stderr, $exit_status) = request_to_api(
            app          => $app,
            path         => 'statistics2',
            method       => 'get',
            ext          => 'json',
            query_string => 'lang=C&oauth_token=mocked-yan-partner&pretty=1'
              . '&period=2020-09-10&period=2020-09-10'
              . '&level=[{"id":"advnet_context_on_site"}]'
              . '&field=rtb_block_direct_shows&field=rtb_block_all_hits'
              . '&limits={"limit":50,"offset":0}'
              . '&dimension_field=date|day'
              . '&entity_field=page_id&entity_field=domain'
              . '&filter=["AND",[["page_id","=","53100"]]]'
              . '&order_by=[{"field":"date","dir":"asc"}]',
        );

        is($exit_status, 1, 'exit status');

        eq_or_diff($stderr, '', 'stderr', {context => 1});

        eq_or_diff_http_message(
            $stdout, "Status: 200 OK
X-Content-Type-Options: nosniff
X-Yandex-Login: mocked-yan-partner
Content-Type: application/json; charset=UTF-8

$MOCKED_BK_ANSWER
", 'stdout', {context => 10}
        );

        $check_bkstatistics_opts_sub = undef;
        ($stdout, $stderr, $exit_status) = request_to_api(
            app          => $app,
            path         => 'statistics2',
            method       => 'get',
            ext          => 'json',
            query_string => 'lang=C&oauth_token=mocked-yan-partner&pretty=1'
              . '&field=partner_wo_nds'
              . '&dimension_field=["date|day"]'
              . '&entity_field=page_id&entity_field=domain'
              . '&filter=["AND",[["page_id","=","53100"]]]',
        );

        is($exit_status, 1, 'exit status');

        eq_or_diff($stderr, '', 'stderr', {context => 1});

        my ($status, $headers, $body) = split_response_stdout($stdout);
        eq_or_diff($status,          $STATUS_EXPECTED_BAD_REQ,   'stdout 400 http status');
        eq_or_diff([sort @$headers], \@HEADERS_EXPECTED_BAD_REQ, 'stdout 400 http headers');
        eq_or_diff(
            from_json($body),
            from_json(
                '{
            "error_type" : "Exception::Validation::BadArguments",
            "message" : "Missed required parameter \"period\"",
            "result" : "error"
        }'
            ),
            'stdout 400 http body'
        );

        $check_bkstatistics_opts_sub = sub {
            my ($opts) = @_;
            eq_or_diff($opts->{currency}, 'EUR', 'expected currency passed');
        };
        ($stdout, $stderr, $exit_status) = request_to_api(
            app          => $app,
            path         => 'statistics2',
            method       => 'get',
            ext          => 'json',
            query_string => 'lang=C&oauth_token=mocked-yan-partner&pretty=1'
              . '&period=2020-09-11&period=2020-09-11&period=2020-09-10&period=2020-09-10'
              . '&dimension_field=["date|day"]'
              . '&entity_field=page_id&entity_field=domain'
              . '&field=partner_wo_nds'
              . '&currency=EUR'
              . '&filter=["AND",[["page_id","=","53100"]]]',
        );

        is($exit_status, 1, 'exit status');

        eq_or_diff($stderr, '', 'stderr', {context => 1});

        eq_or_diff_http_message(
            $stdout, "Status: 200 OK
X-Content-Type-Options: nosniff
X-Yandex-Login: mocked-yan-partner
Content-Type: application/json; charset=UTF-8

$MOCKED_BK_ANSWER
", 'stdout', {context => 10}
        );
    },
    application_package => 'API',
);
