#!/usr/bin/perl

use strict;
use warnings FATAL => 'all';

use qbit;

use Test::Differences;
use Test::Exception;
use Test::More tests => 32;

use Exception::API::AdFoxGraphQL;
use Exception::Validator::Errors;
use Test::Partner2::Simple;

my %normal_args = (
    mobile_apps => [
        {
            store_id  => 'com.yandex',
            store_url => 'https://yandex.com',
            type      => 1,
        },
    ],
    presets => ['MOBILEMEDIATION'],
    user_id => 654321,
);

my $normal_query = {
    query => '
mutation ($yandexUid: Int64!, $mobileApps: [AnketaMobileAppInput!], $presets: [RegistrationPreset!]){
    AdfoxInternal {
        registerUser(params: {
            yandexUid: $yandexUid,
            mobileApps: $mobileApps
        },
        presets: $presets
        ) {
            id
            login
        }
    }
}',
    variables => {
        mobileApps => [
            {
                storeId  => 'com.yandex',
                storeUri => 'https://yandex.com',
                appName  => 'Mobile app campaign',
                platform => 'ANDROID',
            },
        ],
        presets   => ['MOBILEMEDIATION'],
        yandexUid => '654321',
    },
};

run_tests(
    sub {
        my ($app) = @_;

        my $api = $app->api_adfox_graphql;
        $api->set_option('mock', FALSE);

        no warnings 'redefine';
        no strict 'refs';

        lives_and {
            local *{'LWP::UserAgent::request'} = sub {
                my ($self, $request) = @_;
                eq_or_diff(from_json($request->content()), $normal_query, 'call: Get expected request',);
                is($request->method(), 'POST', 'call: Get expected request method',);
                my $r = HTTP::Response->new(200);
                $r->request(HTTP::Request->new());
                $r->content('{"data":{"AdfoxInternal":{"registerUser":{"id":123456,"login":"mocked_adfox_login"}}}}');
                return $r;
            };
            eq_or_diff(
                $api->create_user(%normal_args),
                {
                    id    => 123456,
                    login => 'mocked_adfox_login',
                }
            );
        }
        'Get expected answer (MOBILEMEDIATION)';

        lives_and {
            local *{'LWP::UserAgent::request'} = sub {
                my ($self, $request) = @_;
                my $r = HTTP::Response->new(200);
                $r->request(HTTP::Request->new());
                $r->content('{"data":{"AdfoxInternal":{"registerUser":{"id":123456,"login":"mocked_adfox_login"}}}}');
                return $r;
            };
            local $normal_args{presets};
            $normal_args{presets} = ['ADFOXSITES'];
            local $normal_query->{variables}{presets};
            $normal_query->{variables}{presets} = ['ADFOXSITES'];
            eq_or_diff(
                $api->create_user(%normal_args),
                {
                    id    => 123456,
                    login => 'mocked_adfox_login',
                }
            );
        }
        'Get expected answer (ADFOXSITES)';

        throws_ok {
            local *{'LWP::UserAgent::request'} = sub {
                my $r = HTTP::Response->new(200);
                $r->request(HTTP::Request->new());
                $r->content(
'{"errors":[{"message":"Variable \u0022$yandexUid\u0022 got invalid value \u0022654321\u0022.\nExpected type \u0022Int\u0022, found \u0022654321\u0022.","category":"graphql","locations":[{"line":2,"column":11}]}]}'
                );
                return $r;
            };
            $api->create_user(%normal_args);
        }
        'Exception::API::AdFoxGraphQL', 'Get known negative answer';
        is(
            $@->message(),
            qq[Variable "\$yandexUid" got invalid value "654321".\nExpected type "Int", found "654321".],
            'Known negative answer message'

          );

        throws_ok {
            local *{'LWP::UserAgent::request'} = sub {
                my $r = HTTP::Response->new(200);
                $r->request(HTTP::Request->new());
                $r->content('invalid');
                return $r;
            };
            $api->create_user(%normal_args);
        }
        'Exception::API::AdFoxGraphQL', 'Get invalid json';
        like($@->message(), qr/Error in from_json/, 'Get invalid json message');

        throws_ok {
            local *{'LWP::UserAgent::request'} = sub {
                my $r = HTTP::Response->new(200);
                $r->request(HTTP::Request->new());
                $r->content('{}');
                return $r;
            };
            $api->create_user(%normal_args);
        }
        'Exception::API::AdFoxGraphQL', 'Get unknown negative answer';
        is($@->message(), 'Incorrect answer', 'Unknown negative answer message');

        throws_ok {
            local $normal_args{'mobile_apps'}[0]{'store_id'} = 'ru';
            $api->create_user(%normal_args);
        }
        'Exception::Validator::Errors', 'Validation: short store_id';
        like($@->message(), qr/"Length \\"ru\\" less than \\"3\\""/, 'Validation: short store_id message');

        throws_ok {
            local $normal_args{'mobile_apps'}[0]{'store_id'};
            delete $normal_args{'mobile_apps'}[0]{'store_id'};
            $api->create_user(%normal_args);
        }
        'Exception::Validator::Errors', 'Validation: lost store_id';
        like($@->message(), qr/"Key \\"store_id\\" required"/, 'Validation: lost store_id message');

        throws_ok {
            local $normal_args{'mobile_apps'}[0]{'store_url'} = 'ru';
            $api->create_user(%normal_args);
        }
        'Exception::Validator::Errors', 'Validation: short store_url';
        like($@->message(), qr/"Length \\"ru\\" less than \\"4\\""/, 'Validation: short store_url message');

        throws_ok {
            local $normal_args{'mobile_apps'}[0]{'store_url'};
            delete $normal_args{'mobile_apps'}[0]{'store_url'};
            $api->create_user(%normal_args);
        }
        'Exception::Validator::Errors', 'Validation: lost store_url';
        like($@->message(), qr/"Key \\"store_url\\" required"/, 'Validation: lost store_url message');

        throws_ok {
            local $normal_args{'mobile_apps'}[0]{'type'} = 3;
            $api->create_user(%normal_args);
        }
        'Exception::Validator::Errors', 'Validation: invalid type';
        like($@->message(), qr/"Got value \\"3\\" not in array: 1, 2"/, 'Validation: invalid type message');

        throws_ok {
            local $normal_args{'mobile_apps'}[0]{'type'};
            delete $normal_args{'mobile_apps'}[0]{'type'};
            $api->create_user(%normal_args);
        }
        'Exception::Validator::Errors', 'Validation: lost type';
        like($@->message(), qr/"Key \\"type\\" required"/, 'Validation: lost type message');

        throws_ok {
            local $normal_args{'user_id'} = 3.1415926;
            $api->create_user(%normal_args);
        }
        'Exception::Validator::Errors', 'Validation: invalid user_id';
        like($@->message(), qr/"Must be an unsigned integer"/, 'Validation: invalid user_id message');

        throws_ok {
            local $normal_args{'user_id'};
            delete $normal_args{'user_id'};
            $api->create_user(%normal_args);
        }
        'Exception::Validator::Errors', 'Validation: lost user_id';
        like($@->message(), qr/"Key \\"user_id\\" required"/, 'Validation: lost user_id message');

        throws_ok {
            my $m_app = $normal_args{'mobile_apps'}[0];
            local $normal_args{'mobile_apps'} = $m_app;
            $api->create_user(%normal_args);
        }
        'Exception::Validator::Errors', 'Validation: invalid mobile_apps';
        like($@->message(), qr/"Data must be ARRAY"/, 'Validation: invalid mobile_apps message');

        throws_ok {
            local $normal_args{presets};
            delete $normal_args{presets};
            $api->create_user(%normal_args);
        }
        'Exception::Validator::Errors', 'Validation: invalid presets';
        like($@->message(), qr/"Key \\"presets\\" required"/, 'Validation: invalid presets message');
    },
    'init' => [qw( api_adfox_graphql )],
);
