#!/usr/bin/perl

use strict;
use warnings FATAL => 'all';

use qbit;

use Test::Differences;
use Test::Exception;
use Test::More tests => 15;

use Exception::API::AdFoxGraphQL;
use Exception::Validator::Errors;
use Test::Partner2::Simple;

my %normal_args = (
    adfox_login => 'traden',
    adfox_psw   => '123456',
);
my $normal_header = 'password 123456';
my $normal_query  = {
    'query' => '
query ($adfoxAccount: String!, $passportUid: Int64) {
    AdfoxInternal {
        billingClientInfo(adfoxAccount: $adfoxAccount, passportUid: $passportUid) {
            billingClientId,
            billingContractId,
            hasOtherAccounts,
            hasPaidServices,
            piAccounts,
            complexPiLinks,
            inWhitelist,
            needCloseContract,
            userId,
            hasSameUidInOtherAccount
        }
    }
}',
    'variables' => {'adfoxAccount' => 'traden'},
};
my $normal_response = {
    'data' => {
        'AdfoxInternal' => {
            'billingClientInfo' => {
                'piAccounts'               => ['pi-login'],
                'inWhitelist'              => TRUE,
                'billingContractId'        => '123987',
                'userId'                   => 31415,
                'complexPiLinks'           => FALSE,
                'needCloseContract'        => TRUE,
                'hasOtherAccounts'         => TRUE,
                'billingClientId'          => 987654,
                'hasPaidServices'          => FALSE,
                'hasSameUidInOtherAccount' => FALSE
            }
        }
    }
};
my %normal_args_with_uid = (%normal_args, uid => 666);
my $normal_query_with_uid = {(%$normal_query, 'variables' => {'adfoxAccount' => 'traden', 'passportUid' => 666})};
my $normal_response_with_uid = {
    'data' => {
        AdfoxInternal => {
            billingClientInfo =>
              {(%{$normal_response->{'data'}{AdfoxInternal}{billingClientInfo}}), 'hasSameUidInOtherAccount' => TRUE}
        }
    }
};

my @args_header_query = (
    {
        args     => \%normal_args,
        header   => $normal_header,
        query    => $normal_query,
        response => $normal_response,
    },
    {
        args     => \%normal_args_with_uid,
        header   => $normal_header,
        query    => $normal_query_with_uid,
        response => $normal_response_with_uid,
    },
);

run_tests(
    sub {
        my ($app) = @_;

        my $api = $app->api_adfox_graphql;
        $api->set_option('mock', FALSE);

        no warnings 'redefine';
        no strict 'refs';

        local *{'LWP::UserAgent::request'};

        for my $ahq (@args_header_query) {

            *{'LWP::UserAgent::request'} = sub {
                my ($self, $request) = @_;
                eq_or_diff($request->header('Authorization'),
                    $ahq->{header}, 'call: Get expected authorization header field');
                eq_or_diff(from_json($request->content()), $ahq->{query}, 'call: Get expected request',);
                is($request->method(), 'POST', 'call: Get expected request method',);
                my $r = HTTP::Response->new(200);
                $r->request(HTTP::Request->new());
                $r->content(to_json($ahq->{response}));
                return $r;
            };

            eq_or_diff(
                $api->get_billing_client_info(%{$ahq->{args}}),
                $ahq->{response}{data}{AdfoxInternal}{billingClientInfo},
                'Get expected answer'
            );
        }

        *{'LWP::UserAgent::request'} = sub {
            my $r = HTTP::Response->new(200);
            $r->request(HTTP::Request->new());
            $r->content(
'{"errors":[{"locations":[{"column":5,"line":30}],"path":["AdfoxInternal","billingClientInfo"],"category":"internal","code":8,"message":"Passport client with yandex uid = 301082620 is already exists in adfox."}],"data":{"AdfoxInternal":null}}'
            );
            return $r;
        };
        throws_ok {
            $api->get_billing_client_info(%normal_args);
        }
        'Exception::API::AdFoxGraphQL', 'Correct exception while negative answer';
        eq_or_diff(
            $@->message(),
            'Passport client with yandex uid = 301082620 is already exists in adfox.',
            'Get known negative answer'
        );

        *{'LWP::UserAgent::request'} = sub {
            my $r = HTTP::Response->new(200);
            $r->request(HTTP::Request->new());
            $r->content('invalid');
            return $r;
        };
        throws_ok {
            $api->get_billing_client_info(%normal_args);
        }
        'Exception::API::AdFoxGraphQL', 'Correct exception while invalid json';
        like($@->message(), qr/Error in from_json/, 'Get invalid json');

        *{'LWP::UserAgent::request'} = sub {
            my $r = HTTP::Response->new(200);
            $r->request(HTTP::Request->new());
            $r->content('{}');
            return $r;
        };
        throws_ok {
            $api->get_billing_client_info(%normal_args);
            fail('Get unknown negative answer');
        }
        'Exception::API::AdFoxGraphQL', 'Correct exception while unknown negative answer';
        eq_or_diff($@->message(), 'Incorrect answer', 'Get unknown negative answer');
    },
    init                 => [qw( api_adfox_graphql )],
    dont_create_database => TRUE,
);
