#!/usr/bin/perl

use strict;
use warnings FATAL => 'all';

use qbit;

use Test::Differences;
use Test::Exception;
use Test::More tests => 18;

use Exception::API::AdFoxGraphQL;
use Exception::Validator::Errors;
use Test::Partner2::Simple;

use Data::Dumper;

sub normal_query {
    my ($app_id, $active) = @_;

    return +{
        'query' => '
mutation($appId:Int!, $isActive:Boolean!){
    AdfoxInternal{
        updateMobileAppStatus(piAppId: $appId, isActive: $isActive)
    }
}',
        'variables' => {
            'isActive' => $active ? JSON::XS::true : JSON::XS::false,
            'appId' => $app_id
        },
    };
}

run_tests(
    sub {
        my ($app) = @_;

        my $api = $app->api_adfox_graphql;
        $api->set_option('mock', FALSE);

        set_request_return_errors();
        throws_ok {
            $api->update_mobile_app_status(
                app_id => 1,
                active => 1
            );
        }
        'Exception::API::AdFoxGraphQL', 'valid exception';
        eq_or_diff($@->message(), 'fail', 'valid exception message');

        set_request_return_bad_json();
        throws_ok {
            $api->update_mobile_app_status(
                app_id => 1,
                active => 1
            );
        }
        'Exception::API::AdFoxGraphQL', 'bad json';
        like($@->message(), qr/Error in from_json/, 'bad json message');

        set_request_return_500();
        throws_ok {
            $api->update_mobile_app_status(
                app_id => 1,
                active => 1
            );
        }
        'Exception::API::HTTP', 'got 500';
        like($@->message(), qr/500 Internal Server Error/, 'got 500 message');

        set_request_return_not_ok(1);
        throws_ok {
            $api->update_mobile_app_status(
                app_id => 1,
                active => 1
            );
        }
        'Exception::API::AdFoxGraphQL', 'bad result';
        eq_or_diff($@->message(), 'Incorrect answer', 'bad result message');

        set_request_return_bad_response();
        throws_ok {
            $api->update_mobile_app_status(
                app_id => 1,
                active => 1
            );
        }
        'Exception::API::AdFoxGraphQL', 'bad response';
        eq_or_diff($@->message(), 'Incorrect answer', 'bad response message');

        set_request_return_ok(10, 1);
        lives_and {
            my $res = $api->update_mobile_app_status(app_id => 10, active => 1);
            is($res, 1, 'return approved');
        }
        'success approve';

        set_request_return_ok(11, 0);
        lives_and {
            my $res = $api->update_mobile_app_status(app_id => 11, active => 0);
            is($res, 1, 'return rejected');
        }
        'success reject';
    },
    'init' => [qw(api_adfox_graphql)],
);

sub set_request_return_ok {
    my ($app_id, $active) = @_;

    no warnings 'redefine';
    no strict 'refs';
    require LWP::UserAgent;
    *{'LWP::UserAgent::request'} = sub {
        my ($self, $request) = @_;

        eq_or_diff(from_json($request->content()), normal_query($app_id, $active), 'call: Get expected request',);
        is($request->method(), 'POST', 'call: Get expected request method',);

        my $r = HTTP::Response->new(200);
        $r->request(HTTP::Request->new());
        $r->content('{"data":{"AdfoxInternal":{"updateMobileAppStatus":' . ($active ? 'true' : 'false') . '}}}');
        return $r;
    };
}

sub set_request_return_not_ok {
    my ($active) = @_;

    no warnings 'redefine';
    no strict 'refs';
    require LWP::UserAgent;
    *{'LWP::UserAgent::request'} = sub {
        my $r = HTTP::Response->new(200);
        $r->request(HTTP::Request->new());
        $r->content('{"data":{"AdfoxInternal":{"updateMobileAppStatus":' . (!$active ? 'true' : 'false') . '}}}');
        return $r;
    };
}

sub set_request_return_bad_response {

    no warnings 'redefine';
    no strict 'refs';
    require LWP::UserAgent;
    *{'LWP::UserAgent::request'} = sub {
        my $r = HTTP::Response->new(200);
        $r->request(HTTP::Request->new());
        $r->content('{"data":{}}');
        return $r;
    };
}

sub set_request_return_500 {

    no warnings 'redefine';
    no strict 'refs';
    require LWP::UserAgent;
    *{'LWP::UserAgent::request'} = sub {
        my $r = HTTP::Response->new(500);
        $r->request(HTTP::Request->new());
        return $r;
    };
}

sub set_request_return_errors {

    no warnings 'redefine';
    no strict 'refs';
    require LWP::UserAgent;
    *{'LWP::UserAgent::request'} = sub {
        my $r = HTTP::Response->new(200);
        $r->request(HTTP::Request->new());
        $r->content('{"errors":[{"message":"fail"}]}');
        return $r;
    };
}

sub set_request_return_bad_json {

    no warnings 'redefine';
    no strict 'refs';
    require LWP::UserAgent;
    *{'LWP::UserAgent::request'} = sub {
        my $r = HTTP::Response->new(200);
        $r->request(HTTP::Request->new());
        $r->content('');
        return $r;
    };
}
