#!/usr/bin/perl

use strict;
use warnings FATAL => 'all';

use qbit;

use Test::Partner2::Simple;
use Test::More tests => 2 + 4;
use Test::Differences;
use Test::Exception;

use Test::Partner2::Mock qw(mock_subs restore_subs);

my $TASK_ID       = '1362355c-afa3-4656-91a0-a62922e5d229';
my $NORMAL_RESULT = {
    event_horizon => '2018-09-05 13:48:50',
    fields        => ['correct', 'week_id', 'week_name', 'count',],
    state         => 'SUCCESS',
    table         => [[1, 2448, 49, 99232555,],],
    task_id       => $TASK_ID,
};
my $ERROR_RESULT = {
    code => 159,
    data => undef,
    message =>
'Received from ch11.myt.adfox.ru:9000, 2a02:6b8:b020:4300::ce:11. DB::Exception: Timeout exceeded: elapsed 180.61648591 seconds, maximum: 180. (version 19.9.3.31 (official build))',
    name => 'Error',
};

run_tests(
    sub {
        my ($app) = @_;

        restore_subs(['LWP::UserAgent::request']);

        subtest 'expected answer' => sub {
            plan tests => 2;

            mock_subs(
                {
                    'LWP::UserAgent::request' => sub {
                        my ($ua, $request) = @_;

                        eq_or_diff(
                            from_json($request->content()),
                            {
                                id      => 1,
                                jsonrpc => '2.0',
                                method  => 'get_task_result',
                                params  => [$TASK_ID,],
                            },
                            'sending expected data to adfox',
                        );

                        my $response = HTTP::Response->new(200);
                        $response->request(HTTP::Request->new());
                        $response->content(
                            to_json(
                                {
                                    id      => 1,
                                    jsonrpc => 2.0,
                                    result  => $NORMAL_RESULT,
                                }
                            )
                        );

                        return $response;
                    },
                }
            );

            my $task_result = $app->api_adfox_rtd->get_task_result($TASK_ID);

            eq_or_diff($task_result, $NORMAL_RESULT, 'got expected task result');

            restore_subs(['LWP::UserAgent::request']);
        };
        subtest 'invalid answer' => sub {
            plan tests => 2;

            mock_subs(
                {
                    'LWP::UserAgent::request' => sub {
                        my ($ua) = @_;

                        my $response = HTTP::Response->new(200);
                        $response->request(HTTP::Request->new());
                        $response->content(to_json([1, 2, 3]),);

                        return $response;
                    },
                }
            );

            my $e;

            throws_ok {
                my $task_result = $app->api_adfox_rtd->get_task_result('1362355c-afa3-4656-91a0-a62922e5d229');
            }
            'Exception::API::AdFoxRTD', 'got expected exception';
            like($@->message(), qr/Unexpected answer/, 'got expected message');

            restore_subs(['LWP::UserAgent::request']);
        };
        subtest 'answer with unexpected error' => sub {
            plan tests => 3;

            mock_subs(
                {
                    'LWP::UserAgent::request' => sub {
                        my ($ua, $request) = @_;

                        eq_or_diff(
                            from_json($request->content()),
                            {
                                id      => 1,
                                jsonrpc => '2.0',
                                method  => 'get_task_result',
                                params  => ['1362355c-afa3-4656-91a0-a62922e5d229',],
                            },
                            'sending expected data to adfox',
                        );

                        my $response = HTTP::Response->new(200);
                        $response->request(HTTP::Request->new());
                        $response->content(
                            to_json(
                                {
                                    error   => $ERROR_RESULT,
                                    id      => 1,
                                    jsonrpc => 2.0,
                                }
                            )
                        );

                        return $response;
                    },
                }
            );

            throws_ok {
                my $task_result = $app->api_adfox_rtd->get_task_result('1362355c-afa3-4656-91a0-a62922e5d229');
            }
            'Exception::API::AdFoxRTD', 'got expected exception';
            like($@->message(), qr/Unexpected answer/, 'got expected message');

            restore_subs(['LWP::UserAgent::request']);
        };
        subtest 'answer with error' => sub {
            plan tests => 2;

            mock_subs(
                {
                    'LWP::UserAgent::request' => sub {
                        my ($ua, $request) = @_;

                        eq_or_diff(
                            from_json($request->content()),
                            {
                                id      => 1,
                                jsonrpc => '2.0',
                                method  => 'get_task_result',
                                params  => [$TASK_ID,],
                            },
                            'sending expected data to adfox',
                        );

                        my $response = HTTP::Response->new(200);
                        $response->request(HTTP::Request->new());
                        $response->content(
                            to_json(
                                {
                                    error   => $ERROR_RESULT,
                                    id      => 1,
                                    jsonrpc => 2.0,
                                }
                            )
                        );

                        return $response;
                    },
                }
            );

            my $error_result = $app->api_adfox_rtd->get_task_result($TASK_ID, 'need_error');

            eq_or_diff($error_result, $ERROR_RESULT, 'got expected error result');

            restore_subs(['LWP::UserAgent::request']);
        };

        mock_subs([qw(LWP::UserAgent::request)]);
    },
);
