#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use Test::More tests => 11;
use Test::Exception;

use Test::Partner2::Simple;

use HTTP::Response;
use HTTP::Date;
use HTTP::Headers;

my $list_response_truncated_content = <<LIST;
<?xml version="1.0" encoding="UTF-8"?>
<ListBucketResult xmlns="http://s3.amazonaws.com/doc/2006-03-01/">
    <Name>partner-publisher-stat</Name>
    <Prefix/>
    <Marker/>
    <MaxKeys>1</MaxKeys>
    <IsTruncated>true</IsTruncated>
    <Contents>
        <Key>test_file1</Key>
        <LastModified>2017-04-27T14:39:46.000Z</LastModified>
        <ETag>"f94506a3e95c469bf6c198d03e329543"</ETag>
        <Size>13</Size>
    </Contents>
</ListBucketResult>
LIST

my $list_response_last_content = <<LIST;
<?xml version="1.0" encoding="UTF-8"?>
<ListBucketResult xmlns="http://s3.amazonaws.com/doc/2006-03-01/">
    <Name>partner-publisher-stat</Name>
    <Prefix/>
    <Marker/>
    <MaxKeys>1</MaxKeys>
    <IsTruncated>false</IsTruncated>
    <Contents>
        <Key>test_file2</Key>
        <LastModified>2017-04-27T14:39:56.000Z</LastModified>
        <ETag>"7477adcf2bfb7d724beb3e23d06a0c16"</ETag>
        <Size>13</Size>
    </Contents>
</ListBucketResult>
LIST

my $list_response_filtered_content = <<LIST;
<?xml version='1.0' encoding='UTF-8'?>
<ListBucketResult xmlns="http://s3.amazonaws.com/doc/2006-03-01/">
    <Name>partner-publisher-stat</Name>
    <Prefix>test_file1</Prefix>
    <Marker />
    <MaxKeys>1000</MaxKeys>
    <IsTruncated>false</IsTruncated>
    <Contents>
        <Key>test_file1</Key>
        <LastModified>2017-04-27T14:39:46.000Z</LastModified>
        <ETag>"f94506a3e95c469bf6c198d03e329543"</ETag>
        <Size>13</Size>
    </Contents>
</ListBucketResult>
LIST

my $default_response = HTTP::Response->new(200, 'OK', HTTP::Headers->new(),);

run_tests(
    sub {
        my ($app) = @_;

        test_method(
            $app, 'list',
            [],
            requests => [
                {
                    method     => 'GET',
                    query_form => {},
                    response => HTTP::Response->new(200, 'OK', HTTP::Headers->new(), $list_response_truncated_content,),
                },
                {
                    method     => 'GET',
                    query_form => {marker => 'test_file1',},
                    response   => HTTP::Response->new(200, 'OK', HTTP::Headers->new(), $list_response_last_content,),
                }
            ],
            return_value => ['test_file1', 'test_file2'],
        );

        test_method(
            $app, 'list',
            ['test_file1'],
            requests => [
                {
                    method     => 'GET',
                    query_form => {prefix => 'test_file1'},
                    response => HTTP::Response->new(200, 'OK', HTTP::Headers->new(), $list_response_filtered_content,),
                },
            ],
            return_value => ['test_file1'],
        );

        test_method(
            $app, 'put',
            ['test_key', 'test_type', 'test_scalar'],
            requests => [
                {
                    method          => 'PUT',
                    key             => 'test_key',
                    request_content => 'test_scalar',
                    headers         => HTTP::Headers->new('Content-Type' => 'test_type',),
                    response        => $default_response,
                },
            ],
            return_value => 1,
        );

        test_method(
            $app,
            'put_file',
            ['test_key', 'test_type', 'test_file'],
            requests => [
                {
                    method               => 'PUT',
                    key                  => 'test_key',
                    request_content_file => 'test_file',
                    headers              => HTTP::Headers->new('Content-Type' => 'test_type',),
                    response             => $default_response,
                },
            ],
            return_value => 1,
        );

        test_method(
            $app, 'get',
            ['test_key'],
            requests => [
                {
                    method   => 'GET',
                    key      => 'test_key',
                    response => HTTP::Response->new(
                        200, 'OK', HTTP::Headers->new('Content-Type' => 'application/test'),
                        'test_scalar',
                    ),
                },
            ],
            return_value => {
                content      => 'test_scalar',
                content_type => 'application/test',
            },
        );

        test_method(
            $app, 'get',
            ['test_key'],
            requests => [
                {
                    method   => 'GET',
                    key      => 'test_key',
                    response => HTTP::Response->new(200, 'OK', HTTP::Headers->new(), 'test_scalar',),
                },
            ],
            return_value => {
                content      => 'test_scalar',
                content_type => undef,
            },
        );

        test_method(
            $app,
            'get_file',
            ['test_key'],
            requests => [
                {
                    method => 'GET',
                    key    => 'test_key',
                    response =>
                      HTTP::Response->new(200, 'OK', HTTP::Headers->new('Content-Type' => 'application/test'),),
                }
            ],
            return_value => {content_type => 'application/test',},
        );

        test_method(
            $app,
            'get_file',
            ['test_key'],
            requests => [
                {
                    method   => 'GET',
                    key      => 'test_key',
                    response => HTTP::Response->new(200, 'OK', HTTP::Headers->new(),),
                }
            ],
            return_value => {content_type => undef,},
        );

        test_method(
            $app, 'delete',
            ['test_key'],
            requests => [
                {
                    method   => 'DELETE',
                    key      => 'test_key',
                    response => $default_response,
                },
            ],
            return_value => 1,
        );
    },
    init => [qw(api_media_storage_s3)],
);

sub test_method {
    my ($app, $method, $args, %expected) = @_;
    my @checked_request_args =
      (qw(method key query_form headers request_content request_content_file response_content_file));
    {
        no strict 'refs';
        no warnings 'redefine';
        *{'Application::Model::API::Yandex::MediaStorage::S3::request'} = sub {
            my ($self, %request_opts) = @_;
            my $expected_request = shift(@{$expected{requests}});
            for my $arg (@checked_request_args) {
                is_deeply($request_opts{$arg}, $expected_request->{$arg}, $arg);
            }
            return $expected_request->{response};
          }
    }

    subtest $method => sub {
        plan tests => @checked_request_args * @{$expected{requests}} + 1;
        is_deeply($app->api_media_storage_s3->$method(@$args), $expected{return_value}, 'return value');
    };
}
