#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use Test::More;
use Test::Exception;

use Test::Partner2::Simple;

use HTTP::Response;
use HTTP::Date;
use File::Temp qw(tempfile);
use Digest::MD5;

my $error_response_content = <<ERROR;
<?xml version="1.0" encoding="UTF-8"?>
<Error>
    <Code>NoSuchKey</Code>
    <Message>The specified key does not exist.</Message>
    <Resource>/partner-publisher-stat/lolz</Resource>
    <RequestId>6b3dacdc98b7d632</RequestId>
    <Detail>&lt;Description /&gt;</Detail>
</Error>
ERROR

my $ok = HTTP::Response->new(200, 'OK', HTTP::Headers->new(), '',);

run_tests(
    sub {
        my ($app) = @_;

        test_request(
            $app,
            test_name       => 'simple get',
            args            => {method => 'GET',},
            mocked_response => $ok,
        );

        test_request(
            $app,
            test_name => 'get with key',
            args      => {
                method => 'GET',
                key    => 'test_key',
            },
            mocked_response => $ok,
        );

        test_request(
            $app,
            test_name => 'get not_found',
            args      => {
                method => 'GET',
                key    => 'missing_key',
            },
            mocked_response => HTTP::Response->new(403, 'Not Found', HTTP::Headers->new(), $error_response_content,),
            exception       => 'Exception',
        );

        test_request(
            $app,
            test_name => 'get with custom header',
            args      => {
                method  => 'GET',
                key     => 'test_key',
                headers => HTTP::Headers->new('x-amz-custom-header' => 'value'),
            },
            mocked_response => $ok,
        );

        test_request(
            $app,
            test_name => 'put with reqeust_content',
            args      => {
                method          => 'PUT',
                key             => 'test_key',
                request_content => 'test_content',
            },
            mocked_response => $ok,
        );

        test_request(
            $app,
            test_name => 'put with reqeust_content_file',
            args      => {
                method               => 'PUT',
                key                  => 'test_key',
                request_content_file => (tempfile(UNLINK => 1))[1],
            },
            mocked_response => $ok,
        );

        test_request(
            $app,
            test_name => 'put with missing reqeust_content_file',
            test_plan => 1,
            args      => {
                method               => 'PUT',
                key                  => 'test_key',
                request_content_file => 'test_content_filename',
            },
            mocked_response => $ok,
            exception       => 'Exception::API::MediaStorage::S3',
        );

        test_request(
            $app,
            test_name => 'get with response_content_file',
            args      => {
                method                => 'GET',
                key                   => 'test_key',
                response_content_file => 'test_file',
            },
            mocked_response => $ok,
        );

        test_request(
            $app,
            test_name => 'all args',
            args      => {
                method                => 'HEAD',
                key                   => 'test_key',
                request_content_file  => (tempfile(UNLINK => 1))[1],
                response_content_file => 'test_response_file',
            },
            mocked_response => $ok,
        );
    }
);

sub test_request {
    my ($app, %opts) = @_;

    {
        no warnings 'redefine';

        *LWP::UserAgent::request = create_mocked_request_sub(%opts);
    }

    my ($catch, $catch_args) =
      defined($opts{exception})
      ? (\&Test::Exception::throws_ok, [$opts{exception}, 'exception'])
      : (\&Test::Exception::lives_ok, ['no exception']);

    subtest $opts{test_name} => sub {
        plan tests => $opts{test_plan} // 8;
        $catch->(
            sub {
                $app->api_media_storage_s3->request(%{$opts{args}});
            },
            @$catch_args
        );
    };

    return 1;
}

sub create_mocked_request_sub {
    my (%opts) = @_;

    return sub {
        my ($ua, $request, $content_file) = @_;

        is($request->method, $opts{args}{method}, 'method');
        is($request->uri, 'https://s3.mdst.yandex.net/partner-publisher-stat/' . ($opts{args}{key} // ''), 'uri');
        is($content_file, $opts{args}{response_content_file}, 'response content file');
        ok(str2time($request->header('Date')), 'date header');
        ok($request->header('Authorization') =~ m/^AWS \S+$/, 'authorization header');

        subtest 'request content' => sub {
            plan tests => 2;

            my $ctx = Digest::MD5->new();

            if (defined $opts{args}{request_content_file}) {
                is(ref($request->content), 'CODE', 'callback');
                $ctx->add($_) while $request->content->();
                is($request->header('Content-MD5'), $ctx->b64digest . '==', 'md5 header');
            } elsif (defined $opts{args}{request_content}) {
                is($request->content, $opts{args}{request_content}, 'scalar');
                $ctx->add($request->content);
                is($request->header('Content-MD5'), $ctx->b64digest . '==', 'md5 header');
            } else {
                is($request->content, '', 'no content');
                ok(!$request->header('Content-MD5'), 'md5 header');
            }
        };

        subtest 'custom headers' => sub {
            unless (defined $opts{args}{headers}) {
                pass('no custom headers');
                return;
            }

            my @header_names = $opts{args}{headers}->header_field_names;

            plan tests => scalar(@header_names);

            is($request->header($_), $opts{args}{headers}->header($_), $_) for @header_names;
        };

        return $opts{mocked_response};
    };
}
