use strict;
use warnings FATAL => 'all';

use Test::Partner2::Simple;

use Test::Partner2::Mock qw(mock_subs restore_subs);
use Test::Differences qw(eq_or_diff);
use Test::Exception;

use Test::More;
use Test::Deep qw(cmp_deeply);

use PiConstants qw($PARTNERS_TEST_MAIL $MYSQL_MIN_DATETIME $IGNORABLE_MAIL_LISTS_KEY);

use qbit;

my $TEMPLATE_UNIQUE_KEY = 1;

run_tests(
    sub {
        my ($app) = @_;

        my $fake_response = to_json {result => {status => 'OK', message_id => '<message_id>', message => undef}};

        restore_subs([qw(LWP::UserAgent::request)]);
        mock_subs(
            {
                'LWP::UserAgent::request' => sub {
                    my ($mock, $request) = @_;
                    my $url = $request->uri;

                    my $r = HTTP::Response->new(200);
                    $r->request(HTTP::Request->new());
                    $r->content($fake_response);
                    return $r;
                },
                'QBit::Application::Model::API::HTTP::call' => sub {
                    my ($self, $method, %params) = @_;
                    is(from_json($params{':content'})->{to_email}, $PARTNERS_TEST_MAIL, 'flag real_send disabled');
                    is(from_json($params{':content'})->{$_}, undef, "flag $_ is empty") for (qw/to cc bcc/);

                    ok(defined $params{':headers'}{'X-Sender-Prefer-Account'},
                        'header X-Sender-Prefer-Account defined');

                    return $fake_response;
                  }
            }
        );

        throws_ok(
            sub {$app->api_sender->send();},
            "Exception::Validation::BadArguments",
            "check required field to_email"
        );

        throws_ok(
            sub {$app->api_sender->send(to_email => 'to@me');},
            "Exception::Validation::BadArguments",
            "check required field template"
        );

        $app->api_sender->send(to_email => 'to@me', template => 'template_name' . $TEMPLATE_UNIQUE_KEY++);
        $app->set_option('real_send', 1);

        restore_subs([qw(QBit::Application::Model::API::HTTP::call)]);
        mock_subs(
            {
                'QBit::Application::Model::API::HTTP::call' => sub {
                    my ($self, $method, %params) = @_;
                    is(from_json($params{':content'})->{to_email}, 'to@me', 'flag real_send enabled');
                    is($params{':headers'}{'X-Sender-I-Am-Sender'},
                        JSON::XS::true, 'check headers when it`s system call');
                    is(from_json($params{':content'})->{has_ugc}, undef, 'flag has_ugc disabled');

                    is(from_json($params{':content'})->{$_}, $_, "flag $_ is not empty") for (qw/to cc bcc/);

                    return $fake_response;
                  }
            }
        );

        my $response = $app->api_sender->send(
            to_email => 'to@me',
            template => 'template_name' . $TEMPLATE_UNIQUE_KEY++,
            to       => 'to',
            cc       => 'cc',
            bcc      => 'bcc',
        );

        eq_or_diff($response->{result}->{status}, 'OK', 'check response');
        undef $response;

        my $got = $app->partner_db->sender_log->get_all();
        map {$_->{create_time} = $MYSQL_MIN_DATETIME} @$got;

        cmp_deeply(
            $got,
            [
                {
                    id          => 1,
                    create_time => $MYSQL_MIN_DATETIME,
                    message_id  => '<message_id>',
                    template    => 'template_name1',
                    recipient   => '{"to_email": "partners-test-letters@yandex-team.ru"}',
                    status      => undef,
                    response    => '{"result": {"status": "OK", "message": null, "message_id": "<message_id>"}}',
                    opts        => undef,
                },
                {
                    id          => 2,
                    create_time => $MYSQL_MIN_DATETIME,
                    message_id  => '<message_id>',
                    template    => 'template_name2',
                    recipient   => '{"cc": "cc", "to": "to", "bcc": "bcc", "to_email": "to@me"}',
                    status      => undef,
                    response    => '{"result": {"status": "OK", "message": null, "message_id": "<message_id>"}}',
                    opts        => undef,
                }
            ],
            'got expected data in sender_log',
        );

        restore_subs([qw(QBit::Application::Model::API::HTTP::call)]);
        mock_subs(
            {
                'QBit::Application::Model::API::HTTP::call' => sub {
                    my ($self, $method, %params) = @_;

                    is($params{':headers'}{'X-Sender-I-Am-Sender'},
                        JSON::XS::false, 'check X-Sender-I-Am-Sender when it`s user call');
                    ok($params{':headers'}{'X-Sender-Real-User-IP'}, 'check X-Sender-Real-User-IP when it`s user call');
                    is(from_json($params{':content'})->{has_ugc}, JSON::XS::true, 'flag has_ugc enabled');
                    return $fake_response;
                },
            }
        );

        throws_ok(
            sub {
                $app->api_sender->send(
                    to_email     => 'to@me',
                    template     => 'template_name' . $TEMPLATE_UNIQUE_KEY++,
                    is_real_user => 1,
                    remote_addr  => undef,
                );
            },
            "Exception::Validation::BadArguments",
            "check remote_addr is required when is_real_user=1"
        );

        $response = $app->api_sender->send(
            to_email     => 'to@me',
            template     => 'template_name' . $TEMPLATE_UNIQUE_KEY++,
            is_real_user => 1,
            remote_addr  => '127.0.0.1',
        );

        eq_or_diff($response->{result}->{status}, 'OK', 'check response when is_real_user=1');

        $app->kv_store->set($IGNORABLE_MAIL_LISTS_KEY, '{"disabled_template_name":true}');
        restore_subs([qw(QBit::Application::Model::API::HTTP::call)]);
        mock_subs(
            {
                'QBit::Application::Model::API::HTTP::call' => sub {
                    my ($mock, $method, %params) = @_;
                    ldump($method);
                    if ($method =~ /send$/) {
                        throw new Exception::API;
                    } elsif ($method =~ m|campaign/disabled_template_name/$|) {
                        return to_json {sending_control => 'cancel'};
                    } else {
                        die "unexpected Sender method $method";
                    }
                },
            }
        );

        throws_ok(
            sub {
                $app->api_sender->send(
                    to_email     => 'to@me',
                    template     => 'template_name' . $TEMPLATE_UNIQUE_KEY++,
                    is_real_user => 1,
                    remote_addr  => '127.0.0.1',
                );
            },
            "Exception::API",
            "Exception if template not marked as ignorable, but disabled"
        );

        my $res = $app->api_sender->send(
            to_email     => 'to@me',
            template     => 'disabled_template_name',
            is_real_user => 1,
            remote_addr  => '127.0.0.1',
        );
        eq_or_diff($res->{result}{status}, 'DISABLED', 'disabled ignorable template processed');

    },
    init           => [qw( api_sender )],
    fill_databases => 0
);
