use strict;
use warnings FATAL => 'all';
use utf8;
use open qw(:std :utf8);

use QBit::StringUtils qw(from_json);

use Test::Partner2::Simple;

use Test::More;
use Test::Deep;
use Test::Exception;

sub check_simple_usage {
    my ($app) = @_;

    my $tree = $app->bk_statistics->get_tree2();

    no strict 'refs';
    no warnings 'redefine';
    local *{'HTTP::Tiny::request'} = sub {
        return {
            content => '{
                        "status"     : 0,
                        "error_text" : "",
                        "stat_time"  : 20161003180623,
                        "header"     : ["UpdateTime","DirectShows","DirectBannerShows","DirectBannerClicks"],
                        "totals"     : {
                        "DirectBannerClicks" : "1369190",
                        "DirectShows" : "335213314",
                        "DirectBannerShows" : "456213314"
                    },
                        "total_rows" : 1,
                        "host-aggregator" : "bsfastexport01h.yabs.yandex.ru",
                        "duration"   : 0.677,
                        "data"       : [["2016-10-03","335213314","456213314","1369190"]]
        }',
            headers => {
                connection          => 'close',
                'content-type'      => 'text/plain; charset=utf-8',
                date                => 'Mon, 03 Oct 2016 15:24:24 GMT',
                server              => 'nginx/1.8.0',
                'transfer-encoding' => 'chunked',
                vary                => 'Accept-Encoding',
            },
            protocol => 'HTTP/1.1',
            reason   => 'OK',
            status   => 200,
            success  => 1,
        };
    };

    my $data = $app->bk_statistics->get_statistics2(
        'dimension_filter' => undef,
        'levels'           => [
            {
                'filter' => undef,
                'id'     => 'payment'
            }
        ],
        'entity_fields'    => [],
        'vat'              => -1,
        'period'           => ['2016-10-03', '2016-10-03'],
        'dimension_fields' => ['date|day'],
        'order_by'         => [
            {
                'dir'   => 'asc',
                'field' => 'date'
            },
            {
                'dir'   => 'asc',
                'field' => 'shows_direct'
            },
            {
                'dir'   => 'asc',
                'field' => 'clicks_direct'
            }
        ],
        'total'  => 0,
        'jwt'    => $tree->{jwt},
        'limits' => {
            'limit'  => 50,
            'offset' => 0
        },
        'fields' => ['shows_direct', 'shows_direct_banners', 'clicks_direct']
    );

    cmp_deeply(
        $data,
        {
            currencies => [
                {
                    code => 'USD',
                    id   => 1,
                },
                {
                    code => 'RUB',
                    id   => 2,
                },
                {
                    code => 'EUR',
                    id   => 3,
                },
            ],
            dimensions => {
                date => {
                    index => 1,
                    title => 'mol_date',
                    type  => 'date',
                },
            },
            measures => {
                shows_direct => {
                    index => 2,
                    title => 'mol_rtb_block_direct_shows',
                    type  => 'number',
                    unit  => 'count',
                },
                shows_direct_banners => {
                    index => 3,
                    title => 'mol_rtb_block_direct_banner_shows',
                    type  => 'number',
                    unit  => 'count',
                },
                clicks_direct => {
                    index => 4,
                    title => 'mol_rtb_block_direct_clicks',
                    type  => 'number',
                    unit  => 'count',
                },
            },
            periods => [['2016-10-03', '2016-10-03',],],
            points  => [
                {
                    dimensions => {date => ['2016-10-03',],},
                    measures   => [
                        {
                            clicks_direct        => 1369190,
                            shows_direct         => 335213314,
                            shows_direct_banners => 456213314,
                        },
                    ],
                },
            ],
            report_title => 'Report for period 03.10.2016 - 03.10.2016',
            totals       => {
                2 => [
                    {
                        clicks_direct        => 1369190,
                        shows_direct         => 335213314,
                        shows_direct_banners => 456213314,
                    }
                ],
            },
            is_last_page => bool(1),
            total_rows   => 1,
        },
        'got exepcted answer from get_statistics2()',
    );

    return 1;
}

sub check_top_keys {
    my ($app) = @_;

    my $number = 0;

    no strict 'refs';
    no warnings 'redefine';
    local *{'HTTP::Tiny::request'} = sub {
        $number++;

        if ($number == 1) {
            return {
                content => '{
                "status"     : 0,
                "error_text" : "",
                "stat_time"  : 20171030190238,
                "header"     : ["DSPID","DirectShows"],
                "totals"     : {
                "DirectShows" : "1664138990"
            },
                "total_rows" : 32,
                "host-aggregator" : "bsfastexport01e.yabs.yandex.ru",
                "duration"   : 1.298,
                "data"       : [[1,"1664138990"],[2,"0"],[3,"0"],[5,"0"],[10,"0"],[2317563,"0"],[2563049,"0"],[2563070,"0"],[2563075,"0"],[2563081,"0"],[2563084,"0"],[2563087,"0"],[2563099,"0"],[2563106,"0"],[2563113,"0"]]
}',
                headers => {
                    connection          => 'close',
                    'content-type'      => 'text/plain; charset=utf-8',
                    date                => 'Mon, 30 Oct 2017 16:17:38 GMT',
                    server              => 'nginx/1.8.0',
                    'transfer-encoding' => 'chunked',
                    vary                => 'Accept-Encoding',
                },
                protocol => 'HTTP/1.1',
                reason   => 'OK',
                status   => 200,
                success  => 1,
            };

        } elsif ($number == 2) {
            return {
                content => '{
                "status"     : 0,
                "error_text" : "",
                "stat_time"  : 20171030190238,
                "header"     : ["DSPID","DirectShows"],
                "totals"     : {
                "DirectShows" : "1664138990"
            },
                "total_rows" : 15,
                "host-aggregator" : "bsfastexport02e.yabs.yandex.ru",
                "duration"   : 0.238,
                "data"       : [[2563113,"0"],[2563075,"0"],[3,"0"],[2,"0"],[5,"0"],[2563099,"0"],[2563070,"0"],[2563081,"0"],[2563106,"0"],[2563087,"0"],[2317563,"0"],[2563084,"0"],[2563049,"0"],[10,"0"],[1,"1664138990"]]
}',
                headers => {
                    connection          => 'close',
                    'content-type'      => 'text/plain; charset=utf-8',
                    date                => 'Mon, 30 Oct 2017 16:17:39 GMT',
                    server              => 'nginx/1.8.0',
                    'transfer-encoding' => 'chunked',
                    vary                => 'Accept-Encoding',
                },
                protocol => 'HTTP/1.1',
                reason   => 'OK',
                status   => 200,
                success  => 1,
            };
        } else {
            die "Unexpected request to MOL server";
        }
    };

    my $data = $app->bk_statistics->get_statistics2(
        dimension_fields => [],
        dimension_filter => undef,
        entity_fields    => ['dsp_id',],
        fields           => ['shows_direct',],
        levels           => [
            {
                filter => undef,
                id     => 'payment',
            },
        ],
        order_by => [
            {
                dir   => 'asc',
                field => 'shows_direct',
            },
        ],
        period   => ['2017-08-07', '2017-08-07'],
        top_keys => 15,
        total    => 0,
        vat      => -1,
        limits   => {
            limit  => 500,
            offset => 0,
        },
    );

    cmp_deeply(
        $data,
        {
            currencies => [
                {
                    code => 'USD',
                    id   => 1,
                },
                {
                    code => 'RUB',
                    id   => 2,
                },
                {
                    code => 'EUR',
                    id   => 3,
                },
            ],
            dimensions => {
                dsp_id => {
                    index => 1,
                    title => 'mol_dsp_id',
                    type  => 'categories',
                },
            },
            is_last_page => bool(1),
            measures     => {
                shows_direct => {
                    index => 2,
                    title => 'mol_rtb_block_direct_shows',
                    type  => 'number',
                    unit  => 'count',
                },
            },
            periods => [['2017-08-07', '2017-08-07',],],
            points  => [
                {
                    dimensions => {dsp_id        => 2563113,},
                    measures   => [{shows_direct => 0,},],
                },
                {
                    dimensions => {dsp_id        => 2563075,},
                    measures   => [{shows_direct => 0,},],
                },
                {
                    dimensions => {dsp_id        => 3,},
                    measures   => [{shows_direct => 0,},],
                },
                {
                    dimensions => {dsp_id        => 2,},
                    measures   => [{shows_direct => 0,},],
                },
                {
                    dimensions => {dsp_id        => 5,},
                    measures   => [{shows_direct => 0,},],
                },
                {
                    dimensions => {dsp_id        => 2563099,},
                    measures   => [{shows_direct => 0,},],
                },
                {
                    dimensions => {dsp_id        => 2563070,},
                    measures   => [{shows_direct => 0,},],
                },
                {
                    dimensions => {dsp_id        => 2563081,},
                    measures   => [{shows_direct => 0,},],
                },
                {
                    dimensions => {dsp_id        => 2563106,},
                    measures   => [{shows_direct => 0,},],
                },
                {
                    dimensions => {dsp_id        => 2563087,},
                    measures   => [{shows_direct => 0,},],
                },
                {
                    dimensions => {dsp_id        => 2317563,},
                    measures   => [{shows_direct => 0,},],
                },
                {
                    dimensions => {dsp_id        => 2563084,},
                    measures   => [{shows_direct => 0,},],
                },
                {
                    dimensions => {dsp_id        => 2563049,},
                    measures   => [{shows_direct => 0,},],
                },
                {
                    dimensions => {dsp_id        => 10,},
                    measures   => [{shows_direct => 0,},],
                },
                {
                    dimensions => {dsp_id        => 1,},
                    measures   => [{shows_direct => 1664138990,},],
                },
            ],
            report_title     => 'Report for period 07.08.2017 - 07.08.2017',
            totals           => {2 => [{shows_direct => 1664138990,},],},
            total_rows       => 15,
            top_keys_cropped => 1,
        },
    );

    return 1;
}

sub check_incorrect_request {
    my ($app) = @_;

    throws_ok(
        sub {
            $app->bk_statistics->get_statistics2(
                'dimension_filter' => undef,
                'levels'           => [
                    {
                        'filter' => undef,
                        'id'     => 'payment'
                    }
                ],
                'entity_fields'    => [],
                'vat'              => -1,
                'period'           => ['2016-10-03', '2016-10-03'],
                'dimension_fields' => ['date|day'],
                'order_by'         => [
                    {
                        'dir'   => 'INCORRECT_VALUE',
                        'field' => 'date'
                    },
                    {
                        'dir'   => 'asc',
                        'field' => 'shows_direct'
                    },
                    {
                        'dir'   => 'asc',
                        'field' => 'clicks_direct'
                    }
                ],
                'total'  => 0,
                'limits' => {
                    'limit'  => 50,
                    'offset' => 0
                },
                'fields' => ['shows_direct', 'clicks_direct']
            );
        },
        'Exception::Validator::Errors',
        'got exception Exception::Validator::Errors'
    );

    is_deeply(
        from_json($@->message()),
        [
            {
                "messages" => ["Got value \"INCORRECT_VALUE\" not in array: asc, desc"],
                "name"     => ["order_by", "0", "dir"]
            }
        ],
        'got expected exception message',
    );

    throws_ok(
        sub {
            $app->bk_statistics->get_statistics2(
                'dimension_filter' => undef,
                'levels'           => [
                    {
                        'filter' => undef,
                        'id'     => 'payment'
                    }
                ],
                'entity_fields'    => [],
                'vat'              => -1,
                'period'           => ['2016-10-03', '2016-10-03'],
                'dimension_fields' => ['date|day'],
                'order_by'         => [
                    {
                        'dir'   => 'desc',
                        'field' => 'geo'
                    },
                    {
                        'dir'   => 'asc',
                        'field' => 'hits'
                    },
                    {
                        'dir'   => 'asc',
                        'field' => 'clicks_direct'
                    }
                ],
                'total'  => 0,
                'limits' => {
                    'limit'  => 50,
                    'offset' => 0
                },
                'fields' => ['shows_direct', 'clicks_direct']
            );
        },
        'Exception::Validator::Errors',
        'got exception Exception::Validator::Errors'
    );

    is_deeply(
        from_json($@->message()),
        [
            {
                "messages" => ["Got value \"geo\" not in array: clicks_direct, date, shows_direct"],
                "name"     => ["order_by", "0", "field"]
            },
            {
                "messages" => ["Got value \"hits\" not in array: clicks_direct, date, shows_direct"],
                "name"     => ["order_by", "1", "field"]
            }
        ],
        'got expected exception message',
    );

    return 1;
}

sub check_too_many_fields {
    my ($app) = @_;

    my $tree = $app->bk_statistics->get_tree2();

    no strict 'refs';
    no warnings 'redefine';
    local *{'HTTP::Tiny::request'} = sub {
        return {
            content => '{
            "status"     : 1,
            "error_text" : "\nToo many fields in \"group_by\"",
            "stat_time"  : 0,
            "header"     : [],
            "totals"     : {},
            "total_rows" : 0,
            "host-aggregator" : "",
            "duration"   : 0,
            "data"       : []
        }',
            headers => {
                connection          => 'close',
                'content-type'      => 'text/plain; charset=utf-8',
                date                => 'Thu, 21 Dec 2017 15:15:45 GMT',
                server              => 'nginx/1.8.0',
                'transfer-encoding' => 'chunked',
                vary                => 'Accept-Encoding',
            },
            protocol => 'HTTP/1.1',
            reason   => 'OK',
            status   => 200,
            success  => 1,
        };
    };

    throws_ok(
        sub {
            $app->bk_statistics->get_statistics2(
                period           => '7days',
                fields           => ['shows_direct', 'clicks_direct',],
                dimension_fields => ['date|day',],
                dimension_filter => undef,
                entity_fields    => ['dsp_caption', 'page_level', 'page_caption', 'block_caption',],
                total            => 0,
                vat              => -1,
                levels           => [
                    {
                        id     => 'payment',
                        filter => undef,
                    }
                ],
                order_by => [
                    {
                        field => 'date',
                        dir   => 'asc',
                    },
                    {
                        field => 'shows_direct',
                        dir   => 'asc',
                    },
                    {
                        field => 'clicks_direct',
                        dir   => 'asc',
                    }
                ],
                jwt    => $tree->{jwt},
                limits => {
                    limit  => 50,
                    offset => 0,
                }
            );
        },
        'Exception::Validation::BadArguments',
        'correct exception on [Too many fields in "group_by"]'
    );
    is($@->message, 'Too many fields in group by', 'correct error message on [Too many fields in "group_by"]');

    return 1;
}

sub check_invalid_value {
    my ($app) = @_;

    eval {
        my $data = $app->bk_statistics->get_statistics2(
            'dimension_filter' => undef,
            'levels'           => [
                {
                    'filter' => ['AND', [['page_id', '=', "фыва"]]],
                    'id' => 'payment'
                }
            ],
            'entity_fields'    => ['page_id'],
            'vat'              => -1,
            'period'           => '7days',
            'dimension_fields' => [],
            'order_by'         => [],
            'total'            => 0,
            'limits'           => {
                'limit'  => 50,
                'offset' => 0
            },
            'fields' => ['shows_direct']
        );
    };

    my $error = $@;

    is(ref($error),      'Exception::Validation::BadArguments',     'got correct exception type');
    is($error->{'text'}, 'Incorrect value for field "mol_page_id"', 'got expected error text');

    return 1;
}

sub check_bug_PI_10058 {
    my ($app) = @_;

    no strict 'refs';
    no warnings 'redefine';
    local *{'HTTP::Tiny::request'} = sub {
        return {
            content => '{
                "status"     : 0,
                "error_text" : "",
                "stat_time"  : 20170718125620,
                "header"     : ["UpdateTime","PageID","RenderShows","DirectCtr"],
                "totals"     : {
                "RenderShows" : "0",
                "DirectCtr" : null
            },
                "total_rows" : 1,
                "host-aggregator" : "bsfastexport02f.yabs.yandex.ru",
                "duration"   : 0.129,
                "data"       : [["2017-07-15",198031,"0",null]]
}',
            headers => {
                connection          => 'close',
                'content-type'      => 'text/plain; charset=utf-8',
                date                => 'Tue, 18 Jul 2017 10:19:43 GMT',
                server              => 'nginx/1.8.0',
                'transfer-encoding' => 'chunked',
                vary                => 'Accept-Encoding',
            },
            protocol => 'HTTP/1.1',
            reason   => 'OK',
            status   => 200,
            success  => 1,
        };
    };

    my $data = $app->bk_statistics->get_statistics2(
        'levels' => [
            {
                'filter' => ['AND', [['page_id', '=', '198031']]],
                'id' => 'payment'
            }
        ],
        'dimension_filter' => undef,
        'vat'              => -1,
        'entity_fields'    => ['page_id'],
        'period'           => ['2017-07-15', '2017-07-15'],
        'top_keys'         => 7,
        'order_by'         => [
            {
                'dir'   => 'asc',
                'field' => 'date'
            }
        ],
        'dimension_fields'  => ['date|day'],
        'total'             => 0,
        'top_keys_order_by' => [
            {
                'dir'   => 'asc',
                'field' => 'shows'
            },
            {
                'dir'   => 'asc',
                'field' => 'ctr_direct'
            }
        ],
        'fields' => ['shows', 'ctr_direct'],
        'limits' => {
            limit  => 500,
            offset => 0,
        },
    );

    cmp_deeply(
        $data,
        {
            currencies => [
                {
                    code => 'USD',
                    id   => 1,
                },
                {
                    code => 'RUB',
                    id   => 2,
                },
                {
                    code => 'EUR',
                    id   => 3,
                },
            ],
            dimensions => {
                date => {
                    index => 1,
                    title => 'mol_date',
                    type  => 'date',
                },
                page_id => {
                    index => 2,
                    title => 'mol_page_id',
                    type  => 'page_id',
                },
            },
            is_last_page => bool(1),
            measures     => {
                shows => {
                    index => 3,
                    title => 'mol_rtb_block_shows',
                    type  => 'number',
                    unit  => 'count',
                },
                ctr_direct => {
                    index => 4,
                    title => 'mol_rtb_block_direct_ctr',
                    type  => 'float',
                    unit  => 'ctr_percent',
                },
            },
            periods => [['2017-07-15', '2017-07-15',],],
            points  => [
                {
                    dimensions => {
                        date    => ['2017-07-15',],
                        page_id => 198031,
                    },
                    measures => [
                        {
                            ctr_direct => 0,
                            shows      => 0,
                        },
                    ],
                },
            ],
            report_title => 'Report for period 15.07.2017 - 15.07.2017',
            totals       => {
                2 => [
                    {
                        ctr_direct => 0,
                        shows      => 0,
                    },
                ],
            },
            total_rows => 1,
        },
        'got exepcted answer from get_statistics2()',
    );

    return 1;
}

sub check_incorrect_limits {
    my ($app) = @_;

    eval {
        my $data = $app->bk_statistics->get_statistics2(
            'dimension_filter' => undef,
            'levels'           => [
                {
                    'filter' => undef,
                    'id'     => 'payment'
                }
            ],
            'entity_fields'    => [],
            'vat'              => -1,
            'period'           => ['2016-10-03', '2016-10-03'],
            'dimension_fields' => ['date|day'],
            'order_by'         => [
                {
                    'dir'   => 'asc',
                    'field' => 'date'
                },
                {
                    'dir'   => 'asc',
                    'field' => 'shows_direct'
                },
                {
                    'dir'   => 'asc',
                    'field' => 'clicks_direct'
                }
            ],
            'total'  => 0,
            'limits' => {
                'limit'  => 501,    # <- incorrect value
                'offset' => 0,
            },
            'fields' => ['shows_direct', 'clicks_direct'],
        );
    };

    my $error = $@;

    is(ref($error), 'Exception::Validator::Errors', 'got exception Exception::API::HTTPMOL');

    is(
        $error->message(),
        '[
   {
      "messages" : [
         "limit must be in the range from 1 to 500"
      ],
      "name" : [
         "limits",
         "limit"
      ]
   }
]
',
        'got expected exception message',
      );

    return 1;
}

sub check_empty_fields {
    my ($app) = @_;

    throws_ok(
        sub {
            $app->bk_statistics->get_statistics2(
                'fields'           => [],
                'entity_fields'    => [],
                'period'           => ['2016-10-03', '2016-10-03'],
                'dimension_fields' => ['date|day'],
                'total'            => 0,

            );
        },
        'Exception::Validator::Errors',
        'got exception Exception::Validator::Errors'
    );

    is_deeply(
        from_json($@->message()),
        [
            {
                "messages" => ['Data size "0" less than "1"'],
                "name"     => ["fields"]
            }
        ],
        'got expected exception message',
    );

    return 1;
}

run_tests(
    sub {
        my ($app) = @_;

        check_simple_usage($app);
        check_top_keys($app);
        check_incorrect_request($app);
        check_too_many_fields($app);
        check_invalid_value($app);
        check_bug_PI_10058($app);
        check_incorrect_limits($app);
        check_empty_fields($app);

    },
    user => 'yndx-bessarabov',
);
