#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use Test::Partner2::Simple;
use Test::Partner2::Fixture;

use Test::More;
use Test::Deep;

use qbit;

use lib::abs;

my @context_rtb_fixtures = (
    qw(
      partner_context_page
      adinside_context_page
      business_unit_context_page
      partner_context_graysite_page
      )
);
my @mobile_rtb_fixtures = (
    qw(
      partner_mobile_page
      adinside_mobile_page
      business_unit_mobile_page
      )
);
my @video_page_fixtures = (
    qw(
      partner_video_page
      partner_video_page_with_vpaid_disabled
      )
);

my $indoor_page_fixture           = 'partner_indoor_page';
my $outdoor_page_fixture          = 'partner_outdoor_page';
my $internal_context_page_fixture = 'internal_context_page';
my $internal_mobile_page_fixure   = 'internal_mobile_page';

my @dsp_fixtures = (
    qw(
      dsp_direct
      dsp_awaps
      dsp_own_adv
      dsp_unsold
      dsp_awaps_video_auction
      dsp_awaps_video_auction_in_banner
      dsp_awaps_video_in_banner
      dsp_video_interstitial_mobile_10
      dsp_video_interstitial_mobile_7
      dsp_video_interstitial_mobile_12
      dsp_media
      dsp_media_working_on_all_platforms
      dsp_media_working_on_yandex_services
      dsp_video
      dsp_mobile
      dsp_awaps_mobile
      dsp_media_video_mobile
      dsp_media_video_mobile_with_internal_tag
      dsp_deleted
      dsp_disabled
      )
);

my @methods            = (qw(get_available_dsps get_default_dsps));
my @mobile_block_types = (qw(banner interstitial native rewarded));

run_tests(
    sub {
        my ($app) = @_;

        my @tests = (
            context_on_site_rtb_tests(),     context_on_site_adblock_tests(),
            context_on_site_content_tests(), mobile_app_rtb_tests(),
            video_block_tests(),             indoor_block_tests(),
            internal_context_rtb_tests(),    internal_mobile_rtb_tests(),
            outdoor_block_tests(),
        );

        my $file = lib::abs::path('block_dsp_relation.json');

        unless (-e $file) {
            writefile($file, '{}');
        }

        my $expected_data = from_json(readfile($file));
        my $got_data      = {};

        my $dsps = get_fixtures_hash(@dsp_fixtures);
        my %dsp_name_by_id = map {$dsps->{$_}{id} => $_} keys(%$dsps);

        for my $test (@tests) {
            my ($key, $model, $method, $method_args) = @{$test}{qw(key model method method_args)};

            my $got = [map {$dsp_name_by_id{$_->{id}}} @{$app->$model->$method($method_args)}];

            cmp_deeply($got, bag(@{$expected_data->{$key}}), $key)
              unless need_self_update();

            $got_data->{$key} = [sort @$got];
        }

        if (need_self_update()) {
            writefile($file, to_json($got_data, pretty => TRUE));
            fail('self update mode');
        }
    },
    fill_databases => 0,
    fixtures       => [
        @context_rtb_fixtures, @mobile_rtb_fixtures,           @video_page_fixtures,
        $indoor_page_fixture,  $internal_context_page_fixture, $internal_mobile_page_fixure,
        @dsp_fixtures,         $outdoor_page_fixture,
    ],
);

sub context_on_site_rtb_tests {
    _tests(
        models             => [qw(context_on_site_rtb)],
        pages              => \@context_rtb_fixtures,
        show_video         => 1,
        page_id_field_name => 'page_id',
    );
}

sub context_on_site_adblock_tests {
    _tests(
        models             => [qw(context_on_site_adblock)],
        pages              => [$context_rtb_fixtures[0]],
        show_video         => 1,
        page_id_field_name => 'page_id',
    );
}

sub context_on_site_content_tests {
    _tests(
        models             => [qw(context_on_site_content)],
        pages              => [@context_rtb_fixtures],
        page_id_field_name => 'page_id',
    );
}

sub mobile_app_rtb_tests {
    _tests(
        models             => [qw(mobile_app_rtb)],
        pages              => \@mobile_rtb_fixtures,
        block_type         => [qw(banner interstitial native rewarded)],
        show_video         => 1,
        page_id_field_name => 'context_page_id',
    );
}

sub video_block_tests {
    _tests(
        models             => [qw(video_an_site_inpage video_an_site_instream video_an_site_fullscreen)],
        pages              => \@video_page_fixtures,
        page_id_field_name => 'id',
    );
}

sub indoor_block_tests {
    _tests(
        models             => [qw(indoor_block)],
        pages              => [$indoor_page_fixture],
        page_id_field_name => 'page_id',
    );
}

sub outdoor_block_tests {
    _tests(
        models             => [qw(outdoor_block)],
        pages              => [$outdoor_page_fixture],
        page_id_field_name => 'page_id',
    );
}

sub internal_context_rtb_tests {
    _tests(
        models             => [qw(internal_context_on_site_rtb)],
        pages              => [$internal_context_page_fixture],
        page_id_field_name => 'id',
        show_video         => 1,
    );
}

sub internal_mobile_rtb_tests {
    _tests(
        models             => [qw(internal_mobile_app_rtb)],
        pages              => [$internal_mobile_page_fixure],
        block_type         => [qw(banner interstitial native)],
        page_id_field_name => 'id',
        show_video         => 1,
    );
}

sub _tests {
    my (%opts) = @_;

    my @tests;
    my $sub = sub {
        my ($test_case_details) = @_;
        # Generate args for get_{available,default}_dsps method
        my $method_args = {
            $opts{page_id_field_name} ? (page_id => $test_case_details->{page}{value}{$opts{page_id_field_name}}) : (),
            $opts{show_video} ? (show_video => $test_case_details->{show_video}) : (),
            $opts{block_type} ? (block_type => $test_case_details->{block_type}) : (),
        };

        # Generate and save test case
        push @tests,
          {
            key => join(' - ',
                $test_case_details->{model},      $test_case_details->{method},
                $test_case_details->{page}{name}, build_method_args_key_part($method_args)),
            model       => $test_case_details->{model},
            method      => $test_case_details->{method},
            method_args => $method_args,
          };
    };

    # Result for non-video blocks depends on show_video flag
    if ($opts{show_video}) {
        $sub = for_values(show_video => [0, 1], $sub);
    }

    # Result for mobile blocks depends on block_type
    if ($opts{block_type}) {
        $sub = for_values(block_type => $opts{block_type}, $sub);
    }

    $sub = for_fixtures(page => $opts{pages}, $sub);
    $sub = for_values(method => \@methods,     $sub);
    $sub = for_values(model  => $opts{models}, $sub);

    my $opts = {};
    $sub->($opts);

    return @tests;
}

sub for_values {
    my ($name, $values, $sub) = @_;
    return sub {
        my ($opts) = @_;
        for my $val (@$values) {
            $opts->{$name} = $val;
            $sub->($opts);
        }
    };
}

sub for_fixtures {
    my ($opt_name, $fixture_names, $sub) = @_;
    my $fixtures = get_fixtures_hash(@$fixture_names);
    return sub {
        my ($opts) = @_;
        for my $fixture_name (@$fixture_names) {
            $opts->{$opt_name}{name}  = $fixture_name;
            $opts->{$opt_name}{value} = $fixtures->{$fixture_name};
            $sub->($opts);
        }
    };
}

sub build_method_args_key_part {
    my ($block) = @_;
    return to_json({map {$_ => $block->{$_}} grep {$_ ne 'page_id'} keys(%$block)}, canonical => 1);
}
