#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use Test::Partner2::Simple;
use Test::Partner2::Utils qw($SKIP_MODELS);

use Test::Most;
use Test::Deep;

use qbit;

my $FIELD_TO_CHECK = 'page_id';

# from ./mocked_databases/partner_db/BLOCKS.sql
my $page_ids = {
    default                          => 142898,
    context_on_site_adblock          => 153745,
    outdoor_block                    => 401,
    internal_context_on_site_direct  => 88849,
    internal_context_on_site_rtb     => 88848,
    internal_context_on_site_stripe  => 606060,
    internal_context_on_site_content => 88849,
    internal_context_on_site_natural => 88849,
    internal_mobile_app_rtb          => 132439,
    internal_search_on_site_direct   => 118826,
    internal_search_on_site_premium  => 128843,
    mobile_app_rtb                   => 43569,
    mobile_mediation_block           => 43569,
    search_on_site_direct            => 111161,
    video_an_site_fullscreen         => 154689,
    video_an_site_inpage             => 154689,
    video_an_site_instream           => 128972,
};

my $fixture = {
    'page_id owned by user' => {
        data   => $page_ids,
        errors => {default => {},},
        user   => {
            (
                default => 'mocked-yan-partner',

                context_on_site_adblock => 'mocked-context-adblock-partner',

                outdoor_block => 'mocked-outdoor-partner2',

                video_an_site_fullscreen => 'mocked-video-partner',
                video_an_site_inpage     => 'mocked-video-partner',
                video_an_site_instream   => 'mocked-video-partner',

                mobile_app_rtb         => 'mocked-mobile-app-partner',
                mobile_mediation_block => 'mocked-mobile-app-partner',

                search_on_site_direct => 'mocked-yan-partner',

                # internal blocks' models accessors
                map {$_ => 'mocked-internal-administrator'} qw(
                  internal_context_on_site_direct
                  internal_context_on_site_rtb
                  internal_context_on_site_stripe
                  internal_context_on_site_content
                  internal_context_on_site_natural
                  internal_mobile_app_rtb
                  internal_search_on_site_direct
                  internal_search_on_site_premium
                  ),
            )
        }
    },
    'page_id is not owned by user' => {
        data   => $page_ids,
        errors => {default => {'page_id' => ["Can not find page: %s"]},},
        user   => {
            (
                default => 'mocked-yan-partner-no-campaigns',

                outdoor_block => 'mocked-outdoor-partner',

                video_an_site_fullscreen => 'tutby-partner-with-agreement',
                video_an_site_inpage     => 'tutby-partner-with-agreement',
                video_an_site_instream   => 'tutby-partner-with-agreement',

                search_on_site_direct => 'mocked-yan-partner2',
            )
        }
    },
};

run_tests(
    sub {
        my ($app) = @_;

        my @accessors = sort @{$app->product_manager->get_block_model_accessors()};

        warn scalar @accessors . " tests 2 run\n";

        foreach my $accesor (@accessors) {
            next if $SKIP_MODELS->{$accesor};

            unless ($app->$accesor->can('get_template')) {
                warn "$accesor => is not validated; has no ->get_template";
                next;
            }

            subtest "checked_$accesor" => sub {

                foreach my $sub_test_name (sort keys %{$fixture}) {
                    my ($errors, $data, $user) = @{$fixture->{$sub_test_name}}{qw( errors data user )};

                    foreach ($errors, $data, $user) {
                        $_ = exists $_->{$accesor} ? $_->{$accesor} : $_->{default};
                    }

                    $errors = {map {$_ => [sprintf($errors->{$_}->[0], $data)]} keys %$errors};

                    change_cur_user($app, $user);
                    _check_validation($app, $accesor, $data, $errors, $accesor . '.' . $sub_test_name);

                }
              }
        }

    },
    do_not_die_on_fail => 1,
);

sub _check_validation {
    my ($app, $accesor, $data, $expect_errors, $test_name) = @_;

    my $qv = QBit::Validator->new(
        data       => $data,
        app        => $app->$accesor,
        'template' => $app->$accesor->get_model_fields()->{$FIELD_TO_CHECK}->{need_check},
    );

    my @errors     = $qv->get_fields_with_error();
    my $got_errors = _get_erros(\@errors);

    if (%$expect_errors) {
        my $is_ok = cmp_deeply($got_errors, $expect_errors, $test_name);
        warn Data::Dumper->Dump([$got_errors], ['$got_errors ' . __PACKAGE__ . ':' . __LINE__]) unless $is_ok;
    } else {
        my $is_ok = is($qv->has_errors, FALSE, $test_name);
        warn Data::Dumper->Dump([$got_errors], ['$got_errors ' . __PACKAGE__ . ':' . __LINE__]) unless $is_ok;
    }

    return 1;
}

sub _get_erros {
    my ($errors) = @_;

    my $res = {
        # <path> => [ 'msg1', ... ]
    };
    foreach my $row (@$errors) {
        my $key = join '.', @{$row->{path}} || $FIELD_TO_CHECK;
        my $ar = $res->{$key} //= [];

        # ARRAY(0xf175be0) -> ARRAY(0xFFFFFF)
        push @$ar, map {s/\(0x[^)]+\)/(0xFFFFFF)/g; $_} @{$row->{msgs}};
    }

    return $res;
}
